<?php
/** 2021-12-18 todo verde */

/**
 * Clase para validación de datos
 * @author José Juan del Prado
 * @version 1.0
 *
 * Rules:
 *      required -> verifica que el valor no sea vacío
 *      string -> verifica que el valor sea una cadena de caracteres
 *      numeric -> verifica que el valor sea tipo número (Sin comas) //@TODO si es int no tenga decimales lo hace esta clase o no?
 *      max -> verifica que el valor sea menor o igual a un valor máximo
 *      min -> verifica que el valor sea mayor o igual a un valor mínimo
 *      date -> comprueba que el valor sea una fecha correcta en formato YYYY-mm-dd
 *      in -> verifica que el valor se encuentre dentro de valores posibles (es más flexible porque permite enums, catálogo y queries)
 *      enum -> verifica que el valor se encuentre dentro de valores posibles
 *      bool -> verifica que el valor sea un boolean (["true", "false", "1", "0", "yes", "no", 1, 0, true, false])
 *      unique -> comprueba que el valor sea único dentro de una tabla en la db
 *      sequence -> verifica que el valor sea un consecutivo (El valor anterior tiene que existir en la db) OJO solo al insertar ademas usar unique_insert
 *
 * Implementación:
 *  Para required
 *      $rules = [nombre_campo => ['required']];
 *
 *  Para string
 *      $rules = [nombre_campo => ['string']];
 *
 *  Para numeric
 *      $rules = [nombre_campo => ['numeric']];
 *
 *  Para max
 *      $rules = [nombre_campo => ['max:max_value']]
 *      $rules = [nombre_campo => ['max' => max_value]]
 *    Nota: max_value puede ser numérico o fecha en formato YYYY-mm-dd, se ocupa para validar máximo de caracteres, valor maximo(tope) y fecha máxima.
 *          Para fecha puede recibir frases que estén dentro del standar de la función strtotime (https://www.php.net/manual/en/function.strtotime.php)
 *
 *  Para mim
 *      $rules = [nombre_campo => ['min:min_value']]
 *          o
 *      $rules = [nombre_campo => ['min' => min_value]]
 *    Nota: min_value puede ser numérico o fecha en formato YYYY-mm-dd, se ocupa para validar mínimo de caracteres, valor mínimo(tope) y fecha máxima
 *          Para fecha puede recibir frases que estén dentro del standar de la función strtotime (https://www.php.net/manual/en/function.strtotime.php)
 *
 *  Para date
 *      $rules = [nombre_campo => ['date']]
 *
 *  Para in
 *      $rules = [nombre_campo => ['in:val1,val2,...']]
 *          o
 *      $rules = [nombre_campo => ['in' => ['val1', 'val2' ...]]]
 *
 *  -> @_catalogo para indicar que es catalogo.
 *  -> @_sql para indicar que es un query.
 *
 * in:@_catalogo,tabla,pkfield,where_condition | "in" => ['@_catalogo', tabla, pkfield, where_condition]
 *      Example: in:@_catalogo,producto_general,,activo='Si'
 *      Nota: Si pkfield va vacio se calcula con la tabla (tabla => 'tienda', pkfield => tabla + '_id')
 *  in:@_sql,query
 *      Example: in:@_sql,SELECT * FROM bodega WHERE bodega_id = {@value} AND activo = 'Si'
 *    Nota: Si lleva {@value}, el validador lo reemplaza por el valor de ese campo -> ({@value} -> strit(value))
 *
 *  PARA enum
 *      $rules = [nombre_campo => ['enum:val1,val2,...']]
 *          o
 *      $rules = [nombre_campo => ['enum' => ['Val1', 'Val2', ...]]]
 *
 * Para bool
 *      $rules = [nombre_campo => ['bool']]
 *
 * Para unique
 *      $rules = [nombre_campo => ['unique:table_search,column_search']]
 *          o
 *      $rules = [nombre_campo => ['unique' => ['table_search', 'column_search']]]
 *    Nota: Si no se especifica column_search toma el key (indice) del array de rules
 *
 *  Para sequence
 *      $rules = [nombre_campo => ['sequence:table_search,column_search']]
 *          o
 *      $rules = [nombre_campo => ['sequence' => ['table_search', 'column_search']]]
 *    Nota: Si no se especifica column_search toma el key (indice) del array de rules
 *
 *
 * EXAMPLE
    // rules definition
    $rules = [
        'numero' => ['required', 'numeric', 'unique:nota_bodega'],
        'bodega_id' => ['required', 'string', 'max:32', "in:@_catalogo,bodega,,activo='Si'"],
        'entrada_salida' => ['required', 'string', 'enum:Entrada,Salida'],
        'fecha' => ['required', 'date', 'max:today'],
        'origen_id' => ['required', "in" => ['@_catalogo', 'origen_bodega', '', "activo='Si'"]],
        'paid' => ['required',  "in:1,0"],
        'remarks' => ['string'],
        'remarks_paid' => ['string'],
        'producto_general_id' => ['required', "in:@_catalogo,producto_general,,activo='Si'"],
    ];

    $data = [
        'numero' => '1234',
        'bodega_id' => '54bf6469e2cc9f5511ec0696f1cf62b6',
        'entrada_salida' => 'Entrada'
        'fecha' => '2021-09-24',
        'origen_id' => 8,
        'paid' => 0,
        'remarks' => '',
        'remarks_paid' => '',
        'producto_general_id' => '54bf6469e2cc9f5511ec05d0dc9e35a0',
    ];
    $validation = ValidateHelper::validate($data, $rules);
    // Si $validation es un array vacío no hubo errores
    dd_($validation);
 *
 ****
 ****
 ****
 * NOTA: para hacer override de los labels y mensajes de error se pueden mandar como key en la definición de $rules o como parametros en ::validate
 *
 * $rules = [
 *  'numero' => ['required', 'numeric', 'unique:nota_bodega',
 *      'label' => 'Numero de Nota',
 *      'message' => [
 *          'required' => '<b>@fieldname</b> un dato es requerido'
 *          'numeric' => 'El numero de nota debe ser numerico'
 *          'unique' => 'El numero de nota ya existe'
 *      ]
 *   ],
 * ];
 *
 *  o
 *
 * $labels = ['numero' => 'Numero de Nota', 'entrada_salida' => 'Tipo de Movimiento'];
 * $messages = ['numero' => ['required' => 'El campo es requerido',  'numeric' => 'El numero de nota debe ser numerico', 'unique' => 'El numero de nota ya existe']];
 * $validation = ValidateHelper::validate($data, $rules, $labels, $messages);
 *
 * NOTA: scale necesariamente debe ser: scale => valor
 *
 */
final class ValidateHelper
{
    /**
     * @var array $messages Mensajes default del validador, se pueden mandar mensajes personalizados a la hora de validar
     */
    private static array $messages = [
        'required' => 'El campo <b>@fieldname</b> es requerido',
        'string' => 'El campo <b>@fieldname</b> debe ser un una cadena de letras', //@TODO no debia ser alpha?
        'strings' => 'El campo <b>@fieldname</b> debe ser un una cadena de letras', //@todo is_string('3.1415') es true por las comillas
        'numeric' => 'El campo <b>@fieldname</b> debe ser numérico',
        'int' => 'El campo <b>@fieldname</b> debe ser un número entero',
        'max' => 'El campo <b>@fieldname</b> debe ser menor o igual a @max_min', //@TODO debe ser @min por si queremos poner rango
        'min' => 'El campo <b>@fieldname</b> debe ser mayor o igual a @max_min', //@TODO debe ser @max por si queremos poner rango
        'strlen_max' => 'El campo <b>@fieldname</b> no debe ser mayor a @max_min caracteres', //@TODO en html max_length, min_length?
        'strlen_min' => 'El campo <b>@fieldname</b> debe contener al menos @max_min caracteres',
        'date_max' => 'El campo <b>@fieldname</b> debe ser menor o igual a @max_min', //@TODO hay que pasarselo al datePicker
        'date_min' => 'El campo <b>@fieldname</b> debe ser mayor o igual @max_min', //@TODO hay que pasarselo al datePicker
        'date' => 'El campo <b>@fieldname</b> debe ser una fecha válida',
        'in' => 'El valor para el campo <b>@fieldname</b> no existe',
        'enum' => 'El valor para el campo <b>@fieldname</b> es inválido',
        'bool' => 'Formato para el campo <b>@fieldname</b> inválido',
        'unique' => 'El <b>@fieldname</b> ya existe y no se puede duplicar', //este lo mimso que unique_on_insert
        'unique_on_insert' => 'El <b>@fieldname</b> ya existe y no se puede duplicar',
        'unique_on_update_delete' => 'El <b>@fieldname</b> ya existe y no se puede duplicar',
        'sequence' => 'El <b>@fieldname</b> previo no existe',
        'equal' => 'El valor para el campo <b>@fieldname</b> no coincide',
        'is_equal' => 'El valor para el campo <b>@fieldname</b> no coincide con el esperado'
    ];

    /**
     * PreCondition limpiar $arrKeyValue (quitar , a numeric) los valida contra $rules y regesa array con errores o vacio
     *
     * @param array $arrKeyValue
     * @param array $rules
     * @param array $labels
     * @param array $messages
     * @param bool $estricto en true si hay regla que no tiene campo en $arrKeyValue da error y si hay regla sin case da error
     * @TODO pasar default de $todaReglaTieneEntradaEnArrKeyValue a true
     * @return array con 1 error por campo, indexado por campo ejemplo[quantity=>'No puede ser negativo'] y [] si no hay errores
     */
    static function validate(array $arrKeyValue, array $rules, array $labels=[], array $messages = [], $estricto = false): array
    {
        $errors = [];
        $method = __METHOD__;
        foreach ($arrKeyValue as $key => $value) {

            if (!isset($rules[$key]))
                continue;

            if (!is_array($rules[$key]))
                $rules[$key] = explode("|", $rules[$key]);

            foreach ($rules[$key] as $key_rule => $val_rule) {
                if (array_key_exists($key_rule, ['label'=>1, 'message'=>1, 'scale'=>1]))
                    continue;

                $parts = is_array($val_rule) ? $val_rule: explode(":", $val_rule);

                $label = $rules[$key]['label'] ?? $labels[$key] ?? self::deduceFieldName($key);

                $type_rule = explode(":", is_numeric($key_rule) ? $val_rule : $key_rule)[0];


                switch($type_rule)
                {
                    case 'required':
                        if ($value === '' || $value === NULL) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'strings':
                    case 'string':
                        if (!is_string($value)) { //@todo is_string('3.1415') es true por las comillas
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'numeric':
                        if (!is_numeric($value)) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'int':
                        if($value == '')
                            $value = '0';
                        if (!is_numeric($value)) {
                            $errors[$key] = self::getMessage(($rules[$key]['message']['numeric']??$messages[$key]['numeric']??self::$messages['numeric']), $label, '', strit($value));
                            break 2;
                        }
                        if (floor($value) != $value) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'date':
                        if (!self::validateYMD($value)) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'bool':
                        if (!in_array($value, ["true", "false", "1", "0", "yes", "no", 1, 0, true, false], true)) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages['bool']), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'max':
                        $max = $rules[$key]['max'] ?? $parts[1];
                        if (is_numeric($value)) {
                            if (bccomp($value, $max, $rules[$key]['scale'] ?? 0) > 0) {
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, $max, strit($value));
                                break 2;
                            }
                        } elseif (self::validateYMD($value)) {
                            $fecha_max = $max;
                            if (!self::validateYMD($max))
                                $fecha_max = date("Y-m-d", strtotime($max));
                            if ($value > $fecha_max){
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages['date_max']), $label, $fecha_max, strit($value));
                                break 2;
                            }
                        } else {
                            if (strlen($value) > $max) {
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages['strlen_max']), $label, $max, strit($value));
                                break 2;
                            }
                        }
                        break;
                    case 'min':
                        $min = $rules[$key]['min'] ?? $parts[1];
                        if (is_numeric($value)) {
                            if (bccomp($value, $min, $rules[$key]['scale'] ?? 2) < 0) {
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, $min, strit($value));
                                break 2;
                            }
                        }
                        /* dado que el limpiador trae date=YYYY-MM-DD y dateTime YYYY-MM-DD HH:NN:SS*/
                        elseif (self::validateYMD($value)) {
                            $fecha_min = $min;
                            if (!self::validateYMD($min))
                                $fecha_min = date("Y-m-d", strtotime($min));

                            if ($value < $fecha_min){
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages['date_min']), $label, $fecha_min, strit($value));
                                break 2;
                            }
                        }
                        else {
                            if (strlen($value) < $min) {
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages['strlen_min']), $label, $min, strit($value));
                                break 2;
                            }
                        }
                        break;
                    case 'in':
                        $in = $rules[$key]['in'] ?? $parts[1];
                        $enum = (is_array($in)) ? $in : explode(",", $in);
                        if ($enum[0] == '@_catalogo') {
                            $tabla = $enum[1];

                            $pkfield = self::fieldit( (isset($enum[2]) and !empty($enum[2])) ? $enum[2] : $tabla . '_id');
                            $tabla = self::fieldit($tabla);
                            $whereCondition = $enum[3] ?? '';
                            $whereCondition = !empty($whereCondition) ? " AND $whereCondition" : '';
                            $query = "SELECT /*$method*/ /*ValidateHelper:validate(in):*/* FROM $tabla WHERE $pkfield = ". strit($value) . $whereCondition;
                            $existe = ia_singleton($query);
                            if (empty($existe)) {
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label,  value: strit($value));
                                break 2; // ojo si if elsif se cambia a switch es 3
                            }
                        } elseif ($enum[0] == '@_sql') {
                            $query = str_replace("{@value}", strit($value), $enum[1]);
                            $existe = ia_singleton($query);
                            if (empty($existe)) {
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                                break 2;
                            }
                        } else {
                            if (!in_array($value, $enum)){
                                $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                                break 2;
                            }
                        }
                        break;
                    case 'enum':
                        $in = $rules[$key]['enum'] ?? $parts[1];
                        $enum = (is_array($in)) ? $in : explode(",", $in);
                        if (!in_array($value, $enum)){
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'unique': case 'unique_on_insert':
                        $_parts = $rules[$key]['unique'] ?? $parts[1];
                        $parts_unique =   (is_array($_parts)) ? $_parts : explode(",", $_parts);
                        $column = self::fieldit( $parts_unique[1] ?? $key );
                        $query = "SELECT /*$method unique insert */ COUNT(*) FROM $parts_unique[0] WHERE $column = " .strit($value);
                        if (!empty(ia_singleread($query))) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'unique_on_update_delete':
                        $_parts = $rules[$key]['unique'] ?? $parts[1];
                        $parts_unique =   (is_array($_parts)) ? $_parts : explode(",", $_parts);
                        $column = self::fieldit( $parts_unique[1] ?? $key );
                        $query = "SELECT /*$method unique update delete*/ COUNT(*) FROM $parts_unique[0] WHERE $column = " .strit($value);
                        if (ia_singleread($query) != '1') {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'sequence':
                        $_parts = $rules[$key]['sequence'] ?? $parts[1];
                        $parts_unique = (is_array($_parts)) ? $_parts : explode(",", $_parts);
                        $column = $parts_unique[1] ?? $key;
                        $query = "SELECT /*$method sequence*/ COUNT(*) FROM $parts_unique[0] WHERE $column = " .strit(bcsub($value, 1, 0));
                        if (empty(ia_singleread($query))) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'equal':
                        $valor_equal = $rules[$key]['equal'] ?? $parts[1];
                        if (strcasecmp($value, $valor_equal) != 0) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    case 'is_equal':
                        $valor_equal = $rules[$key]['is_equal'] ?? $parts[1];
                        if ($value != $valor_equal) {
                            $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                            break 2;
                        }
                        break;
                    default:
                        if($estricto)
                            $errors[$key] = "Para $key Regla desconocida: '$key_rule'";
                        break;
                }
            }

        }
        if($estricto) {
            $type_rule = 'required';
            $arrKeyValueFaltan = array_diff_key($rules, $arrKeyValue);
            if(!empty($arrKeyValueFaltan))
                foreach($arrKeyValueFaltan as $key => $rules) {
                    $label = $rules['label'] ?? $labels[$key] ?? self::deduceFieldName($key);
                    $errors[$key] = self::getMessage(($rules[$key]['message'][$type_rule]??$messages[$key][$type_rule]??self::$messages[$type_rule]), $label, '', strit($value));
                }
        }
        return $errors;
    }



    /**
     * @param string $type
     * @param string $fieldname
     * @param string $max_min
     * @return string
     */
    static private function getMessage(string $message, string $fieldname, string $max_min = '', mixed $value = ''): string
    {
        return str_ireplace(['@fieldname', '@max_min', '{@value}'], [$fieldname, $max_min, $value], $message);
        // return str_replace("@fieldname", $fieldname, str_replace('@max_min', $max_min, self::$messages[$type]));
    }

    /**
     * @param string $type
     * @param string $fieldname
     * @param string $max_min
     * @return string
     */
    /*
    static private function getMessageOption(string $message, string $key = '', array $fieldRules = []): string
    {

        $search = ['@fieldname'];
        $replace = [$rules['label'] ?? $labels[$key] ?? self::deduceFieldName($key)];
        foreach($fieldRules as $k => $d) {
            if(is_array($d))
                continue;
            $search[] = "@$k";
            $replace[] = $d;
        }
        return str_ireplace($search, $replace, $message);
    }
    */

    static private function deduceFieldName(string $fieldname): string
    {
        $fieldname = str_replace("_id", "", $fieldname);
        return str_replace("_", " ", ucwords($fieldname, '_'));
    }

    /**
     * Protect a string to use in Sql like, so % and _ won't have a special value
     *
     * @param string $str
     * @return string
     */
    /*
    static private function strlike(string $str): string
    {
        return str_replace(array('%', '_'), array("\\%", "\\_"), $str);
    }
    */

    /**
     * Protect with ` quotes a: column name to `column name` respecting . table.column to `table`.`column`
     * Protege con ` nombres de columnas y tablas respetando los . table.column a `table`.`column`
     *
     * @param string $fieldName
     * @return string
     */
    static private function fieldit(string $fieldName): string
    {
        $protected = [];
        $n = explode('.',$fieldName);
        foreach($n as $field) {
            $protected[]= '`'.str_replace('`', '', strim($field) ).'`';
        }
        return implode('.', $protected);
    }

    /**
     * @param string $ymd
     * @return bool true ymd is a valid date
     */
    static private function validateYMD(string $ymd):bool {
        if(strlen($ymd) !== 10) {
            return false;
        }
        $dateParts = explode('-', $ymd);
        if(count($dateParts) !== 3) {
            return false;
        }
        foreach($dateParts as $d) {
            if(!is_numeric($d)) {
                return false;
            }
        }
        return checkdate((int)$dateParts[1], (int)$dateParts[2], (int)$dateParts[0]);
    }


}
