<?php

use Mpdf\Mpdf;
use Mpdf\MpdfException;
use Mpdf\Output\Destination;

require_once("../inc/config.php");
require_once("../inc/iaJqGrid/Filter2where.php");
require_once("xlsxwriter.class.php");

class ExportAllRows {
    /** @var array  $params */
    protected array $params;

    protected string $where;
    protected string $orderBy;

    protected array $header = [];
    /** @var array $relate [fieldName => [ id=>label, ..], ... ] */
    protected array $relate = [];
    protected array $perfilReporte = [];
    /** @var array|string[] [$fieldName => 'SELECT id, label...', ]  */
    protected array $pkLabel = [
        'bodega_id' => "SELECT bodega_id, bodega FROM bodega",
        'color_id' => "SELECT color_id, color FROM color",
        'producto_general_id' => "SELECT producto_general_id, producto FROM producto_general",
        'producto_id' => "SELECT producto_general_id, producto FROM producto_general",
        'origen_bodega_id' => "SELECT origen_bodega_id, clave FROM origen_bodega",
        'origen_id' => "SELECT origen_bodega_id, clave FROM origen_bodega",
        'cliente_id' => "SELECT cliente_id, nombre FROM cliente",
        'ayudante_id' => "SELECT ayudante_id, nombre FROM ayudantes",
        'pedido_por_id' => "SELECT ayudante_id, nombre FROM ayudantes",
    ];
    protected array $grandTotals = [];

    /**
     * @param array $request
     */
    public function __construct(array $request) {
        $this->params = $this->getReportParameters($request);
        $this->perfilReporte = $this->readPerfil($this->params['perfil_reporte_grid_id'] ?? '');

        $this->where = $this->getWhere();
        $this->orderBy = $this->getOrderBy();
        $sql = $this->getSql();
        $data = ia_sqlArrayIndx($sql);
        if($data === false) {
            ia_errores_a_dime();
            die("SQL ERROR<pre>$sql</pre>" );
        }
        $this->fix($data);

        $totalsFor = [
            'Quantity', 'Rollos', 'Rolls', 'Total Quantity', 'Total Rollos', 'Total Rolls',
            'quantity', 'rollos', 'rolls', 'total_quantity', 'total_rolls',
            'Saldo', 'Debe',
            'saldo', 'debe',
        ];
        $subTotalsBy = [
            'Moneda', 'Moneda_id', 'Entrada Salida', 'Tipo',
            'moneda', 'moneda_id', 'entrada_salida', 'tipo',
        ];
        $totals = new TotalIt();
        $grandTotals = $totals->grandTotalBy($data, $totalsFor, $subTotalsBy);

        if($this->params['exportType'] === 'PDF') {
            try {
                $this->exportPdf($data, $grandTotals);
            } catch(MpdfException $e) {
                echo "<h1>ERROR AL GENERAR EL PDF</h1><pre>$e</pre>";
            }
        } else {
            $this->exportExcel($data, $grandTotals);
        }
        ia_errores_a_dime();
    }

    /**
     *
     * @param array $data
     * @param array $grandTotals
     * @return void
     * @throws MpdfException
     */
    protected function exportPdf(array $data, array $grandTotals):void
    {
        $base = reset($data);
        foreach($base as &$d) {
            $d = '';
        }
        $data[] = $base;
        $data[] = $base;
        $titula = $base;
        $titula[array_key_first($base)] = "TOTALES";
        $data[] = $titula;
        if(empty($this->grandTotals)) {
            foreach($grandTotals as $for => $subTotalArray) {
                foreach($subTotalArray as $by => $t) {
                    $this->grandTotals[] = [$for, $by, $t['sum']];
                }
            }
        }
        if( !empty($this->grandTotals)) {
            $keys = array_keys($base);
            foreach($this->grandTotals as $t) {
                $i = 0;
                foreach($keys as $k) {
                    $base[$k] = $t[$i];
                    $i++;
                }
                $data[] = $base;
            }
        }

        $title = $this->titleDeduce();
        $fileName = XLSXWriter::sanitize_filename(
                str_replace([".pdf"], "",  $title)
            ) . " ".Date('Y-m-d G_i') . ".$_SESSION[usuario_id]" . ".pdf";

        $pageOrientation = param('pageOrientation','L');
        /** @noinspection DuplicatedCode */
        $constructor = [
            'mode' => 'UTF-8',
            'format' => 'LETTER',
            'default_font_size' => 0,
            'default_font' => '',
            'margin_left' => 5,
            'margin_right' => 5,
            'margin_top' => 5,
            'margin_bottom' => 20,
            'margin_header' => 10,
            'margin_footer' => 5,
            'orientation' => $pageOrientation,
        ];

        $mpdf = new Mpdf($constructor);
        $mpdf->simpleTables = true;
        $mpdf->useSubstitutions = false;
        $mpdf->SetTitle($title);
        $mpdf->SetAuthor("IA");
        $mpdf->SetDisplayMode('fullpage');
        $mpdf->keep_table_proportions = TRUE;

        $mpdf->shrink_tables_to_fit=1;

        $mpdf->AddPageByArray(['orientation' => $pageOrientation ]);
        $mpdf->WriteHTML($this->tableCSS(),1);
        $fecha = Date('d/M/y H:i');
        $mpdf->setFooter("$_SESSION[usuario]|$fecha|{PAGENO} / {nbpg}");
        $mpdf->WriteHTML(iaTableIt::getTableIt($data, $title ),2);
        global $gWebDir;
        $mpdf->Output("c:\\wamp\\www\\$gWebDir\\backoffice\\pdfs\\$fileName", Destination::FILE);
        header("Content-Type: application/pdf");
        header("Content-Disposition: attachment; filename='$fileName.pdf'");
        header("Location: ../backoffice/pdfs/$fileName");
    }

    protected function tableCSS() {
        return <<< CSSTEXT
        .tableitdiv{overflow-x:auto;border:4px blue double;display:inline-block;padding:1em}
        .tableit,  {border: 1px solid silver}
         .tableit {border-collapse:collapse;border-spacing:0;margin:auto;overflow-x:hidden;empty-cells:show}
        .tableit TH, .tableit TD {margin:1em}
       
        .tableit CAPTION {
            font-weight:bold;font-size:larger;color:blue; background-color:white;
        }
        .tableit THEAD {margin-top:0;margin-bottom:0}
        .tableit TH {padding:4px;font-weight:bold; border:none;border-bottom:1px silver solid!important}
        .tableit TD {padding:4px;vertical-align:top;border:none;}
        .tableit TR:nth-child(even) {}
        
        .tableit .lft {text-align:left}
        .tableit .izq {text-align:left}
        .tableit .cen {text-align:center}
        .tableit .rgt {text-align:right}
        .tableit .der {text-align:right}
        
        .tableit .date {text-align:center;white-space:pre-line}
        .tableit .red {color:red}
        .tableit .green{color:darkgreen}
        .tableit .blue{color:blue}
        .tableit .remark{background-color:yellow}
CSSTEXT;
    }

    /**
     * @param array $data
     * @param array $grandTotals
     * @return void
     */
    protected function exportExcel(array $data, array $grandTotals):void {
        $writer = new XLSXWriter();
        if(empty($this->header)) {
            foreach(reset($data) as $k => $_) {
                if(is_numeric($_)) {
                    $style = str_contains($_, '.') ? "#,##0.00" : "#,##0";
                } else {
                    $style = "@";
                }
                $this->header[$k] = $style;
            }
        }
        $writer->writeSheetHeader('Sheet1', $this->header );
        foreach($data as $row) {
            $writer->writeSheetRow('Sheet1', $row);
        }


        if(empty($this->grandTotals)) {
            $totals = [];
            foreach($grandTotals as $for => $subTotalArray) {
                foreach($subTotalArray as $by => $t) {
                    $totals[] = [$for, $by, $t['sum']];
                }
            }
            $writer->writeSheet($totals, 'Totals', ['Total'=>'@', 'Sub Total'=>'@', ''=>'#,##0.00'] );
        } else {
            $writer->writeSheet($this->grandTotals, 'Totals', ['Total'=>'@', 'Sub Total'=>'@', ''=>'#,##0.00'] );
        }

        $fileName = XLSXWriter::sanitize_filename(
          str_replace([".xls", ".xlsx"], "",  $this->params['fileName'] ?? 'download')
        );
        $fileName .= "_" . Date('Y_m_d') . ".xlsx";
        header('Content-Disposition: attachment; filename="'. $fileName .'"');
        header("Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        header('Content-Transfer-Encoding: binary');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        $writer->writeToStdOut();
    }

    /**
     * Arregla por reporte específico o en general: resuleve *_id, jsons con es_error
     *
     * @param array $data
     * @return void
     */
    protected function fix(array &$data):void {
        switch($this->params['perfil_reporte_grid_id'] ?? '') {
            case '74867af2fe60a1cd11ecf349184678bd':
                $this->fixNotaBodega($data);
                return;
        }
        foreach($data as  &$row) {
            foreach($row as $fieldName => &$d) {
                if(empty($d) || $d === '[]' || $d === '{}') {
                    $d = '';
                    continue;
                }
                if(strcasecmp($fieldName, 'tipo') === 0 && strcasecmp($d, 'Cancelacion') === 0) {
                    foreach(['total_quantity','quantity','Quantity', 'total_rolls', 'rolls', 'Rolls', 'rollos', 'Rollos'] as $emptyMe) {
                        if(array_key_exists($emptyMe, $row)) {
                            $row[$emptyMe] = '';
                        }
                    }
                }
                if(str_starts_with($d ?? '', "[{")) {
                    $d =  $this->jsonDisplay($d);
                    continue;
                }
                if( !array_key_exists($fieldName, $this->pkLabel)) {
                    continue;
                }
                if( !array_key_exists($fieldName, $this->relate)) {
                    $this->relate[$fieldName] = ia_sqlKeyValue($this->pkLabel[$fieldName]);
                }
                $d = $this->relate[$fieldName][$d];
            }
        }
    }

    /**
     * Reporte especial perfil Notas de Bodega
     *
     * @param array $data
     * @return void
     */
    protected function fixNotaBodega(array &$data):void {
        $this->header = array(
            'Fecha'=>'date',
            'Numero'=>'integer',
            'Num Compra'=>'string',
            'Bodega'=>'string',
            'Tipo'=>'string',
            'Producto'=>'string',
            'Colores'=>'string',
            'Quantity'=>'#,##0.00',
            'Rolls'=>'#,##0',
            '# Colores'=>'#,##0',
        );
        foreach($data as &$d) {
            if(strcasecmp('Cancelacion', $d['Tipo']) === 0) {
                $d['Rollos'] = $d['Quantity'] = '';
            }
            if(empty($d['numero_compra'])) {
                continue;
            }
            $d['numero_compra'] =  $this->jsonDisplay($d['numero_compra']);
        }
        $tot = ['Quantity' => 0, 'Rollos' => 0, 'Notas' => 0 ];
        $this->grandTotals = [];
        $whereClause = empty($this->where) ? " WHERE tipo <> 'Cancelacion'" : "$this->where AND tipo <> 'Cancelacion'";
        $sums = ia_sqlArrayIndx(
            "SELECT entrada_salida, COUNT(*) as cuenta, SUM(total_quantity) as Quantity, SUM(total_rolls) as Rollos
            FROM nota_bodega $whereClause ORDER BY 1 DESC");
        foreach($sums as $t) {
            $this->grandTotals[] = [$t['entrada_salida'], 'Quantity', $t['Quantity'], '', $t['cuenta'], 'Notas' ];
            $this->grandTotals[] = [$t['entrada_salida'], 'Rollos', $t['Rollos'], ''  ];
            $tot['Notas'] += $t['cuenta'];
            if(strcasecmp('Salida', $t['entrada_salida']) === 0) {
                $tot['Quantity'] += $t['Quantity'];
                $tot['Rollos'] += $t['Rollos'];
            } else {
                $tot['Quantity'] -= $t['Quantity'];
                $tot['Rollos'] -= $t['Rollos'];
            }
        }
        $this->grandTotals[] = ['Total Venta', 'Quantity', $tot['Quantity'] ];
        $this->grandTotals[] = ['Total Venta', 'Rollos', $tot['Rollos'] ];
        $this->grandTotals[] = ['Total', 'Notas', $tot['Notas'], ' no canceladas' ];
    }

    /**
     * Despliega del json display, label, lbl o nombre cuando fue_error !=SI
     *
     * @param string|null $str
     * @return string
     */
    protected function jsonDisplay(string|null $str): string {
        if($str === null) {
            return '';
        }
        $str = trim($str);
        if($str === '' || $str === '[]' || $str === '{}') {
            return '';
        }
        $json = json_decode($str, true);
        if(empty($json)) {
            return '';
        }
        $ret = [];
        foreach($json as $arr) {
            if(($arr['fue_error'] ?? '') === 'SI') {
                continue;
            }
            foreach(['display', 'label', 'lbl', 'nombre', 'numero_compra'] as $f) {
                if(array_key_exists($f, $arr)) {
                    $ret[] = $arr[$f];
                    break;
                }
            }
        }
        return implode(", ", $ret);
    }

    /**
     * Obtiene el sql select del colModel o por perfil_reporte_grid_id
     *
     * @return string
     */
    protected function getSql():string {
        $table = $this->params['allPostData']['iactbl'] ?? 'table';
        $sqlPerfil = $this->perfil2Sql($table, $this->params['perfil_reporte_grid_id'] ?? '');
        if(!empty($sqlPerfil)) {
            return $sqlPerfil;
        }
        return $this->colModel2Sql($table, $this->params['allColModel']);
    }

    /**
     * Obtiene el sql select del colModel
     *
     * @param string $table
     * @param array $colModel
     * @return string
     */
    protected function colModel2Sql(string $table, array $colModel):string {
        $cols = [];
        foreach($colModel as $c) {
            $name = $c['name'];
            if(strcasecmp('rn', $name) === 0 ||
                strcasecmp('action', $name) === 0 ||
                strcasecmp('actions', $name) === 0 ||
                strcasecmp('subgrid', $name) === 0 ||
                strcasecmp('iacsel', $name) === 0
            ) {
                continue;
            }
            if($c['hidden'] != 1) {
                $cols[] = fieldit($c['index'] ?? $name);
            }
        }
        $columns = implode(', ', $cols);
        if(empty($columns)) {
            $columns = " * ";
        }
        $method = __METHOD__;
        return "SELECT /*$method*/ $columns\r\nFROM $table\r\n$this->where $this->orderBy";
    }

    /**
     * Deduce el sql select del perfil_reporte_grid_id o toma excepcion a mano de knownPerfilSql()
     *
     * @param string $table
     * @param string $perfil_reporte_grid_id
     * @return string
     */
    protected function perfil2Sql(string $table, string $perfil_reporte_grid_id):string {
        if(empty($perfil_reporte_grid_id)) {
            return '';
        }
        $sql = $this->knownPerfilSql($perfil_reporte_grid_id);
        if(!empty($sql)) {
            return $sql;
        }
        $method = __METHOD__;
        $cols = [];
        $perfil = $this->perfilReporte;
        if(empty($perfil)) {
            return '';
        }
        $colModel = [];
        foreach($this->params['allColModel'] as $c) {
            $name = $c['name'];
            $colModel[$c['label'] ?? $name] = $c['index'] ?? $name;
        }
        foreach($perfil['columnas'] as $c) {
            $cols[] = ($colModel[$c] ?? 'ERROR') . ' as ' . strit($c);
        }
        $columns = implode(', ', $cols);
        $table = fieldit($table);
        return "SELECT /*$method*/ $columns\r\nFROM\r\n$table $this->where\r\n$this->orderBy";
    }

    /**
     * Select sql a mano para excepciones por $perfil_reporte_grid_id
     *
     * @param string $perfil_reporte_grid_id
     * @return string
     */
    protected function knownPerfilSql(string $perfil_reporte_grid_id):string {
        $method = __METHOD__;
        switch($perfil_reporte_grid_id) {
            case '74867af2fe60a1cd11ecf349184678bd':
                $whereClause = str_replace(
                    ['`bodega_id`', 'bodega_id',
                        '`producto_general_id`', 'producto_general_id'],
                    ['nota_bodega.bodega_id', 'nota_bodega.bodega_id',
                        'nota_bodega.producto_general_id', 'nota_bodega.producto_general_id'],
                    $this->where
                );
                $orderByClause = str_replace(
                    ['bodega_id', 'producto_general_id'],
                    ['nota_bodega.bodega_id', 'nota_bodega.producto_general_id'],
                    $this->orderBy
                );
                return <<< SQL
            SELECT /*$method*/ Fecha, Numero, numero_compra, Bodega, 
                   IF(tipo = 'Movimiento', 'Venta Tienda', tipo) as Tipo, 
                   Producto, Colores, total_quantity as Quantity, total_rolls as Rollos,
                   num_colores as '# Colores'
                FROM nota_bodega 
                    JOIN bodega b on nota_bodega.bodega_id = b.bodega_id
                    JOIN producto_general pg on nota_bodega.producto_general_id = pg.producto_general_id
            $whereClause $orderByClause
SQL;

            default:
                 return '';
        }
    }

    /**
     * Deduce el order by solicitado.
     *
     * @return string
     */
    protected function getOrderBy():string {
        $ords = [];
        $orderdBy = ($this->params['allPostData']['sidx'] ?? '') . ' ' . ($this->params['allPostData']['sord'] ?? '');
        foreach(explode(',', $orderdBy) as $o) {
            $expr = explode(' ', strim($o));
            $say = $expr[0];
            if(isset($expr[1]) && strcasecmp('DESC', $expr[1]) === 0 ) {
                $say .= ' DESC';
            }
            $ords[] = $say;
        }
        if(empty($ords)) {
            return '';
        }
        return ' ORDER BY ' . implode(', ', $ords);
    }

    /**
     * Arma el where del grid considerando iac_where y postdata.filter
     *
     * @return string
     */
    protected function getWhere():string {
        $postData = $this->params['allPostData'];
        $w = [];
        if(!empty($postData['iacwhere'])) {
            $w[] = "($postData[iacwhere])";
        }
        $filters = (new Filter2where())->filter2where($postData['filters'] ?? '');
        if(!empty($filters)) {
            $w[] = "($filters)";
        }

        return empty($w) ? '' : " WHERE " . implode(" AND ", $w );
    }

    /**
     * Obtiene los datos para el $perfil_reporte_grid_id
     *
     * @param string|null $perfil_reporte_grid_id
     * @return array
     */
    protected function readPerfil(string|null $perfil_reporte_grid_id):array {
        if($perfil_reporte_grid_id === '' || $perfil_reporte_grid_id === null) {
            return [];
        }
        $method = __FUNCTION__;
        $perfil_reporte_grid = ia_singleton(
            "SELECT /*$method*/ perfil_reporte_grid_id, perfil,reporte_grid_id 
                FROM perfil_reporte_grid 
                WHERE perfil_reporte_grid_id=".strit($perfil_reporte_grid_id));
        if(empty($perfil_reporte_grid)) {
            return [];
        }
        $sqlReporte =
            "SELECT /*$method*/ origen,nombre,descripcion,arrTotales,arrGroupBy 
            FROM reportes_grid 
            WHERE reporte_grid_id=" . strit($perfil_reporte_grid['reporte_grid_id']);
        $sqlColsPerfil =
            "SELECT /*$method*/ columna 
            FROM perfil_reporte_grid_col 
            WHERE perfil_reporte_grid_id=".strit($perfil_reporte_grid_id) .
            " ORDER BY orden, columna";
        return [
            'perfil' => $perfil_reporte_grid,
            'reporte' => ia_singleton($sqlReporte),
            'columnas' => ia_sqlVector($sqlColsPerfil)
        ];
    }

    /** Normaliza los valores del request */
    protected function getReportParameters(array $request):array {
        $getParams =  [
            'perfil_reporte_grid_id',
            'exportType',
            'fileName',
            'titulo',
            'allColModel',
            'allPostData', // [ iaccols:'', iactbl:'', iacwhere:'', sidx:'', sord:'']

            'totalsOnly',

            'totales',
            'groupby',
            'datosAdicionales',
            'origen',
            'hijos',
            'campoID',
            'extraInfo',

        ];
        foreach($getParams as $p) {
            $ret[$p] = $request[$p] ?? '';
        }
        if(!empty($ret['allPostData'])) {
            $ret['allPostData'] = json_decode($ret['allPostData'], true);
        } else {
            $ret['allPostData'] = [];
        }
        if(!empty($ret['allColModel'])) {
            $ret['allColModel'] = json_decode($ret['allColModel'], true);
        } else {
            $ret['allColModel'] = [];
        }
        return $ret;
    }

    protected function titleDeduce():string {
        $titulo = $this->perfilReporte['perfil']['perfil'] ?? $this->params['titulo'];
        return empty($titulo) ?
            str_replace(['.pdf', '.xls'], '', $this->params['fileName']) :
            $titulo;
    }
}

