<?php
use React\EventLoop\Loop;
// Integrate WsClient into the execution flow for resilient async calls
// Resilient function for handling WebSocket or fallback behavior
function resilient_async_call(
    string $websocketUrl,
    string $fallbackUrl,
    array $params = [],
    int $timeout = 5,
    int $retryTimeout = 15,
    bool $waitForResponse = false,
    int $maxRetries = 3
): bool {
    global $gIAParametros;

    // Check if WebSocket server usage is disabled
    if (isset($gIAParametros['use_ws_server']) && !$gIAParametros['use_ws_server']) {
        $func = $params['func'] ?? null;
        $func_params = $params['func_params'] ?? [];

        if ($func && function_exists($func)) {
            async_ws_log_event("Executing fallback function: $func");
            return call_user_func($func, $func_params);
        }

        async_ws_log_event("Fallback function $func does not exist.");
        return false;
    }

    $wsSuccess = false;
    $attempts = 0;
    $effectiveTimeout = min($timeout, $retryTimeout); // Adjust timeout for retries
    $loop = Loop::get();

    $retry = function () use (
        &$attempts,
        $websocketUrl,
        $params,
        &$wsSuccess,
        $fallbackUrl,
        $effectiveTimeout,
        $loop,
        $waitForResponse,
        $maxRetries
    ) {
        $attempts++;
        async_ws_log_event("WebSocket attempt #$attempts for URL $websocketUrl with params: " . json_encode($params));

        try {
            $wsClient = new WsClient($websocketUrl);
            $wsClient->connectAndRun($params, $waitForResponse);

            if (!$waitForResponse) {
                // Skip loop stop if fire-and-forget is required
                async_ws_log_event("Fire-and-forget mode active. Not waiting for response. Params: " . json_encode($params));
                return;
            }

            $wsSuccess = true;
        } catch (Exception $e) {
            async_ws_log_event("Attempt #$attempts failed: " . $e->getMessage());
            if ($attempts < $maxRetries) {
                $loop->addTimer($effectiveTimeout, $retry);
            }
        }

        if (!$wsSuccess && $attempts >= $maxRetries) {
            async_ws_log_event("All WebSocket attempts failed. Falling back to async function.");
            if (function_exists('get_async')) {
                get_async($fallbackUrl, $params['func_params'] ?? []);
            } else {
                async_ws_log_event("Error: get_async function is not available.");
            }
        }
    };

    $retry();
    if ($waitForResponse) {
        // Only run the loop if waiting for a response
        $loop->run();
    }

    return $wsSuccess;
}


function execute_with_resilience(string $func, string $fallbackScript, array|string $params, int $timeout = 5, $waitForResponse = false): bool
{
    global $gWebDir, $gIAParametros;

    // Wrap scalar params into an array
    if (!is_array($params)) {
        $params = [$params];
    }

    // Check if WebSocket server usage is disabled
    if (true || isset($gIAParametros['use_ws_server']) && !$gIAParametros['use_ws_server']) {
        if ($func && function_exists($func)) {
            async_ws_log_event("Executing fallback function: $func");
            call_user_func($func, $params);
            return true;
        }
        return false;
    }

    $fallbackUrl = "http://localhost:80/$gWebDir/backoffice/chron/{$fallbackScript}.php";

    $ws_params = [
        'usuario_id' => $_SESSION['usuario_id'] ?? null,
        'usuario' => $_SESSION['usuario'] ?? null,
        'accion' => 'run_async',
        'func' => $func,
        'func_params' => $params,
    ];

    return resilient_async_call('ws://127.0.0.1:2000', $fallbackUrl, $ws_params, $timeout, 1, $waitForResponse);
}
function async_ws_log_event(string $message): void {
    return;
    ia_errores_a_dime('resilient_async.log', "[" . date('Y-m-d H:i:s') . "] $message\n", FILE_APPEND);
}
function async_ws_generic_no_params(string $funcName, string $fallbackScript): void
{
    execute_with_resilience($funcName, $fallbackScript, []);
}
// Specific Functions Refactored
function async_ws_actualizaLinksAlBancoLive($force = 'NO'): void
{
    execute_with_resilience('async_actualizaLinksAlBancoLive', 'actualiza_links_al_banco_live', $force);
}

function async_ws_generaCatalogos($catalogos = ['QWERTY']): void
{
    execute_with_resilience('async_generaCatalogos', 'generacat', $catalogos);
}

function async_ws_lock_nota_bodega(): void
{
    async_ws_generic_no_params('async_lock_nota_bodega', 'lock_nota_bodega');
}

function async_ws_actualiza_clientes_saldos($cliente_id = null): void
{
    execute_with_resilience('async_actualiza_clientes_saldos', 'actualiza_clientes_saldos', array('cliente_id' => $cliente_id));
}

function async_ws_actualiza_clientes_saldos_grid($categoria_id = null): void
{
    execute_with_resilience('async_actualiza_clientes_saldos_grid', 'actualiza_clientes_saldos_grid', array('categoria_id' => $categoria_id));
}

function async_ws_actualizaFiduciariosAtrasados(): void
{
    async_ws_generic_no_params('async_actualizaFiduciariosAtrasados', 'fidatrasados');
}

function async_ws_releaseBackofficeFile2EdoCtaAllLock(): void
{
    async_ws_generic_no_params('async_releaseBackofficeFile2EdoCtaAllLock', 'releasebackofficefile2edoctaalllock');
}
function async_ws_actualizaInversionesAtrasadas(): void
{
    async_ws_generic_no_params('async_actualizaInversionesAtrasadas', 'invatrasadas');
}

function async_ws_actualizaFacturasCanceladas(): void
{
    async_ws_generic_no_params('async_actualizaFacturasCanceladas', 'facturas');
}

function async_ws_actualizaGastosImportantes(): void
{
    async_ws_generic_no_params('async_actualizaGastosImportantes', 'gastosimportantes');
}

function async_ws_actualizaWithdrawalsImportantes(): void
{
    async_ws_generic_no_params('async_actualizaWithdrawalsImportantes', 'withdrawalsimportantes');
}

function async_ws_mergeCtaTMovs(): void
{
    async_ws_generic_no_params('async_mergeCtaTMovs', 'actualiza_cuentat_mov_live');
}

function async_ws_actualizaIngresos(): void
{
    async_ws_generic_no_params('async_actualizaIngresos', 'actualiza_ingresos');
}

function async_ws_checkAndRunAts(): void
{
    async_ws_generic_no_params('async_checkAndRunAts', 'checkAndRunAts');
}

function async_ws_actualizaACuentasLive($force = 'NO'): void
{
    execute_with_resilience('async_actualizaACuentasLive', 'actualiza_a_cuentas_live', $force);
}

function async_ws_guardaConsulta($sqlData = ""): void
{
    execute_with_resilience('async_guardaConsulta', 'guardaconsulta', $sqlData);
}

function async_ws_guarda_despliega_autorizaciones($sqlData = ""): void
{
    execute_with_resilience('async_guarda_despliega_autorizaciones', 'guarda_despliega_autorizaciones', $sqlData);
}

function async_ws_ejecuta_transaccion_sql($sqlData = ""): void
{
    execute_with_resilience('async_ejecuta_transaccion_sql', 'ejecuta_transaccion_sql', $sqlData);
}

function async_ws_actualizaGastosPorRevisar(): void
{
    async_ws_generic_no_params('async_actualizaGastosPorRevisar', 'actualiza_gastos_por_revisar');
}

function async_ws_actualizaConteoMovimientosDivisa(): void
{
    async_ws_generic_no_params('async_actualizaConteoMovimientosDivisa', 'actualiza_conteo_mov_divisa');
}

function async_ws_generaHistorian(string $tabla, string $id, string $action, string $user_nick): void
{
    execute_with_resilience('async_generaHistorian', 'genera_historian', ['tabla' => $tabla, 'id' => $id, 'action' => $action, 'user_nick' => $user_nick]);
}

function async_ws_disable_tc_temp_timeout(): void
{
    async_ws_generic_no_params('async_disable_tc_temp_timeout', 'disable_tc_temp_time_out');
}

function async_ws_banco_cuenta_mov_puede_borrar(): void
{
    execute_with_resilience('async_banco_cuenta_mov_puede_borrar', 'disable_tc_temp_time_out', ['function' => 'timeout_puede_borrar']);
}

function async_ws_updateMaxMinCountAll(): void
{
    async_ws_generic_no_params('async_updateMaxMinCountAll', 'updateMaxMinCountAllBodegas');
}

function async_ws_recalculaInconsistenciasEnNotas($cliente_id = '', $cliente_nombre = ''): void
{
    execute_with_resilience('async_recalculaInconsistenciasEnNotas', 'recalculaInconsistenciasEnNotas', ['cliente_id' => $cliente_id, 'cliente_nombre' => $cliente_nombre]);
}

function async_ws_update_plantillas_withdrawals(): void
{
    async_ws_generic_no_params('async_update_plantillas_withdrawals', 'update_plantillas_withdrawals');
}

function async_ws_update_plantillas_cuenta_t_gasto(): void
{
    async_ws_generic_no_params('async_update_plantillas_cuenta_t_gasto', 'update_plantillas_cuenta_t_gasto');
}
