<?php
use Iac\inc\sql\IacSqlBuilder;
function getCommodityMeta(string $code = "XYZ"): ?array {
    $map = [
        'OIL' => [
            'code' => 'OIL',
            'currency'      => 'USD',
            'friendly_name' => 'Crude Oil WTI (OIL)',
            'investing_com' => 'crude-oil',
            'url' => 'https://www.investing.com/commodities/crude-oil-historical-data',
            'type' => 'Commodity',
            'original_file' => 'Crude Oil WTI (OIL)'
        ],
        'CTN5' => [
            'code' => 'CTN5',
            'currency'      => 'USD',
            'friendly_name' => 'US Cotton #2 (CTN5)',
            'investing_com' => 'us-cotton-no.2',
            'url' => 'https://www.investing.com/commodities/us-cotton-no.2-historical-data',
            'type' => 'Commodity',
            'original_file' => 'US Cotton #2 (CTN5)'
        ],
        'CCYc1' => [
            'code' => 'CCYc1',
            'currency'      => 'CNY',
            'friendly_name' => 'ZCE Cotton Yarn (CCYc1)',
            'investing_com' => 'zce-cotton-yarn-future',
            'url' => 'https://www.investing.com/commodities/zce-cotton-yarn-futures-historical-data',
            'type' => 'Commodity',
            'original_file' => 'ZCE Cotton Yarn (CCYc1)'
        ],
        'USD/CNY' => [
            'code' => 'USD/CNY',
            'currency'      => 'CNY',
            'friendly_name' => 'USDCNY - US Dollar Chinese Yuan',
            'investing_com' => 'usd-cny',
            'url' => 'https://www.investing.com/currencies/usd-cny-historical-data',
            'type' => 'Currency',
            'original_file' => 'USDCNY - US Dollar Chinese Yuan'
        ],
        '603029' => [
            'code' => '603029',
            'currency'      => 'CNY',
            'friendly_name' => 'Shandong Swan Cotton Industrial Machinery Stock Co Ltd (603029)',
            'investing_com' => 'shandong-swan-cotton-industrial-mac',
            'url' => 'https://www.investing.com/equities/shandong-swan-cotton-industrial-mac-historical-data',
            'type' => 'Equity',
            'original_file' => 'Shandong Swan Cotton Industrial Machinery Stock Co Ltd (603029)'

        ],
        '002042' => [
            'code' => '603029',
            'currency'      => 'CNY',
            'friendly_name' => 'Huafu Top Dyed Melange Yarn Co Ltd (002042)',
            'investing_com' => 'huafu-melange-a',
            'url' => 'https://www.investing.com/equities/huafu-melange-a-historical-data',
            'type' => 'Equity',
            'original_file' => 'Huafu Top Dyed Melange Yarn Co Ltd (002042)'
        ],
        'WGSR' => [
            'code' => 'WGSR',
            'currency'      => 'INR',
            'friendly_name' => 'Welspun Corp Ltd (WGSR)',
            'investing_com' => 'welspun-corp',
            'url' => 'https://www.investing.com/equities/welspun-corp-historical-data',
            'type' => 'Equity',
            'original_file' => 'Welspun Corp Ltd (WGSR)'
        ],
    'VART' => [
        'code' => 'VART',
        'currency'      => 'INR',
        'friendly_name' => 'Vardhman Textiles Ltd (VART)',
        'investing_com' => 'vardhman-textiles-ltd',
        'url' => 'https://www.investing.com/equities/vardhman-textiles-ltd-historical-data',
        'type' => 'Equity',
        'original_file' => 'Vardhman Textiles Ltd (VART)'
    ],
    'RYMD' => [
        'code' => 'RYMD',
        'currency'      => 'INR',
        'friendly_name' => 'Raymond Ltd (RYMD)',
        'investing_com' => 'raymond',
        'url' => 'https://www.investing.com/equities/raymond-historical-data',
        'type' => 'Equity',
        'original_file' => 'Raymond Ltd (RYMD)'
    ],
    'TNG' => [
        'code' => 'TNG',
        'currency'      => 'VND',
        'friendly_name' => 'Tng Investment And Trading JSC (TNG)',
        'investing_com' => 'tng-investment-and-trading-jsc',
        'url' => 'https://www.investing.com/equities/tng-investment-and-trading-jsc-historical-data',
        'type' => 'Equity',
        'original_file' => 'Tng Investment And Trading JSC (TNG)'
    ],

    'TCM' => [
        'code' => 'TCM',
        'currency'      => 'VND',
        'friendly_name' => 'Thanh Cong Textile Garment Investment Trading JSC (TCM)',
        'investing_com' => 'thanh-cong-textile-garment-invest',
        'url' => 'https://www.investing.com/equities/thanh-cong-textile-garment-invest-historical-data',
        'type' => 'Equity',
        'original_file' => 'Thanh Cong Textile Garment Investment Trading JSC (TCM)'
    ],
    'VDN' => [
        'code' => 'VDN',
        'currency'      => 'VND',
        'friendly_name' => 'Vinatex DaNang JSC (VDN)',
        'investing_com' => 'vinatex-danang',
        'url' => 'https://www.investing.com/equities/vinatex-danang-historical-data',
        'type' => 'Equity',
        'original_file' => 'Vinatex DaNang JSC (VDN)'
    ],
    'CNY/VND' => [
        'code' => 'CNY/VND',
        'currency'      => 'VND',
        'friendly_name' => 'CNY/VND - Chinese Yuan Vietnamese Dong',
        'investing_com' => 'cny-vnd',
        'url' => 'https://www.investing.com/currencies/cny-vnd-historical-data',
        'type' => 'Currency',
        'original_file' => 'CNYVND - Chinese Yuan Vietnamese Dong'
    ],

];

    return $map[$code] ?? $map;
}

function saveCommodityPrice(array $data = [], array $meta_commodity = [], $debug = false ): false|array
{
    if (!$data || !isset($data['price'], $data['date'], $data['code'])) {
        return false;
    }

    if (!$meta_commodity || empty($meta_commodity['code'])) {
        return false;
    }

    // Build clean insert array with enriched metadata
    $arr_com = $data;
    $arr_com['alta_por'] = $_SESSION['usuario'] ?? 'sistema';
    if ($debug) echo "<pre>[arr_com]\n".print_r($arr_com, true)."</pre>";
    $sqlCom = [];
    $sqlCom[] = ia_insert('stock_prices', $arr_com, [], '',$autoOnUpdate=true);
    if ($debug) echo "<pre>[sqlCom]\n".print_r($sqlCom, true)."</pre>";
    ia_transaction($sqlCom);

    return $sqlCom;
}

function extract_usdcny_prices(string $html, $code = 'USD/CNY', $currency = 'CNY', $friendly_name = 'US Dollar Chinese Yuan'): array
{
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date'  => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open'  => parse_price($cells->item(2)->nodeValue),
            'high'  => parse_price($cells->item(3)->nodeValue),
            'low'   => parse_price($cells->item(4)->nodeValue),
            'volume'   => parse_price($volume_arr[0]),
            'volume_unit'   => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => "Currency",
        ];
    }

    return $data;
}

function extract_shandong_swan_cotton_prices(string $html, $code = '603029', $currency = 'CNY', $friendly_name = 'Shandong Swan Cotton Industrial Machinery Stock Co Ltd'): array
{
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date'  => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open'  => parse_price($cells->item(2)->nodeValue),
            'high'  => parse_price($cells->item(3)->nodeValue),
            'low'   => parse_price($cells->item(4)->nodeValue),
            'volume'   => parse_price($volume_arr[0]),
            'volume_unit'   => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => "Equity",
        ];
    }

    return $data;
}
function extract_huafutopdyed_prices(string $html, $code = '002042', $currency = 'CNY', $friendly_name = 'Huafu Top Dyed Melange Yarn Co Ltd (002042)'): array
{
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date'  => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open'  => parse_price($cells->item(2)->nodeValue),
            'high'  => parse_price($cells->item(3)->nodeValue),
            'low'   => parse_price($cells->item(4)->nodeValue),
            'volume'   => parse_price($volume_arr[0]),
            'volume_unit'   => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => "Equity",
        ];
    }

    return $data;
}

function extract_welspuncorpltd_prices(string $html, $code = 'WGSR', $currency = 'INR', $friendly_name = 'Welspun Corp Ltd (WGSR)'): array
{
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date' => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open' => parse_price($cells->item(2)->nodeValue),
            'high' => parse_price($cells->item(3)->nodeValue),
            'low' => parse_price($cells->item(4)->nodeValue),
            'volume' => parse_price($volume_arr[0]),
            'volume_unit' => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => "Equity",
        ];
    }
}
function extractNumberAndUnit(string $input): array {
    // Use regex to split number and unit
    preg_match('/^([\d.]+)([a-zA-Z]*)$/', $input, $matches);

    // Return structured output
    return [
        isset($matches[1]) ? (float)$matches[1] : 0,
        isset($matches[2]) ? $matches[2] : ''
    ];
}

function extract_yarn_prices(string $html, $code = 'CCYc1', $currency = 'CNY', $friendly_name = 'ZCE Cotton Yarn'): array
{
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date'  => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open'  => parse_price($cells->item(2)->nodeValue),
            'high'  => parse_price($cells->item(3)->nodeValue),
            'low'   => parse_price($cells->item(4)->nodeValue),
            'volume'   => parse_price($volume_arr[0]),
            'volume_unit'   => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => "Equity",
        ];
    }

    return $data;
}

function extract_oil_prices(string $html, $code = 'OIL', $currency = 'USD', $friendly_name = 'Crude Oil WTI'): array
{
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date'  => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open'  => parse_price($cells->item(2)->nodeValue),
            'high'  => parse_price($cells->item(3)->nodeValue),
            'low'   => parse_price($cells->item(4)->nodeValue),
            'volume'   => parse_price($volume_arr[0]),
            'volume_unit'   => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => "Commodity",
        ];
    }

    return $data;
}

function extract_cotton_prices(string $html, $code = 'CTN5', $currency = 'USD', $friendly_name = 'US Cotton #2'): array
{
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date'  => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open'  => parse_price($cells->item(2)->nodeValue),
            'high'  => parse_price($cells->item(3)->nodeValue),
            'low'   => parse_price($cells->item(4)->nodeValue),
            'volume'   => parse_price($volume_arr[0]),
            'volume_unit'   => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => "Commodity",
        ];
    }

    return $data;
}
function parse_price(string $str): float
{
    return (float) str_replace([',', ' '], '', $str);
}

function getCommodityURL(string $commodity): ?string
{
    $urlMap = [
        'cotton' => 'https://www.investing.com/commodities/us-cotton-no.2-historical-data',
        'oil' => 'https://www.investing.com/commodities/crude-oil-historical-data',
        'yarn' => 'https://www.investing.com/commodities/zce-cotton-yarn-futures-historical-data',
        'usdcny' => 'https://www.investing.com/currencies/usd-cny-historical-data',
        'shandong_swan_cotton' => 'https://www.investing.com/equities/shandong-swan-cotton-industrial-mac-historical-data',
        'huafutopdyed' => 'https://www.investing.com/equities/huafu-melange-a-historical-data',
        'welspuncorpltd_prices' => 'https://www.investing.com/equities/welspun-corp-historical-data',
    ];

    return $urlMap[$commodity] ?? null;
}
function getCommodityPriceHTML($meta_commodity = [], $debug = false): ?array
{
    if (empty($meta_commodity)) return null;

    $url = $meta_commodity['url'];

    $headers = [
        'Content-Type: text/html'
    ];

    $agent = 'Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/69.0.3497.12 Safari/537.36 OPR/56.0.3051.0 (Edition developer)';

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_USERAGENT      => $agent,
    ]);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $commodity_type = $meta_commodity['type'];

    if ($debug) {
        if($commodity_type == 'Equity')
            echo "<pre>[Response] HTTP Code: $http_code\n</pre>";
        else
            echo "<pre>[Response] HTTP Code: $http_code\n" . print_r($response, true) . "</pre>";
    }

    $data = $response;
    if (!$data) return null;

    return [
        'timestamp' => date('Y-m-d H:i:s'),
        'html' => $data,
        'http_code' => $http_code,
        'url' => $url,
        'meta_commodity' => $meta_commodity,
    ];
}
function fetchAndSaveLatestCommodity(string $code = '', $all = false, $debug = false): false|array
{
    if(empty($code)) return false;


    $meta_commodity = getCommodityMeta($code);
    if (!$meta_commodity || empty($meta_commodity['code'])) {
        return false;
    }
    // Step 1: Get raw HTML
    $html_commodity = getCommodityPriceHTML($meta_commodity, $debug);
    if (empty($html_commodity['html'])) return false;

    // Step 2: Extract structured data from HTML
    $data_commodity = extract_stock_prices($html_commodity['html'], $meta_commodity);
    if ($debug) echo "<pre>extract_stock_prices\n".print_r($data_commodity, true)."</pre>";
    if (empty($data_commodity)) return false;

    // Step 3: Sort and get latest record
    usort($data_commodity, function ($a, $b) {
        return strcmp($b['date'], $a['date']);
    });
    if ($debug) echo "<pre>usort_data_commodity\n".print_r($data_commodity, true)."</pre>";

    if($all) {
        foreach ($data_commodity as $record_commodity) {
            if ($debug) echo "<pre>record_commodity\n" . print_r($record_commodity, true) . "</pre>";
            saveCommodityPrice($record_commodity, $meta_commodity, $debug); // Reuse the existing DB insert logic
        }
    }
    else {
        $latest_commodity = $data_commodity[0] ?? null;

        if (!$latest_commodity) return false;
        if ($debug) echo "<pre>latest_commodity\n" . print_r($latest_commodity, true) . "</pre>";
        // Step 4: Save to DB
        $saveCommodityPrice = saveCommodityPrice($latest_commodity, $meta_commodity, $debug);
        if ($debug) echo "<pre>saveCommodityPrice\n" . print_r($saveCommodityPrice, true) . "</pre>";
    }
    return $saveCommodityPrice;
}


function extract_stock_prices(string $html, $meta = [], $debug = false): array
{
    $code = $meta['code'];
    $currency = $meta['currency'];
    $friendly_name = $meta['friendly_name'];
    $type = $meta['type'];

    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html);
    $xpath = new DOMXPath($doc);

    $rows = $xpath->query('//table//tbody//tr');
    $data = [];

    foreach ($rows as $tr) {
        $cells = $tr->getElementsByTagName('td');
        if ($cells->length < 6) continue; // Expecting 6 columns (Date, Price, Open, High, Low, % Change)

        $dateRaw = trim($cells->item(0)->nodeValue);
        $date = DateTime::createFromFormat('M d, Y', $dateRaw);
        if (!$date) continue;

        $volume = $cells->item(5)->nodeValue;
        $volume_arr = extractNumberAndUnit($volume);

        $data[] = [
            'date' => $date->format('Y-m-d'),
            'price' => parse_price($cells->item(1)->nodeValue),
            'open' => parse_price($cells->item(2)->nodeValue),
            'high' => parse_price($cells->item(3)->nodeValue),
            'low' => parse_price($cells->item(4)->nodeValue),
            'volume' => parse_price($volume_arr[0]),
            'volume_unit' => trim($volume_arr[1]),
            'change_percent' => parse_price($cells->item(6)->nodeValue),
            'code' => trim($code),
            'currency' => $currency,
            'friendly_name' => $friendly_name,
            'type' => $type,
        ];
    }

    return $data;
}