<?php
class ProcessLock {

    private string $lockFilePath = '';
    private string $lockDirPath;
    private int $retries;
    private int $retryUsleep;
    private string $script;

    /**
     * Constructor for the ProcessLock class.
     *
     * @param string $script Name of the script for which the lock is being created.
     * @param string $lockDirPath Directory path where lock files will be stored.
     * @param int $retries Number of retry attempts for acquiring the lock.
     * @param int $retryUsleep Microseconds to wait between retries.
     */
    public function __construct(
        string $script = '',
        string $lockDirPath = "T:\\phptmp_vitex\\",
        int $retries = 13,
        int $retryUsleep = 89999
    ) {
        $this->script = $script;
        $this->lockDirPath = $lockDirPath;
        $this->retries = $retries;
        $this->retryUsleep = $retryUsleep;
    }

    /**
     * @return int
     */
    public function getRetries(): int {
        return $this->retries;
    }

    /**
     * @param int $retries
     */
    public function setRetries(int $retries): void {
        $this->retries = $retries;
    }

    /**
     * @return int
     */
    public function getRetryUsleep(): int {
        return $this->retryUsleep;
    }

    /**
     * @param int $retryUsleep
     */
    public function setRetryUsleep(int $retryUsleep): void {
        $this->retryUsleep = $retryUsleep;
    }

    /**
     * Check if a lock file exists for the given script.
     *
     * @return bool
     */
    public function lockExists(): bool {
        $this->lockFilePath = $this->lockDirPath . $this->script . ".lock";
        return file_exists($this->lockFilePath);
    }

    /**
     * Attempt to acquire the lock for the script.
     *
     * @return bool
     */
    public function acquireLock(): bool {
        $this->lockFilePath = $this->lockDirPath . $this->script . ".lock";
        $retries = 0;

        do {
            if (file_exists($this->lockFilePath)) {
                usleep($this->retryUsleep);
            } else {
                $pid = getmypid();
                @file_put_contents($this->lockFilePath, $pid);
                return file_exists($this->lockFilePath) && ((int) file_get_contents($this->lockFilePath) === $pid);
            }
        } while (++$retries < $this->retries);

        return false;
    }

    /**
     * Release the lock for the script.
     *
     * @return bool
     */
    public function releaseLock(): bool {
        try {
            if (file_exists($this->lockFilePath)) {
                return @unlink($this->lockFilePath);
            }
        } catch (Throwable $e) {
            return false;
        }
        return true;
    }

    /**
     * Release all lock files older than the specified age (in seconds).
     *
     * @param int $olderThan Age of lock files to delete (in seconds).
     * @return bool
     */
    public function releaseOldLocks(int $olderThan = 1): bool {
        $limit = time() - $olderThan;

        if (!is_dir($this->lockDirPath)) {
            return true;
        }

        $files = scandir($this->lockDirPath);
        foreach ($files as $file) {
            if (strpos($file, ".lock") !== false) {
                $filePath = $this->lockDirPath . $file;
                if (is_file($filePath) && filemtime($filePath) < $limit) {
                    @unlink($filePath);
                }
            }
        }

        return true;
    }

    /**
     * Compatibility method for `exists`. Calls `lockExists`.
     *
     * @param string $script
     * @return bool
     */
    public function exists(string $script = ''): bool {
        $this->script = $script;
        return $this->lockExists();
    }

    /**
     * Compatibility method for `aquire`. Calls `acquireLock`.
     *
     * @param string $script
     * @return bool
     */
    public function aquire(string $script): bool {
        $this->script = $script;
        return $this->acquireLock();
    }

    /**
     * Compatibility method for `release`. Calls `releaseLock`.
     *
     * @return bool
     */
    public function release(): bool {
        return $this->releaseLock();
    }

    /**
     * Compatibility method for `release_all`. Calls `releaseOldLocks`.
     *
     * @param int $olderThan
     * @return bool
     */
    public function release_all(int $olderThan = 1): bool {
        return $this->releaseOldLocks($olderThan);
    }

    /**
     * Compatibility method for `release_old`. Calls `releaseOldLocks`.
     *
     * @param int $olderThan
     * @return bool
     */
    public function release_old(int $olderThan = 1): bool {
        return $this->releaseOldLocks($olderThan);
    }
}
