<?php

/**
 * Fetches banco cuenta mov link based on the provided IDs.
 *
 * @param string $bancoCuentaMovLinkId The ID of the banco cuenta mov link.
 * @param string $origenBancoCuentaMovId The ID of the origen banco cuenta mov.
 * @param string $destinoBancoCuentaMovId The ID of the destino banco cuenta mov.
 *
 * @return bool|array Returns an array containing the fetched banco cuenta mov link data, or an empty array if no IDs are provided.
 */
function fetchBancoCuentaMovLinkBasedOnIds(string $bancoCuentaMovLinkId = "", string $origenBancoCuentaMovId = "", string $destinoBancoCuentaMovId = "", $debug = false): bool|array
{
    $method = __METHOD__;
    if(empty($bancoCuentaMovLinkId) && empty($origenBancoCuentaMovId) && empty($destinoBancoCuentaMovId))
        return [];
    if(!empty($bancoCuentaMovLinkId))
        $where = "banco_cuenta_mov_link_id = '$bancoCuentaMovLinkId'";
    elseif(!empty($origenBancoCuentaMovId))
        $where = "link = '$origenBancoCuentaMovId'";
    else
        $where = "banco_cuenta_mov_id = '$destinoBancoCuentaMovId'";
    $bancoCuentaMovLinkSql = "SELECT /** $method **/ * FROM banco_cuenta_mov_link WHERE $where";
    if($debug)
        debugAndEcho($bancoCuentaMovLinkSql, false);
    return ia_singleton($bancoCuentaMovLinkSql);
}

/**
 * Retrieves the bank account transactions based on the specified ID(s).
 *
 * @param string $bancoCuentaTransId The ID of the bank account transaction. (optional)
 * @param string $bancoCuentaMovLinkId The ID of the related bank account movement link. (optional)
 *
 * @return bool|array Returns an array of bank account transactions if successful, otherwise returns an empty array.
 */
function fetchBancoCuentaTrans(string $bancoCuentaTransId = '', string $bancoCuentaMovLinkId = '', $debug = false): bool|array{
    $method = __METHOD__;

    if ($debug)
        debugAndEcho(['$bancoCuentaTransId'=> $bancoCuentaTransId,'$bancoCuentaMovLinkId' => $bancoCuentaMovLinkId, '$method' => $method], false, $method);

    $where = "";
    if(empty($bancoCuentaMovLinkId) && empty($bancoCuentaTransId))
        return [];
    if(!empty($bancoCuentaMovLinkId))
        $where = "banco_cuenta_mov_link_id = '$bancoCuentaMovLinkId'";
    elseif(!empty($bancoCuentaTransId))
        $where = "banco_cuenta_trans_id = '$bancoCuentaTransId'";
    $bancoCuentaTransSql = "SELECT /** $method **/ * FROM banco_cuenta_trans WHERE $where";

    if ($debug)
        debugAndEcho(['$bancoCuentaTransId'=> $bancoCuentaTransId,'$bancoCuentaMovLinkId' => $bancoCuentaMovLinkId,'$bancoCuentaTransSql' => $bancoCuentaTransSql,'$method' => $method], false, $method);

    return ia_singleton($bancoCuentaTransSql);
}

/**
 * Gets the user remarks from the original remarks.
 *
 * @param string $remarksOri The original remarks (optional).
 * @return string The user remarks extracted from the original remarks.
 */
function getUserRemarks(string $remarksOri = ''): string{
    $remarksUsuario = '';
    if(empty($remarksOri))
        return $remarksUsuario;
    $remarksOri = str_replace(["\n", "\r", "TRASPASO ENTRE CUENTAS PROPIAS", "PAGO DE ASIMILADOS"], '', $remarksOri);
    $pattern = '/<br>(.*?)<br data-remarks/';
    if (preg_match($pattern, $remarksOri, $matches)) {
        $remarksUsuario = $matches[1];
    }
    return $remarksUsuario;
}

/**
 * Retrieves the remarks made by a different owner of the user.
 *
 * @param string $remarksOri The original remarks made by the different owner.
 *
 * @return string The remarks made by the different owner. If no remarks are found, an empty string is returned.
 */
function getUserRemarksDifferentOwner(string $remarksOri = ''): string{
    $remarksUsuario = '';
    if(empty($remarksOri))
        return $remarksUsuario;
    $pattern = '/<strong[^>]*\btxt_bold_red\b[^>]*>(.*?)<\/strong>/';
    if (preg_match($pattern, $remarksOri, $matches)) {
        $remarksUsuario = $matches[1];
    }
    return $remarksUsuario;
}

/**
 * Fetches the details of a banco cuenta mov based on the bancoCuentaMovId
 *
 * @param string $bancoCuentaMovId The ID of the banco cuenta mov
 * @return bool|array Returns an empty array if the bancoCuentaMovId is empty or an array containing the banco cuenta mov details
 */
function fetchBancoCuentaMov(string $bancoCuentaMovId = ''): bool|array{
    $method = __METHOD__;
    $debug = false;
    if(empty($bancoCuentaMovId))
        return [];
    $bcmSql = "SELECT bcm.*, 
       bc.nombre as banco, bc.tipo_inversion_empresa_privada, 
       e.empresa as empresa,
       e.empresa_id as bcm_empresa_id
    FROM banco_cuenta_mov bcm 
        LEFT OUTER JOIN banco_cuenta bc ON bcm.banco_cuenta_id = bc.banco_cuenta_id 
        LEFT OUTER JOIN empresa e ON bc.empresa_id = e.empresa_id 
    WHERE banco_cuenta_mov_id = '{$bancoCuentaMovId}'";
    if($debug)
        debugAndEcho($bcmSql, false);
    return ia_singleton($bcmSql);
}

/**
 * Fetches a transaction if it is empty.
 *
 * @param string $transactionId The transaction ID UUID 32 characters.
 * @param array $transaction The transaction data.
 * @param string $originAccountMovId The origin account movement ID.
 * @param string $destinationAccountMovId The destination account movement ID.
 *
 * @return bool|array|null The fetched transaction data or null if not found.
 */
function fetchTransactionIfEmpty(string $transactionId = "", array $transaction = [], string $originAccountMovId = "", string $destinationAccountMovId = "", $debug = false): bool|array|null
{
    $method = __METHOD__;
    if (!empty($transaction)) return $transaction;
    if ($debug)
        debugAndEcho(['$transactionId' => $transactionId, '$transaction'=> $transaction,'$originAccountMovId' => $originAccountMovId, '$destinationAccountMovId' => $destinationAccountMovId,'$method' => $method], false, "fetchTransactionIfEmpty: get args");
    $transaction = fetchBancoCuentaTrans(bancoCuentaTransId: $transactionId, debug: $debug);

    if($debug !== false) {
        debugAndEcho(['after fetchBancoCuentaTrans' => $transaction], false, "fetchTransactionIfEmpty _ fetchBancoCuentaTrans");
    }

    if (empty($transaction)) {
        $accountMovLink = fetchBancoCuentaMovLinkBasedOnIds(origenBancoCuentaMovId: $originAccountMovId, destinoBancoCuentaMovId: $destinationAccountMovId, debug: $debug);
        if(!empty($accountMovLink)){
            $transaction = fetchBancoCuentaTrans(bancoCuentaMovLinkId: $accountMovLink['banco_cuenta_mov_link_id']);
        }
    }
    if($debug !== false) {
        debugAndEcho(['after fetchBancoCuentaMovLinkBasedOnIds' => $transaction], false, "after fetchBancoCuentaMovLinkBasedOnIds");
    }
    return $transaction;
}

/**
 * Fetches the account movement link if it is empty.
 *
 * @param array $transaction The transaction details.
 * @param array $accountMovLink The account movement link.
 *
 * @return array|bool The fetched account movement link.
 */
function fetchAccountMovLinkIfEmpty(array $transaction = [], array $accountMovLink = []): array|bool
{
    if(!empty($accountMovLink)) return $accountMovLink;
    return fetchBancoCuentaMovLinkBasedOnIds(bancoCuentaMovLinkId: $transaction['banco_cuenta_mov_link_id']);
}

/**
 * Returns the currency based on the provided currency ID.
 *
 * @param int $monedaId The currency ID to retrieve the currency for.
 *
 * @return string The currency name. Returns "PESOS" if the currency ID is 1, otherwise returns "USD".
 */
function getMoneda(int $monedaId = 1): string {
    return $monedaId == 1 ? "PESOS" : "USD";
}

/**
 * Returns the CSS class based on the given currency ID.
 *
 * @param int $monedaId The currency ID.
 *
 * @return string The CSS class: "txt_shadow_azul_marino" if the currency ID is 1,
 *                "txt_shadow_verde" otherwise.
 */
function getClass(int $monedaId = 1): string {
    return $monedaId == 1 ? "txt_shadow_azul_marino" : "txt_shadow_verde";
}

/**
 * Get the account link for a bank account transaction.
 *
 * @param array $bancoCuentaTrans The bank account transaction.
 * @param array $bancoCuentaMov The bank account movement.
 * @param string $classMoneda The currency class.
 * @return string The account link HTML.
 */
function getCuentaLink(array $bancoCuentaTrans = [], array $bancoCuentaMov = [], string $classMoneda = "txt_shadow_azul_marino"): string {
    return "<a title='{$bancoCuentaMov['banco']}' href='../cobranza/edocta.php?ACC=CB&ECO=SI&banco_cuenta_mov_id={$bancoCuentaMov['banco_cuenta_mov_id']}' target='_blank' class='{$classMoneda}'>";
}

/**
 * Generates the remarks for a transaction with different owners.
 *
 * @param array $bancoCuentaTrans The banco cuenta transaction data.
 * @param string $linkToCuentaOrigen The link to the cuenta origen.
 * @param array $origenBancoCuentaMov The origen banco cuenta movement data.
 * @param string $linkToCuentaDestino The link to the cuenta destino.
 * @param array $destinoBancoCuentaMov The destino banco cuenta movement data.
 * @param string $remarksUsuario The remarks provided by the user.
 * @return string The generated remarks.
 */
function generateRemarksDifferentOwner(array $bancoCuentaTrans, string $linkToCuentaOrigen, array $origenBancoCuentaMov, string $linkToCuentaDestino, array $destinoBancoCuentaMov, string $remarksUsuario, string $usuario_automatico, array $stylesArr): string
{
    $method = __METHOD__;

    $txtFactura = $destinoBancoCuentaMov['tipo_inversion_empresa_privada'] === "privada" ? "SEMANA" : "FACTURA";
    $classTransfer = "txt_shadow_purpura bold";

    $linkToTransfer = "<a title='Ir a Links entre Cuentas Bancarias' href='../backoffice/banco_a_banco_list.php?banco_cuenta_trans_id={$bancoCuentaTrans['banco_cuenta_trans_id']}' target='_blank' class='{$classTransfer}'>";
    $referenciaOriginal="";
    if(!empty($usuario_automatico)) {
        $tituloFinal = "";
        $beneficiario = getBeneficiarioFromBanco($destinoBancoCuentaMov, ";", true);
        $usuario_automatico = PHP_EOL . $beneficiario . PHP_EOL . $usuario_automatico;
        if($destinoBancoCuentaMov['tipo_inversion_empresa_privada'] === "privada") {
            $numero_original = $destinoBancoCuentaMov['numero_original'];
            $remarksUsuario .= extractArrenText($numero_original, $origenBancoCuentaMov['banco'], $referenciaOriginal);
        }
        else{
            $remarksUsuario .= "PAGO PROVEEDOR NACIONAL ";
        }
    }
    $identifierRemarksUsuario = empty($remarksUsuario) && empty($usuario_automatico) ? "<br>" : '<br data-remarks="mitternacht">';
    return "<strong class='txt18pxfr txt_bold_red txt_centered' style='min-width: 335px;'><span class='remarks_por_usuario_'>" . strip_tags($remarksUsuario) . "</strong></span>$usuario_automatico{$identifierRemarksUsuario}{$linkToTransfer}PAGO</a> DE $linkToCuentaOrigen<strong>{$origenBancoCuentaMov['banco']}</strong></a> A $linkToCuentaDestino<strong>{$destinoBancoCuentaMov['banco']}</strong></a>\r\n<br />$txtFactura {$bancoCuentaTrans['factura_numero']} $referenciaOriginal";
}
function getMonth($text, $defaultDate = '2024-10-30'): string
{
    $monthMapping = [
        '/\bENERO\b|\bEN\b|\bER\b/' => 'ENERO',
        '/\bFEBRERO\b|\bFEB\b|\bFE\b/' => 'FEBRERO',
        '/\bMARZO\b|\bMAR\b|\bMA\b|\bMZO\b/' => 'MARZO',
        '/\bABRIL\b|\bABRI\b|\bAB\b/' => 'ABRIL',
        '/\bMAYO\b|\bMAY\b|\bMY\b/' => 'MAYO',
        '/\bJUNIO\b|\bJUN\b|\bJN\b/' => 'JUNIO',
        '/\bJULIO\b|\bJUL\b|\bJL\b/' => 'JULIO',
        '/\bAGOSTO\b|\bAGO\b|\bAG\b/' => 'AGOSTO',
        '/\bSEPTIEMBRE\b|\bSEP\b|\bSE\b|\bSET\b/' => 'SEPTIEMBRE',
        '/\bOCTUBRE\b|\bOCT\b|\bOC\b/' => 'OCTUBRE',
        '/\bNOVIEMBRE\b|\bNOV\b|\bNO\b/' => 'NOVIEMBRE',
        '/\bDICIEMBRE\b|\bDIC\b|\bDI\b/' => 'DICIEMBRE',
    ];

    foreach ($monthMapping as $pattern => $month) {
        if (preg_match($pattern, $text)) {
            return $month;
        }
    }

    $defaultMonthNumber = date('m', strtotime($defaultDate));
    $monthsInSpanish = [
        '01' => 'ENERO',
        '02' => 'FEBRERO',
        '03' => 'MARZO',
        '04' => 'ABRIL',
        '05' => 'MAYO',
        '06' => 'JUNIO',
        '07' => 'JULIO',
        '08' => 'AGOSTO',
        '09' => 'SEPTIEMBRE',
        '10' => 'OCTUBRE',
        '11' => 'NOVIEMBRE',
        '12' => 'DICIEMBRE',
    ];

    return $monthsInSpanish[$defaultMonthNumber];
}

function extractArrenText($text, $origenBanco = "", &$referenciaOriginal = ""): string
{
    $year = '2024';
    $month = getMonth($text);

    // Step 1: Capture "HONOR" or "ASIM" with everything that follows, storing in $referenciaOriginal
    if (preg_match('/\b(HONOR|ASIM)\b(.*)/', $text, $matches)) {
        $referenciaOriginal = trim($matches[0]);
        return "HONORARIOS ASIMILADOS A SALARIOS CORRESPONDIENTES AL MES DE $month $year";
    }

    // Step 2: Find "ARREN", "ARREND", or "ARRE" variations
    if (preg_match('/\b(ARREN|ARREND|ARRE)\b(.*)/', $text, $matches)) {
        $referenciaOriginal = trim($matches[0]);

        // Step 3: Look for "PUE", "PUEB", etc. to identify Puebla and Bodega/Tienda
        if (preg_match('/\bPUE\b|\bPUEB\b|\bPUEBLA\b/', $text)) {
            if (preg_match('/\bBOD\b|\bBODEGA\b/', $text)) {
                return "ARRENDAMIENTO $month $year BODEGA PUEBLA";
            } elseif (preg_match('/\bT\b|\bTIENDA\b/', $text)) {
                return "ARRENDAMIENTO $month $year TIENDA PUEBLA";
            }
        }

        // Step 4: Look for "CLAV", "CLAVEL", etc. to identify Clavel 224 and Bodega 1/2
        if (preg_match('/\bCLA\b|\bCLAV\b|\bCLAVEL\b/', $text)) {
            if (preg_match('/\bB1\b|\bBODEGA 1\b/', $text)) {
                return "ARRENDAMIENTO $month $year CLAVEL 224 BODEGA 1";
            } elseif (preg_match('/\bB2\b|\bBODEGA 2\b/', $text)) {
                return "ARRENDAMIENTO $month $year CLAVEL 224 BODEGA 2";
            } else {
                return "ARRENDAMIENTO $month $year CLAVEL 224";
            }
        }

        // Step 5: Handle Guatemala variations, including "GT"
        if (preg_match('/\bGUAT\b|\bGUATEMALA\b|\bGT\b/', $text)) {
            if (preg_match('/\b127\b/', $text)) {
                return "ARRENDAMIENTO $month $year GUATEMALA 127";
            } elseif (preg_match('/\b126\b/', $text)) {
                return "ARRENDAMIENTO $month $year GUATEMALA 126 LOCAL A";
            } else {
                return "ARRENDAMIENTO $month $year GUATEMALA";
            }
        }

        // Step 6: Check if $origenBanco contains "STEMAX"
        if (stripos($origenBanco, 'STEMAX') !== false) {
            $referenciaOriginal = trim($matches[0]);
            return "ARRENDAMIENTO $month $year BODEGA PUEBLA";
        }
    }

    // Default case if no specific pattern is matched
    $referenciaOriginal = $text;
    return "ARRENDAMIENTO $month $year $text";
}

/**
 * Generates the remarks for a transaction between accounts owned by the same user.
 *
 * @param array $bancoCuentaTrans The details of the transaction.
 * @param string $linkToCuentaOrigen The link to the source account.
 * @param string $linkToCuentaDestino The link to the destination account.
 * @param string $ctaBancariaOrigen The details of the source account.
 * @param string $ctaBancariaDestino The details of the destination account.
 * @param string $remarksUsuario The remarks provided by the user.
 *
 * @return string The generated remarks for the transaction.
 */
function generateRemarksSameOwner(array $bancoCuentaTrans, string $linkToCuentaOrigen, string $linkToCuentaDestino, string $ctaBancariaOrigen, string $ctaBancariaDestino, string $remarksUsuario, string $usuario_automatico, $origenBancoCuentaMov, $destinoBancoCuentaMov, $stylesArr, $debug = false): string
{
    $method = __METHOD__;

    if($debug) {
        debugAndEcho(echo_get_args(func_get_args(), $method), false, "generateRemarksSameOwner");
    }

    $remarksUsuario = str_replace(["TRASPASO ENTRE CUENTAS PROPIAS"], '', $remarksUsuario);

    $identifierRemarksUsuario = empty($remarksUsuario) && empty($usuario_automatico) ? "<br>" : '<br data-remarks="mitternacht">';
    $classTransfer = "txt_shadow_purpura bold";

    // SI EN $remarksUsuario CONTIENE LA PALABRA DIVISA Ó DIVISAS
    //      AGREGAR AL TITULO 'TRASPASO ENTRE CUENTAS PROPIAS' UN PREFIX 'COMPRA DE DOLARES';
    //      y $remarksUsuario se queda vacio
    $prefix_traspaso_entre_cuentas = '';
    $pattern = "/(divisa|divisas)/i";
    if (preg_match($pattern, $remarksUsuario) ||
        preg_match($pattern, ($destinoBancoCuentaMov['numero_original'] ?? "")) ||
        ($stylesArr['empidOrigen'] == $stylesArr['empidDestino'] && $stylesArr['monedaOrigen'] != $stylesArr['monedaDestino'])) {
        $remarksUsuario = '';
        $prefix_traspaso_entre_cuentas = "COMPRA DE DOLARES" . PHP_EOL;
    }
    else
        $prefix_traspaso_entre_cuentas = "TRASPASO ENTRE CUENTAS PROPIAS" . PHP_EOL;

    $beneficiario = getBeneficiarioFromBanco($destinoBancoCuentaMov, ";", false);
    $usuario_automatico = $beneficiario . PHP_EOL . $usuario_automatico;

    $linkToTransfer = "<a title='Ir a Links entre Cuentas Bancarias' href='../backoffice/banco_a_banco_list.php?banco_cuenta_trans_id={$bancoCuentaTrans['banco_cuenta_trans_id']}' target='_blank' class='{$classTransfer}'>";

    return "<strong class='txt18pxfr txt_bold_red txt_centered' style='min-width: 335px;'>{$prefix_traspaso_entre_cuentas}</strong><br><span class='remarks_por_usuario_'>". strip_tags($remarksUsuario) ."$usuario_automatico</span>{$identifierRemarksUsuario}{$linkToTransfer}TRANSFERENCIA</a> DE {$linkToCuentaOrigen}{$ctaBancariaOrigen} </a> A {$linkToCuentaDestino}{$ctaBancariaDestino}</a>";
}

/**
 * Generates a link to the 'movimiento_divisa_list.php' page with a specified movimiento_divisa_id.
 *
 * @param array $bancoCuentaTrans The bancoCuentaTrans details [].
 * @param string $movimientoDivisaId The ID of the movimiento_divisa.
 * @param string $classTipoCambio The CSS class for the link. Defaults to "txt_shadow_orange".
 * @return string The generated link HTML.
 */
function getMovimientoDivisaLink(array $bancoCuentaTrans, string $movimientoDivisaId, string $classTipoCambio = "txt_shadow_orange"): string {

    return "<a title='Ir Movimiento con Tipo de cambio' href='../backoffice/movimiento_divisa_list.php?movimiento_divisa_id={$movimientoDivisaId}' target='_blank' class='{$classTipoCambio}'>";
}

/**
 * Fetches the movimiento_divisa_id based on the provided docId and docOrigen.
 *
 * @param string $docId The document ID.
 * @param string $docOrigen The document origen.
 * @return string The movimiento_divisa_id, or an empty string if the docId is empty.
 */
function fetchMovimientoDivisaId(string $docId = '', string $docOrigen = 'banco_cuenta_trans'): string
{
    if(empty($docId)){
        return "";
    }
    $method = __METHOD__;
    $movimientoDivisaIdSql = "SELECT /** $method **/ movimiento_divisa_id FROM movimiento_divisa WHERE doc_id = '{$docId}' AND doc_origen = '{$docOrigen}'";
    return ia_singleread($movimientoDivisaIdSql);
}

/**
 * Generate remarks for a different currency transfer
 *
 * @param array $bancoCuentaTrans The banco cuenta trans array
 * @param array $stylesArr The styles array
 * @param string $remarksTransfer The existing remarks transfer string
 * @return string The updated remarks transfer string
 */
function generateRemarksDifferentCurrency(array $bancoCuentaTrans = [], array $stylesArr = [], string $remarksTransfer = "", string $usuario_automatico = ""): string
{
    $movimientoDivisaId = fetchMovimientoDivisaId($bancoCuentaTrans['banco_cuenta_trans_id'], 'banco_cuenta_trans');
    $linkToMovimientoDivisa = getMovimientoDivisaLink($bancoCuentaTrans, $movimientoDivisaId, $stylesArr['classTipoCambio']);

    $usuario_automatico = "";

    $remarksTransfer .= "<br/><li>MONTO ORIGEN: {$stylesArr['linkToCuentaOrigen']}<span class='bold {$stylesArr['classMonedaOrigen']}'>$ " . echonf($bancoCuentaTrans['monto_origen'], true) . " {$stylesArr['monedaOrigen']}</span></li></a><li>MONTO DESTINO: {$stylesArr['linkToCuentaDestino']}<span class='bold {$stylesArr['classMonedaDestino']}'>$ " . echonf($bancoCuentaTrans['monto'], true) . " {$stylesArr['monedaDestino']}</span></li></a><li>TIPO DE CAMBIO: {$linkToMovimientoDivisa}<strong>" . echonf($bancoCuentaTrans['tipo_cambio'], true, 6)."</strong></a></li>{$usuario_automatico}";

    return $remarksTransfer;
}

/**
 * Generates remarks with the same currency.
 *
 * @param array $bancoCuentaTrans The BancoCuentaTrans array.
 * @param array $stylesArr The styles array.
 * @param string $remarksTransfer The transfer remarks.
 * @return string The generated remarks with the same currency.
 */
function generateRemarksSameCurrency(array $bancoCuentaTrans = [], array $stylesArr = [], string $remarksTransfer = ""): string{
    return "{$remarksTransfer}\r\n<br /><li>MONTO: {$stylesArr['linkToCuentaDestino']}<span class='bold {$stylesArr['classMonedaDestino']}'>$ " . echonf($bancoCuentaTrans['monto'], true) . " {$stylesArr['monedaDestino']}</span></li></a>";
}

/**
 * Generates the remarks for a transfer between own accounts.
 *
 * @param string $bancoCuentaTransId The ID of the bank transaction.
 * @param string $origenBancoCuentaMovId The ID of the source bank account movement.
 * @param string $destinoBancoCuentaMovId The ID of the destination bank account movement.
 * @param array $bancoCuentaTrans The details of the bank transaction.
 * @param array $bancoCuentaMovLink The details of the bank account movement link.
 * @param string $remarksUsuario The remarks provided by the user (optional).
 *
 * @return string The generated remarks for the transfer.
 *
 * $stylesArr keys:
 * - ctaBancariaOrigen: The name of the originating bank.
 * - ctaBancariaDestino: The name of the destination bank.
 * - monedaOrigen: The currency of the originating bank. It can be "PESOS" or "USD".
 * - monedaDestino: The currency of the destination bank. It can be "PESOS" or "USD".
 * - classMonedaOrigen: The CSS class for the currency of the originating bank.
 * - classMonedaDestino: The CSS class for the currency of the destination bank.
 * - linkToCuentaOrigen: The HTML link to the originating bank's account.
 * - linkToCuentaDestino: The HTML link to the destination bank's account.
 * - empidDestino: The ID of the destination bank associated with the company.
 * - empidOrigen: The ID of the originating bank associated with the company.
 * - empresaDestino: The name of the destination bank associated with the company.
 * - classTipoCambio: The CSS class for the currency exchange link. Default value is "txt_shadow_orange".
 */
function generaRemarksTransferCuentasPropias(
    string $bancoCuentaTransId = '', string $origenBancoCuentaMovId = '', string $destinoBancoCuentaMovId = '', array $bancoCuentaTrans = array(), array $bancoCuentaMovLink = array(), string $remarksUsuario = "", string $usuario_automatico = "", bool $debug = false
): string
{
    try {
        $method = __METHOD__;

        if ($debug)
            debugAndEcho(['$bancoCuentaTrans' => $bancoCuentaTrans, '$bancoCuentaTransId'=> $bancoCuentaTransId,'$usuario_automatico' => $usuario_automatico,'$method' => $method], false, "generaRemarksTransferCuentasPropias");

        $remarksTransfer = '';
        $bancoCuentaTrans = fetchTransactionIfEmpty($bancoCuentaTransId, $bancoCuentaTrans, $origenBancoCuentaMovId, $destinoBancoCuentaMovId, $debug);

        $usuario_automatico = !empty($usuario_automatico) || $bancoCuentaTrans['alta_por'] == 'sistema' ? "<strong>LINK AUTOMATICO</strong><br>" . PHP_EOL : "";

        if ($debug)
            debugAndEcho($bancoCuentaTrans, false, "bancoCuentaTrans");

        if (empty($bancoCuentaTrans)) return $remarksTransfer;
        $bancoCuentaMovLink = fetchAccountMovLinkIfEmpty($bancoCuentaTrans, $bancoCuentaMovLink);

        if ($debug)
            debugAndEcho($bancoCuentaMovLink, false);

        $stylesArr = [];

        $origenBancoCuentaMov = fetchBancoCuentaMov($bancoCuentaMovLink['link']);
        $destinoBancoCuentaMov = fetchBancoCuentaMov($bancoCuentaMovLink['banco_cuenta_mov_id']);

        if ($debug) {
            debugAndEcho($origenBancoCuentaMov, false, "origenBancoCuentaMov");
            debugAndEcho($destinoBancoCuentaMov, false, "destinoBancoCuentaMov");
        }

        if (empty($remarksUsuario)) {
            if ($origenBancoCuentaMov['empresa_id'] != $destinoBancoCuentaMov['empresa_id'])
                $remarksUsuario = getUserRemarksDifferentOwner($bancoCuentaTrans['referencia']). PHP_EOL;
            else
                $remarksUsuario = getUserRemarks($bancoCuentaTrans['referencia']). PHP_EOL;
        }
        else{
            $remarksUsuario = str_replace(["\n", "\r", "TRASPASO ENTRE CUENTAS PROPIAS"], '', $remarksUsuario);
        }

        if ($debug)
            debugAndEcho($remarksUsuario, false, "remarksUsuario");

        $stylesArr['ctaBancariaOrigen'] = $origenBancoCuentaMov['banco'];
        $stylesArr['ctaBancariaDestino'] = $destinoBancoCuentaMov['banco'];
        $stylesArr['monedaOrigen'] = getMoneda($origenBancoCuentaMov['moneda_id']);
        $stylesArr['monedaDestino'] = getMoneda($destinoBancoCuentaMov['moneda_id']);
        $stylesArr['monedaOrigenID'] = $origenBancoCuentaMov['moneda_id'];
        $stylesArr['monedaDestinoID'] = $destinoBancoCuentaMov['moneda_id'];
        $stylesArr['classMonedaOrigen'] = getClass($origenBancoCuentaMov['moneda_id']);
        $stylesArr['classMonedaDestino'] = getClass($destinoBancoCuentaMov['moneda_id']);
        $stylesArr['linkToCuentaOrigen'] = getCuentaLink($bancoCuentaTrans, $origenBancoCuentaMov, $stylesArr['classMonedaOrigen']);
        $stylesArr['linkToCuentaDestino'] = getCuentaLink($bancoCuentaTrans, $destinoBancoCuentaMov, $stylesArr['classMonedaDestino']);
        $stylesArr['empidDestino'] = $destinoBancoCuentaMov['bcm_empresa_id'];
        $stylesArr['empidOrigen'] = $origenBancoCuentaMov['bcm_empresa_id'];
        $stylesArr['empresaDestino'] = $destinoBancoCuentaMov['empresa'];
        $stylesArr['classTipoCambio'] = "txt_shadow_orange";

        if ($debug)
            debugAndEcho($stylesArr, false, "stylesArr");


        if ($stylesArr['empidOrigen']  != $stylesArr['empidDestino']) {
            $remarksTransfer = generateRemarksDifferentOwner($bancoCuentaTrans, $stylesArr['linkToCuentaOrigen'], $origenBancoCuentaMov, $stylesArr['linkToCuentaDestino'], $destinoBancoCuentaMov, $remarksUsuario, $usuario_automatico, $stylesArr);
        } else {
            $remarksTransfer = generateRemarksSameOwner($bancoCuentaTrans, $stylesArr['linkToCuentaOrigen'], $stylesArr['linkToCuentaDestino'], $stylesArr['ctaBancariaOrigen'], $stylesArr['ctaBancariaDestino'], $remarksUsuario, $usuario_automatico, $origenBancoCuentaMov, $destinoBancoCuentaMov, $stylesArr, $debug);
        }

        if ($debug)
            debugAndEcho($remarksTransfer, false, "remarksTransfer");

        if ($origenBancoCuentaMov['moneda_id'] != $destinoBancoCuentaMov['moneda_id']) {
            return generateRemarksDifferentCurrency($bancoCuentaTrans, $stylesArr, $remarksTransfer, $usuario_automatico);
        }
        return generateRemarksSameCurrency($bancoCuentaTrans, $stylesArr, $remarksTransfer);
    }
    catch (Exception $e) {
        ia_errores_a_dime("Error en " . __METHOD__ . " - " . $e->getMessage());
        return "";
    }
}

function debugAndEcho(array|string|object $debugArray, bool $die = false, string $nombreVariable = ""): void
{
    ia_errores_a_dime("$nombreVariable:<pre>". PHP_EOL . print_r(echo_get_args($debugArray, $nombreVariable), true) . "</pre>");
    sleep(1);
    if($die) die();
}

function echo_get_args($args, $method): string
{
    $args_txt = "<ol>$method:". PHP_EOL;
    if (!is_array($args)) {
        $args_ = $args;
        $args = array();
        $args[] = $args_;
    }
    foreach ($args as $k => $arg) {
        $args_txt .= "<li>$k:". PHP_EOL . print_r($arg ?? "", true);
    }

    $args_txt .= "</ol>";

    return $args_txt;
}

function genera_poliza_contabilidad_transfer_cuentas_propias($bancoCuentaTransId, $remarksSet = ''): string
{
    $polizaContabilidad = "";
    if (empty($bancoCuentaTransId) || empty($remarksSet)) {
        return $polizaContabilidad;
    }

    $transferencia = retrieveTransferencia($bancoCuentaTransId);
    if (empty($transferencia)) {
        return $polizaContabilidad;
    }

    $bancoCuentaDestino = retrieveBancoCuenta($transferencia['destino_banco_cuenta_id']);
    $bancoCuentaOrigen = retrieveBancoCuenta($transferencia['origen_banco_cuenta_id']);

    $bancoCuentaDestino['empresa_id'] = $bancoCuentaDestino['empresa_id'] ?? -1;
    $bancoCuentaOrigen['empresa_id'] = $bancoCuentaOrigen['empresa_id'] ?? -2;

    if (isCompraDeDivisas($remarksSet, $transferencia, $bancoCuentaDestino, $bancoCuentaOrigen)) {
        $polizaContabilidad = processCompraDeDivisas($remarksSet, $bancoCuentaDestino, $bancoCuentaOrigen);
    } elseif (str_contains($remarksSet, 'TRASPASO')) {
        $polizaContabilidad = processTraspasoEntreCuentasPropias($remarksSet, $bancoCuentaDestino, $bancoCuentaOrigen);
    } elseif (str_contains($remarksSet, 'HONORA') || str_contains($remarksSet, 'ARRENDA')) {
        $isArrendamiento = str_contains($remarksSet, 'ARRENDA');
        $polizaContabilidad = processHonorariosOrArrendamiento($remarksSet, $bancoCuentaDestino, $isArrendamiento);
    }
    return trim($polizaContabilidad);
}

function retrieveTransferencia($bancoCuentaTransId): bool|array
{
    return ia_singleton("SELECT * FROM banco_cuenta_trans WHERE banco_cuenta_trans_id = " . strit($bancoCuentaTransId));
}

function retrieveBancoCuenta($bancoCuentaId): bool|array
{
    $sql = "SELECT bc.*, b.* FROM banco_cuenta bc 
            JOIN banco b ON b.banco_id = bc.banco_id WHERE bc.banco_cuenta_id = $bancoCuentaId";
    return ia_singleton($sql);
}

function isCompraDeDivisas($remarksSet, $transferencia, $bancoCuentaDestino, $bancoCuentaOrigen): bool
{
    return str_contains($remarksSet, 'COMPRA') || $transferencia['origen_moneda_id'] != $transferencia['moneda_id'];
}
function fetchEmpresaFromBancoCuenta($banco_cuenta_id): bool|array
{
    return ia_singleton("SELECT * FROM empresa LEFT JOIN banco_cuenta ON empresa.empresa_id = banco_cuenta.empresa_id WHERE banco_cuenta.banco_cuenta_id = " . strit($banco_cuenta_id));
}

function getBeneficiarioFromBanco($bancoCuenta = '', $delimiter = '****', $distintaEmpresa = false): string
{
    // Direct match for RONY
    if (str_contains($bancoCuenta['banco'] ?? '', 'RONY')) {
        return "RONY AKIKI" . $delimiter . PHP_EOL;
    }

    // If distintaEmpresa is true, fetch empresa and return its razon_social
    if ($distintaEmpresa) {
        $empresa = fetchEmpresaFromBancoCuenta($bancoCuenta['banco_cuenta_id']);
        return $empresa['razon_social'] . $delimiter . PHP_EOL;
    }

    // Mapping bank keywords to their full names
    $banks = [
        'BBVA' => "BBVA MÉXICO SA",
        'MONEX' => "MONEX GRUPO FINANCIERO SA DE CV",
        'HSBC' => "HSBC MÉXICO SA",
        'BANORTE' => "BANORTE SA INSTITUCIÓN DE BANCA MÚLTIPLE",
        'SANTANDER' => "BANCO SANTANDER (MÉXICO) SA INSTITUCIÓN DE BANCA MÚLTIPLE",
        'SCOTIABANK' => "SCOTIABANK INVERLAT SA",
        'BANAMEX' => "BANAMEX SA INSTITUCIÓN DE BANCA MÚLTIPLE GRUPO FINANCIERO BANAMEX",
        'CITI' => "BANAMEX SA INSTITUCIÓN DE BANCA MÚLTIPLE GRUPO FINANCIERO BANAMEX"
    ];

    // Default to bancoCuenta['banco'] if no match is found
    $beneficiario = $bancoCuenta['banco'] ?? '';

    // Loop through each bank keyword to find a match
    foreach ($banks as $keyword => $name) {
        if (str_contains($bancoCuenta['banco'] ?? '', $keyword)) {
            $beneficiario = $name;
            break;
        }
    }

    return $beneficiario . $delimiter . PHP_EOL;
}

function encuentraBeneficiarioEnRemarks($text, &$textSinBeneficiario): ?string
{
    // Split the text into lines
    $lines = explode("\n", $text);

    $textSinBeneficiario = "";
    $beneficiario = "";
    // Find the line with a semicolon
    foreach ($lines as $line) {
        if (str_contains($line, ';')) {
            // Remove the semicolon and trim whitespace
            $beneficiario = trim(str_replace(";", "", $line));
        }
        else {
            if(str_contains($line, 'LINK AUTOMATICO')) continue;
            $line = preg_replace('/[ ]{2,}/u', ' ', trim(strip_tags(($line))));
            $textSinBeneficiario .= !empty($line) ? $line . PHP_EOL : "";
        }
    }
    $textSinBeneficiario = trim($textSinBeneficiario);
//    echo "<pre>cleandLines" . print_r($textSinBeneficiario, true) . "</pre>";
//    echo "<pre>beneficiario" . print_r($beneficiario, true) . "</pre>";

    return trim($beneficiario);
}


function processText($remarksSet): array
{
    // Remove unwanted tags and parse HTML content
    $remarksCleaned = removeTags($remarksSet, ["TRASPASO ENTRE CUENTAS PROPIAS", "LINK AUTOMATICO", "MONTO:"]);
    $dom = new DOMDocument();
    @$dom->loadHTML(htmlspecialchars($remarksCleaned));

    // Extract plain text content
    $content = strip_tags($dom->textContent);
    return ['cleaned' => vx_trim($content), 'dom' => $dom];
}

function removeTags($text, $termsToRemove = []): string
{
    // Remove <strong> tags and specific terms
    $text = preg_replace('/<strong[^>]*> *TRASPASO ENTRE CUENTAS PROPIAS *<\/strong>/i', "\\r\\n", $text);
    return str_replace(["\n", "\r", ...$termsToRemove], '', $text);
}

function getBeneficiary($remarksSet, &$content): string
{
    $beneficiario = encuentraBeneficiarioEnRemarks($remarksSet, $content);
    return $beneficiario ? $beneficiario . '****<br>' : '';
}

function processTransaction($remarksSet, $bancoCuentaDestino, $bancoCuentaOrigen, $isArrendamiento = false, $isDivisas = false): string
{
    // Process text to get cleaned content
    ['cleaned' => $content, 'dom' => $dom] = processText($remarksSet);

    // Extract title if present
    preg_match('/<strong[^>]*>(.*?)<\/strong>/s', $remarksSet, $match);
    $title = !empty($match[1]) ? vx_trim(strip_tags($match[1]) . "<br>") : "";

    // Adjust title for specific cases
    if ($isArrendamiento) {
        $title = str_replace("ARRENDAMINETO", "ARRENDAMIENTO", $title);
        $content = str_replace("ARRENDAMINETO", "ARRENDAMIENTO", $content);
    }

    if ($isDivisas) {
        if (empty($title) && (str_contains($remarksSet, 'COMPRA') || ($bancoCuentaOrigen['origen_moneda_id'] ?? 1) != $bancoCuentaDestino['moneda_id'])) {
            $title = "COMPRA DE DIVISAS<br/>";
        }
    }

    // Get beneficiary and remove its name from content
    $beneficiario = getBeneficiary($remarksSet, $content);
    $content = str_replace([$beneficiario, $title], '', $content);

    return $beneficiario . $title . $content;
}

// Specific functions using the consolidated helper
function processTraspasoEntreCuentasPropias($remarksSet, $bancoCuentaDestino, $bancoCuentaOrigen): string
{
    return processTransaction($remarksSet, $bancoCuentaDestino, $bancoCuentaOrigen);
}

function processHonorariosOrArrendamiento($remarksSet, $bancoCuenta, bool $isArrendamiento): string
{
    return processTransaction($remarksSet, $bancoCuenta, null, $isArrendamiento);
}

function processCompraDeDivisas($remarksSet, $bancoCuentaDestino, $bancoCuentaOrigen): string
{
    return processTransaction($remarksSet, $bancoCuentaDestino, $bancoCuentaOrigen, false, true);
}
