<?php

use GreenApi\RestApi\GreenApiClient;

class WhatsAppClient {

    private GreenApiClient $greenApi;
    private string $basePath;

    private const ID_INSTANCE = "7105178749";
    private const API_TOKEN_INSTANCE = "c60b118c738c406d915d241839ba83b11221521e6a504bc1ac";

    public function __construct() {
        $this->greenApi = new GreenApiClient(self::ID_INSTANCE, self::API_TOKEN_INSTANCE);
        $this->basePath = "c:/wamp/www/vitex/uploads/";
    }

    /**
     * Set the base path for file uploads.
     * @param string $path
     */
    public function setBasePath(string $path): void {
        $this->basePath = rtrim($path, '/') . '/';
    }

    /**
     * Clean and format a phone number to the required format.
     * @param string $phoneNumber
     * @return string
     * @throws InvalidArgumentException
     */
    private function formatPhoneNumber(string $phoneNumber): string {
        $cleanedNumber = preg_replace('/\D/', '', $phoneNumber);

        if (strlen($cleanedNumber) !== 10) {
            throw new InvalidArgumentException("Phone number must be 10 digits long.");
        }

        return "521" . $cleanedNumber . "@c.us";
    }

    /**
     * Send a text or web link message.
     * @param string $phoneNumber
     * @param string $message
     * @param object|null $sendMessageResponse
     * @return bool
     */
    public function sendTextMessage(string $phoneNumber, string $message, ?object &$sendMessageResponse = null): bool {
        $formattedNumber = $this->formatPhoneNumber($phoneNumber);

        if (filter_var($message, FILTER_VALIDATE_URL)) {
            return $this->sendWebLink($phoneNumber, $message, $sendMessageResponse);
        }

        $sendMessageResponse = $this->greenApi->sending->sendMessage($formattedNumber, $message);
        return $sendMessageResponse->code == 200 && empty($sendMessageResponse->error);
    }

    /**
     * Send a web link message.
     * @param string $phoneNumber
     * @param string $webLink
     * @param object|null $sendLinkResponse
     * @return bool
     */
    public function sendWebLink(string $phoneNumber, string $webLink, ?object &$sendLinkResponse = null): bool {
        $formattedNumber = $this->formatPhoneNumber($phoneNumber);
        $sendLinkResponse = $this->greenApi->sending->sendLink($formattedNumber, $webLink);
        return $sendLinkResponse->code == 200 && empty($sendLinkResponse->error);
    }

    /**
     * Send a media file (PDF, PNG, or JPEG).
     * @param string $phoneNumber
     * @param string $filePath
     * @param string|null $caption
     * @param object|null $sendMediaMessageResponse
     * @return bool
     * @throws InvalidArgumentException
     */
    public function sendMediaMessage(string $phoneNumber, string $filePath, ?string $caption = null, ?object &$sendMediaMessageResponse = null): bool {
        $formattedNumber = $this->formatPhoneNumber($phoneNumber);
        $fullFilePath = $this->basePath . ltrim($filePath, '/');

        $allowedExtensions = ["pdf", "png", "jpeg", "jpg"];
        $fileExtension = strtolower(pathinfo($fullFilePath, PATHINFO_EXTENSION));
        if (!in_array($fileExtension, $allowedExtensions)) {
            throw new InvalidArgumentException("Unsupported file type. Allowed types: PDF, PNG, JPEG.");
        }

        $sendMediaMessageResponse = $this->greenApi->sending->sendFileByUpload($formattedNumber, $fullFilePath, $caption ?? "");
        return $sendMediaMessageResponse->code == 200 && empty($sendMediaMessageResponse->error);
    }

    /**
     * Create a new group.
     * @param string $groupName
     * @param array $participants
     * @param object|null $createGroupResponse
     * @return bool
     * @throws InvalidArgumentException
     */
    public function createGroup(string $groupName, array $participants, ?object &$createGroupResponse = null): bool {
        if (empty($participants)) {
            throw new InvalidArgumentException("Participants list cannot be empty.");
        }

        $formattedParticipants = array_map(fn($number) => $this->formatPhoneNumber($number), $participants);

        $createGroupResponse = $this->greenApi->groups->createGroup($groupName, $formattedParticipants);
        return $createGroupResponse->code == 200 && empty($createGroupResponse->error);
    }

    /**
     * Check if a group exists.
     * @param string $groupId
     * @return bool
     */
    public function groupExists(string $groupId): bool {
        try {
            $groupInfo = $this->greenApi->groups->getGroupData($groupId);
            return !empty($groupInfo);
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Send a message (text or media) to a group.
     * @param string $groupId
     * @param string $message
     * @param string|null $filePath
     * @param string|null $caption
     * @param object|null $sendMessageToGroupResponse
     * @return bool
     * @throws InvalidArgumentException
     */
    public function sendMessageToGroup(string $groupId, string $message, ?string $filePath = null, ?string $caption = null, ?object &$sendMessageToGroupResponse = null): bool {
        if ($filePath) {
            $fullFilePath = $this->basePath . ltrim($filePath, '/');

            $allowedExtensions = ["pdf", "png", "jpeg", "jpg"];
            $fileExtension = strtolower(pathinfo($fullFilePath, PATHINFO_EXTENSION));
            if (!in_array($fileExtension, $allowedExtensions)) {
                throw new InvalidArgumentException("Unsupported file type. Allowed types: PDF, PNG, JPEG.");
            }

            $sendMessageToGroupResponse = $this->greenApi->sending->sendFileByUpload($groupId . "@g.us", $fullFilePath, $caption ?? "");
            return $sendMessageToGroupResponse->code == 200 && empty($sendMessageToGroupResponse->error);
        }

        $sendMessageToGroupResponse = $this->greenApi->sending->sendMessage($groupId . "@g.us", $message);
        return $sendMessageToGroupResponse->code == 200 && empty($sendMessageToGroupResponse->error);
    }
}

// Example Usage
//try {
//    $client = new WhatsAppClient();
//    $client->setBasePath("c:/wamp/www/vitex/uploads/");
//
//    // Send a text message
//    $textResult = null;
//    $textSuccess = $client->sendTextMessage("5512950121", "Hello from WhatsAppClient!", $textResult);
//    var_dump($textSuccess, $textResult);
//
//    // Send a web link
//    $linkResult = null;
//    $linkSuccess = $client->sendTextMessage("5512950121", "https://example.com", $linkResult);
//    var_dump($linkSuccess, $linkResult);
//
//    // Send a media file
//    $mediaResult = null;
//    $mediaSuccess = $client->sendMediaMessage("5512950121", "inversion/c45ab1d86841b60a11eddd42c6f16561/whatsapp_image_2023-04-14_at_5.58.31_pm.jpeg", "Here is your file.", $mediaResult);
//    var_dump($mediaSuccess, $mediaResult);
//
//    // Create a group
//    $groupResult = null;
//    $groupSuccess = $client->createGroup("Test Group", ["5512950121", "5512345678"], $groupResult);
//    var_dump($groupSuccess, $groupResult);
//
//    // Check if a group exists
//    $groupExists = $client->groupExists("1234567890-1234567890@g.us");
//    var_dump($groupExists);
//
//    // Send a message to a group
//    $groupMessageResult = null;
//    $groupMessageSuccess = $client->sendMessageToGroup("1234567890-1234567890", "Hello Group!", null, null, $groupMessageResult);
//    var_dump($groupMessageSuccess, $groupMessageResult);
//
//} catch (Exception $e) {
//    echo "Error: " . $e->getMessage();
//}
