// noinspection JSUnusedGlobalSymbols

/**
 * Format numbers, dates, links, strings may add html tag with attriubtes
 *
 */
class Fmt {
    #integerFormat = {decimals: 0, class:"right"};
    #number2decimals = {decimals: 2, class:"right"};
    #fmtInt =
        new Intl.NumberFormat('en-US',{minimumFractionDigits:0, maximumFractionDigits:0});
    #fmt2 =
        new Intl.NumberFormat('en-US',{minimumFractionDigits:2, maximumFractionDigits:2});

    // ¡el key en minusculas!
    columnFormat = {
        count: this.#integerFormat,
        _count: this.#integerFormat,
        rolls: this.#integerFormat,
        rollos: this.#integerFormat,
        quantity: this.#number2decimals,
        qty: this.#number2decimals,

        price: this.#number2decimals,
        prices: this.#number2decimals,
        precio: this.#number2decimals,
        precios: this.#number2decimals,
        cost: this.#number2decimals,
        costs: this.#number2decimals,
        costo: this.#number2decimals,
        costos: this.#number2decimals,
        balance: this.#number2decimals,
        saldo: this.#number2decimals,
        saldos: this.#number2decimals,
        deposit: this.#number2decimals,
        deposits: this.#number2decimals,
        deposito: this.#number2decimals,
        depositos: this.#number2decimals,
        withdrawal: this.#number2decimals,
        withdrawals: this.#number2decimals,
        retiro: this.#number2decimals,
        retiros: this.#number2decimals,
        cargo: this.#number2decimals,
        abono: this.#number2decimals,
        debe: this.#number2decimals,
        haber: this.#number2decimals,
        charge: this.#number2decimals,
        charges: this.#number2decimals,
        credit: this.#number2decimals,
        credits: this.#number2decimals,

        dob:{dateFormat: "d/M/y"},
        date:{dateFormat: "d/M/y"},
        fecha:{dateFormat: "d/M/y"},
        fecha_nacimiento:{dateFormat: "d/M/y"},
    };

    /**
     * zero based array of month names
     *
     * @type {Array<string>}
     */
    longMonth= [];
    /**
     * zero based array of short month names.
     *
     * @type {Array<string>}
     */
    shortMonth= [];
    /**
     * Array of day names 0=Sunday.
     *
     * @type {Array<string>}
     */
    longDay= [];
    /**
     * Array of short day names 0=Sunday.
     *
     * @type {Array<string>}
     */
    shortDay = [];

    constructor(lang = 'es') {
        this.setLanguage(lang);
    }

    setLanguage(lang = "es") {
        switch(lang.toLowerCase()) {
            case 'en':
                this.longMonth = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September',
                    'October', 'November', 'December'];
                this.shortMonth = ['Jan', 'Feb', 'Mar', 'Apr', 'May',  'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
                this.longDay = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                this.shortDay = ['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa'];
                return;
            case 'es':
            default:
                this.longMonth = ['Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio', 'Julio', 'Agosto',
                    'Septiembre', 'Octubre', 'Noviembre', 'Diciembre'];
                this.shortMonth = ['Ene', 'Feb', 'Mar', 'Abr', 'May',  'Jun', 'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'];
                this.longDay = ['Domingo', 'Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes', 'Sábado'];
                this.shortDay = ['Do', 'Lu', 'Ma', 'Mi', 'Ju', 'Vi', 'Sa'];
                return;
        }
    }
    col(colName, value, record, htmlAttributes = {}) {
        let colIndex = colName.toLowerCase().trim();
        if(!this.columnFormat.hasOwnProperty(colIndex))
            return this.auto(value, record,htmlAttributes);
        let colOptions = {...this.columnFormat[colIndex], ...htmlAttributes};
        if(colOptions.hasOwnProperty('decimals'))
            return this.num(value, colOptions.decimals,record, colOptions);
        if(colOptions.hasOwnProperty('dateFormat'))
            return this.date(value, colOptions.dateFormat, record,colOptions);
        if(colOptions.hasOwnProperty('href'))
            return this.link(value, record, colOptions);
        return this.auto(value,record, colOptions);
    }

    auto(value, record, htmlAttributes = {}) {
        if(value === null || value === "")
            return this.htmlTag("", record,htmlAttributes);
        if(value instanceof Date || this.#is_ymd(value))
            return this.date(value, "d/M/y", record, { ...{class:"cen"}, ...htmlAttributes});
        if(!isNaN(value))
            return  this.num(value, htmlAttributes.decimals ?? 2, record,{...{class:"der"},  ...htmlAttributes} );
        return this.htmlTag(value, record, htmlAttributes);
    }

    #is_ymd(value) {
        if(value === null)
            return false;
        if(typeof value !== 'string')
            return false;
        if(!isNaN(value))
            return false;
        if(value.length < 10 || value.length > 27)
            return false;
        const regex =
            /^\d\d\d\d[-.\\\/_](0[1-9]|1[0-2])[-.\\\/_](0[1-9]|[1-2][0-9]|3[0|1])($|([ T]([0-1]\d|2[0-4]):[0-5]\d:[0-5]\d))/gm;
        return value.match(regex) !== null;
    }

    num(n, decimals = 2, record, htmlAttributes = {}) {
        if(n === null)
            return this.htmlTag('', record,htmlAttributes);
        if(isNaN(n))
            return this.htmlTag(n, record, htmlAttributes);
        switch(decimals) {
            case "0":
            case 0:
                return this.htmlTag( this.#fmtInt.format(n),  record,htmlAttributes);
            case "2":
            case 2:
                return this.htmlTag(this.#fmt2.format(n),  record,htmlAttributes);
            default:
                let fmt = new Intl.NumberFormat('en-US',
                    {minimumFractionDigits:decimals, maximumFractionDigits:decimals});
                return this.htmlTag(fmt.format(n),  record,htmlAttributes);
        }
    }

    date(inputDate, format = "d/M/y", record, htmlAttributes = {})  {
        if(inputDate === null)
            return this.htmlTag('', htmlAttributes);
        try {
            let date;
            if(inputDate instanceof Date)
                date = inputDate;
            else if(typeof inputDate === 'object')
                return this.htmlTag("[object]", htmlAttributes);
            else if(isNaN(inputDate))
                switch(inputDate.trim().length) {
                    case 10:
                        date = new Date(`${inputDate}T00:00:00`);
                        break;
                    default:
                        date = new Date(inputDate);
                }
            else
                date = new Date(inputDate);

            const parts = {
                d: this.#padZero(date.getDate()),
                j: date.getDate(),
                D: this.shortDay[date.getDay()],
                l: this.longDay[date.getDay()],
                w: date.getDay(),

                m: this.#padZero(date.getMonth() + 1),
                n: date.getMonth() + 1,
                M: this.shortMonth[date.getMonth()],
                F: this.longMonth[date.getMonth()],

                Y: date.getFullYear(),
                y: date.getFullYear().toString().slice(-2),

                H: this.#padZero(date.getHours()),
                G: date.getHours(),
                h: this.#padZero(date.getHours() > 12 ? date.getHours() - 12 : date.getHours()),
                g: date.getHours() > 12 ? date.getHours() - 12 : date.getHours(),
                i: this.#padZero(date.getMinutes()),
                s: this.#padZero(date.getSeconds()),

                a: date.getHours() < 12 ? 'am' : 'pm',
                A: date.getHours() < 12 ? 'AM' : 'PM',
            };

            let skip = false;
            let ret = [];
            for(let i = 0, len = format.length; i < len; ++i) {
                let c = format[i];
                if(c === "\\") {
                    skip = true;
                    continue;
                }
                if(skip) {
                    skip = false;
                    ret.push(c);
                    continue;
                }
                ret.push(parts.hasOwnProperty(c) ? parts[c] : c);
            }
            return this.htmlTag(ret.join(""), record, htmlAttributes);
        } catch(error) {
            console.log("ERROR: fmt.date arguments:", arguments);
            console.log("       fmt.date error:", error);
            return this.htmlTag('', record , htmlAttributes);
        }
    }

    link(text, record, htmlAttributes = {}) {
        return this.htmlTag(text, htmlAttributes, "a");
    }

    string(text, record, htmlAttributes, tag) {return this.htmlTag(text, htmlAttributes, tag);}
    htmlTag(html, record, htmlAttributes = {}, tag) {
        if(html === null)
            html = '';
        if(typeof tag === "undefined")
            tag = htmlAttributes.tag ?? "span";
        let opt = {...htmlAttributes}
        delete(opt.decimals);
        delete(opt.dateFormat);
        delete(opt.tag);

        let attributes = [];
        for(let attribute in opt)
            if(opt.hasOwnProperty(attribute)) {
                let value = typeof opt[attribute] === 'object' ?
                    JSON.stringify(opt[attribute]) : opt[attribute] + "";

                attributes.push(attribute.replaceAll(/[="'>]/g, "") + "='" +
                    value.replaceAll("'", "&quote;") + "'");
            }
        return  `<${tag} ${attributes.join(" ")}>${html}</${tag}>`;
    }

    label(str, record, htmlAttributes = {}) {
        return this.htmlTag(
            this.ucWords( str.replace( /([A-Z\u00C0-\u00DC]|\d+)/gm, " $1").replaceAll('_', ' ')),
            record, htmlAttributes
        );
    }

    breadcrumbs(breadcrumbs, record) {
        let crumbs = [];
        for(let b of breadcrumbs) {
            let key = Object.keys(b)[0];
            crumbs.push(`<div>${this.label(key, record,{tag:"span"})}<p>${this.label(b[key], record,{tag:"span"})}</p></div>`);
        }
        return `<div class="">${crumbs.join("<div>→</div>")}</div>`;
    }

    #padZero(value) {return value < 10  ? `0${value}` : `${value}`;}

    ucWords = string => {
        return String(string).toLowerCase()
            .replace(/\b[a-z]/g, (l) => l.toUpperCase());
    }
}
