<?php
/**
 * ============================================================================
 * Propiedad Expansion Script - Thoth's Algorithm
 * ============================================================================
 *
 * Purpose: Transform property rows into individual apartment rows
 * Transformation: 114 properties → 350 individual units
 * Method: In-place expansion using num_deptos as multiplier
 *
 * Author: Claude Code (Thoth's Algorithm)
 * Date: 2025-01-02
 *
 * Usage:
 *   Preview: /lamp/php/bin/php 02_expand_propiedades_to_units.php --preview
 *   Execute: /lamp/php/bin/php 02_expand_propiedades_to_units.php --execute
 *   Report:  /lamp/php/bin/php 02_expand_propiedades_to_units.php --report
 *
 * ============================================================================
 */

// Minimal bootstrap - set up global database configuration
$gIAsql = [
    'host' => 'localhost',
    'port' => null,
    'socket' => '/lamp/mysql/mysql.sock',
    'user' => 'root',
    'pwd' => 'M@chiavell1',
    'dbname' => 'quantix'
];

// Direct database connection
$gIAsql['link'] = mysqli_connect(
    $gIAsql['host'],
    $gIAsql['user'],
    $gIAsql['pwd'],
    $gIAsql['dbname'],
    null,
    $gIAsql['socket']
);

if (!$gIAsql['link']) {
    die("❌ ERROR: Could not connect to database: " . mysqli_connect_error() . "\n");
}

mysqli_set_charset($gIAsql['link'], 'utf8mb4');

// Include helper functions
require_once(__DIR__ . '/../../inc/ia_utilerias.php');

// ============================================================================
// CONFIGURATION
// ============================================================================

$MODE = 'preview'; // preview | execute | report
if (isset($argv[1])) {
    $MODE = str_replace('--', '', $argv[1]);
}

if (!in_array($MODE, ['preview', 'execute', 'report'])) {
    die("❌ Invalid mode. Use: --preview, --execute, or --report\n");
}

$REPORT_DATA = [];
$STATS = [
    'original_rows' => 0,
    'new_rows_created' => 0,
    'rows_updated' => 0,
    'total_final_rows' => 0,
    'single_units' => 0,
    'multi_units' => 0,
];

echo "\n";
echo "🔱 Propiedad Expansion - Thoth's Algorithm\n";
echo "============================================\n";
echo "Mode: " . strtoupper($MODE) . "\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

/**
 * Extract unit number from property name
 */
function extractUnitNumber($nombre_propiedad) {
    // Pattern 1: "Street Name - 101" (explicit unit number)
    if (preg_match('/\s-\s(\d+[A-Z]?|[A-Z]\d*)$/', $nombre_propiedad, $matches)) {
        return $matches[1];
    }

    // Pattern 2: "Property Name A" or "Property Name B" (letter suffix)
    if (preg_match('/\s([A-Z])$/', $nombre_propiedad, $matches)) {
        return $matches[1];
    }

    // Pattern 3: Parenthetical hint like "(depto 5)"
    if (preg_match('/\(depto\s+(\d+|[A-Z])\)/i', $nombre_propiedad, $matches)) {
        return $matches[1];
    }

    // Default: No explicit unit found
    return null;
}

/**
 * Clean property name (remove hints like "(12 Deptos)")
 */
function cleanPropertyName($nombre_propiedad) {
    // Remove patterns like "(12 Deptos)" or "(9 deptos)"
    $cleaned = preg_replace('/\s*\(\d+\s*deptos?\)/i', '', $nombre_propiedad);

    // Remove trailing " - " if exists
    $cleaned = preg_replace('/\s*-\s*$/', '', $cleaned);

    return trim($cleaned);
}

/**
 * Extract floor number from unit number
 * Examples: "101" → 1, "201" → 2, "1001" → 10, "A" → null
 */
function extractFloorFromUnit($numero_unidad) {
    if (empty($numero_unidad)) return null;

    // 3-digit format: 101, 201, 301 (first digit = floor)
    if (preg_match('/^(\d)(\d{2})$/', $numero_unidad, $matches)) {
        return (int)$matches[1];
    }

    // 4-digit format: 1001, 2001 (first 2 digits = floor)
    if (preg_match('/^(\d{2})(\d{2})$/', $numero_unidad, $matches)) {
        return (int)$matches[1];
    }

    // Letter or non-standard format
    return null;
}

/**
 * Generate unit number based on index and total count
 */
function generateUnitNumber($index, $total_units) {
    // For small buildings (≤10 units), use letters: A, B, C, etc.
    if ($total_units <= 10 && $total_units > 1) {
        return chr(64 + $index); // A=65, B=66, etc.
    }

    // For larger buildings, use zero-padded numbers: 01, 02, ..., 24
    $padding = strlen((string)$total_units); // Auto-adjust padding (24→2 digits, 100→3 digits)
    return str_pad($index, $padding, '0', STR_PAD_LEFT);
}

/**
 * Determine unit type based on property name
 */
function determineUnitType($nombre_propiedad) {
    $lower = mb_strtolower($nombre_propiedad, 'UTF-8');

    if (str_contains($lower, 'casa')) return 'Casa';
    if (str_contains($lower, 'penthouse') || str_contains($lower, 'ph')) return 'Penthouse';
    if (str_contains($lower, 'loft')) return 'Loft';
    if (str_contains($lower, 'estudio')) return 'Estudio';

    return 'Departamento'; // Default
}

// ============================================================================
// MAIN EXECUTION
// ============================================================================

// 1. Load all properties
$sql = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql);

if (empty($propiedades)) {
    die("❌ ERROR: No properties loaded from database!\n");
}

$STATS['original_rows'] = count($propiedades);

echo "✓ Loaded {$STATS['original_rows']} properties\n";
echo "Calculating expansions...\n\n";

// 2. Process each property
foreach ($propiedades as $idx => $propiedad) {
    $original_id = $propiedad['propiedad_id'];
    $num_deptos = (int)$propiedad['num_deptos'];
    $base_name = cleanPropertyName($propiedad['nombre_propiedad']);
    $tipo_unidad = determineUnitType($base_name);

    if ($num_deptos == 1) {
        // ================================================================
        // SINGLE UNIT: Just update with extracted unit number
        // ================================================================

        $STATS['single_units']++;

        $extracted_unit = extractUnitNumber($propiedad['nombre_propiedad']);
        $numero_unidad = $extracted_unit ?: '01';
        $piso = extractFloorFromUnit($numero_unidad);

        if ($MODE == 'preview') {
            echo "[SINGLE] {$propiedad['nombre_propiedad']}\n";
            echo "  → Keep as-is, add unit: $numero_unidad" . ($piso ? ", floor: $piso" : "") . "\n\n";
        }

        if ($MODE == 'execute') {
            $sql = ia_update('propiedad', [
                'numero_unidad' => $numero_unidad,
                'piso' => $piso,
                'tipo_unidad' => $tipo_unidad,
            ], ['propiedad_id' => $original_id]);
            ia_query($sql);
            $STATS['rows_updated']++;
        }

        $REPORT_DATA[] = [
            'original_id' => $original_id,
            'new_id' => $original_id,
            'nombre_original' => $propiedad['nombre_propiedad'],
            'nombre_nuevo' => $propiedad['nombre_propiedad'],
            'numero_unidad' => $numero_unidad,
            'piso' => $piso ?: '',
            'action' => 'updated_single',
        ];

    } else {
        // ================================================================
        // MULTI-UNIT: Split into multiple rows
        // ================================================================

        $STATS['multi_units']++;

        if ($MODE == 'preview') {
            echo "[MULTI] {$propiedad['nombre_propiedad']} ($num_deptos units)\n";
        }

        for ($i = 1; $i <= $num_deptos; $i++) {
            $numero_unidad = generateUnitNumber($i, $num_deptos);
            $new_name = "$base_name - $numero_unidad";
            $piso = extractFloorFromUnit($numero_unidad);

            if ($i == 1) {
                // Update first row (keep original ID)
                if ($MODE == 'preview') {
                    echo "  [1/$num_deptos] UPDATE original: $new_name (unit: $numero_unidad)\n";
                }

                if ($MODE == 'execute') {
                    $sql = ia_update('propiedad', [
                        'num_deptos' => 1,
                        'nombre_propiedad' => $new_name,
                        'numero_unidad' => $numero_unidad,
                        'piso' => $piso,
                        'tipo_unidad' => $tipo_unidad,
                    ], ['propiedad_id' => $original_id]);
                    ia_query($sql);
                    $STATS['rows_updated']++;
                }

                $REPORT_DATA[] = [
                    'original_id' => $original_id,
                    'new_id' => $original_id,
                    'nombre_original' => $propiedad['nombre_propiedad'],
                    'nombre_nuevo' => $new_name,
                    'numero_unidad' => $numero_unidad,
                    'piso' => $piso ?: '',
                    'action' => 'updated_multi_first',
                ];

            } else {
                // Insert new rows for units 2-N
                $new_id = ia_guid('propiedad_expansion');

                if ($MODE == 'preview') {
                    echo "  [$i/$num_deptos] INSERT new: $new_name (unit: $numero_unidad, id: $new_id)\n";
                }

                if ($MODE == 'execute') {
                    $sql = ia_insert('propiedad', [
                        'propiedad_id' => $new_id,
                        'num_deptos' => 1,
                        'nombre_propiedad' => $new_name,
                        'numero_unidad' => $numero_unidad,
                        'piso' => $piso,
                        'propietario_id' => $propiedad['propietario_id'],
                        'direccion' => $propiedad['direccion'],
                        'vale' => $propiedad['vale'],
                        'codigo_postal' => $propiedad['codigo_postal'],
                        'colonia' => $propiedad['colonia'],
                        'estado' => $propiedad['estado'],
                        'estado_descripcion' => $propiedad['estado_descripcion'],
                        'municipio' => $propiedad['municipio'],
                        'municipio_descripcion' => $propiedad['municipio_descripcion'],
                        'tipo_unidad' => $tipo_unidad,
                        'alta_por' => 'expansion_script',
                    ]);
                    ia_query($sql);
                    $STATS['new_rows_created']++;
                }

                $REPORT_DATA[] = [
                    'original_id' => $original_id,
                    'new_id' => $new_id,
                    'nombre_original' => $propiedad['nombre_propiedad'],
                    'nombre_nuevo' => $new_name,
                    'numero_unidad' => $numero_unidad,
                    'piso' => $piso ?: '',
                    'action' => 'inserted_new',
                ];
            }
        }

        if ($MODE == 'preview') {
            echo "\n";
        }
    }
}

// ============================================================================
// FINAL STATISTICS
// ============================================================================

$STATS['total_final_rows'] = $STATS['original_rows'] + $STATS['new_rows_created'];

echo "\n" . str_repeat("=", 70) . "\n";
echo "STATISTICS\n";
echo str_repeat("=", 70) . "\n";
echo sprintf("Original rows:        %4d\n", $STATS['original_rows']);
echo sprintf("Single-unit props:    %4d\n", $STATS['single_units']);
echo sprintf("Multi-unit props:     %4d\n", $STATS['multi_units']);
echo sprintf("Rows updated:         %4d\n", $STATS['rows_updated']);
echo sprintf("New rows created:     %4d\n", $STATS['new_rows_created']);
echo sprintf("Total final rows:     %4d (expected: 350)\n", $STATS['total_final_rows']);
echo "\n";

if ($MODE == 'execute') {
    echo "✅ EXPANSION COMPLETE!\n\n";
    echo "Verify with:\n";
    echo "  /lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock quantix\n";
    echo "  SELECT COUNT(*) FROM propiedad; -- Should be 350\n";
    echo "  SELECT COUNT(*) FROM propiedad WHERE num_deptos = 1; -- Should be 350\n";
    echo "  SELECT nombre_propiedad FROM propiedad WHERE nombre_propiedad LIKE 'Medellin 148%';\n";
    echo "\n";
}

if ($MODE == 'preview') {
    echo "ℹ️  This was a preview - no changes made to database\n";
    echo "   Run with --execute to apply changes\n\n";
}

// ============================================================================
// REPORT GENERATION
// ============================================================================

if ($MODE == 'report' || $MODE == 'execute') {
    $report_file = __DIR__ . '/propiedad_expansion_report_' . date('Y-m-d_His') . '.csv';
    $fp = fopen($report_file, 'w');

    if (!$fp) {
        echo "⚠️  WARNING: Could not create report file\n";
    } else {
        // CSV Header
        fputcsv($fp, [
            'original_id',
            'new_id',
            'nombre_original',
            'nombre_nuevo',
            'numero_unidad',
            'piso',
            'action'
        ]);

        // CSV Data
        foreach ($REPORT_DATA as $row) {
            fputcsv($fp, $row);
        }

        fclose($fp);
        echo "📄 Report saved: $report_file\n";
        echo "   " . count($REPORT_DATA) . " rows exported\n\n";
    }
}

// ============================================================================
// CLOSING
// ============================================================================

mysqli_close($gIAsql['link']);

echo "✨ Script complete - " . date('Y-m-d H:i:s') . "\n";
echo "   Thoth's Algorithm: The undeniable has been revealed.\n\n";
?>
