<?php

namespace ia\Date;


use function Date;
use DateTime;
use DateTimeImmutable;
use DateTimeInterface;
use Exception;

class DateUtil {

    /**
     * @param string $ymd
     * @return bool true ymd is a valid date
     */
    public static function validYMD($ymd) {
        if(strlen($ymd) !== 10) {
            return false;
        }
        $dateParts = explode('-', $ymd);
        if(count($dateParts)!=3) {
            return false;
        }
        foreach($dateParts as $d) {
            if(!is_numeric($d)) {
                return false;
            }
        }
        return checkdate($dateParts[1], $dateParts[2], $dateParts[0]);
    }

    //https://ourcodeworld.com/articles/read/756/how-to-round-up-down-to-nearest-10-or-5-minutes-of-datetime-in-php

    /**
     * @param DateTimeInterface $dateTime
     * @param int $minuteInterval
     * @return DateTimeImmutable|false
     */
    public static function roundToNearestMinuteInterval(DateTimeInterface $dateTime, $minuteInterval = 10) {
        if( !($dateTime instanceof DateTimeImmutable) ) {
            $dateTime = clone $dateTime;
        }
        return $dateTime->setTime(
            $dateTime->format('H'),
            round($dateTime->format('i') / $minuteInterval, 0) * $minuteInterval,
            0
        );
    }

    /**
     * @param DateTimeInterface $dateTime
     * @param int $minuteInterval
     * @return DateTimeImmutable|false
     */
    public static function roundUpToMinuteInterval(DateTimeInterface $dateTime, $minuteInterval = 10) {
        if( !($dateTime instanceof DateTimeImmutable) ) {
            $dateTime = clone $dateTime;
        }
        return $dateTime->setTime(
            $dateTime->format('H'),
            ceil($dateTime->format('i') / $minuteInterval) * $minuteInterval,
            0
        );
    }

    /**
     * @param DateTimeInterface $dateTime
     * @param int $minuteInterval
     * @return DateTimeImmutable|false
     */
    public static function roundDownToMinuteInterval(DateTimeInterface $dateTime, $minuteInterval = 10) {
        if( !($dateTime instanceof DateTimeImmutable) ) {
            $dateTime = clone $dateTime;
        }
        return $dateTime->setTime(
            $dateTime->format('H'),
            floor($dateTime->format('i') / $minuteInterval) * $minuteInterval,
            0
        );
    }


    /**
     * Get quarter, trimestre, for a date
     *
     * @param string|integer|DateTimeInterface $anyDate
     * @return int quarter 1-4
     * @throws Exception
     */
    public static function quarter($anyDate) {
        /*
            http://de.wikipedia.org/wiki/Quartal#Quartale_und_ihre_Monate
                First quarter: from the beginning of January to the end of March
                Second quarter: from the beginning of April to the end of June
                Third quarter: from the beginning of July to the end of September
                Fourth quarter: from the beginning of October to the end of December

        */

        if($anyDate instanceOf DateTimeInterface ) {
            $month = (int)$anyDate->format('n');
        } elseif(is_numeric($anyDate)) {
            $month = (int)Date('n', $anyDate);
        } else {
            $month =  (int)Date('n', strtotime($anyDate));
        }

        if($month <= 3) {
            return 1;
        }
        if($month <= 6) {
            return 2;
        }

        return $month <= 9 ? 3 : 4;
    }

    /**
     * Get day of week
     *
     * @param string|integer|DateTimeInterface $anyDate
     * @return int 0 Sunday ... 6 Saturday
     * @throws Exception
     */
    public static function dayOfWeek($anyDate) {
        if($anyDate instanceOf DateTimeInterface) {
            return (int)$anyDate->format('w');
        }
        if(is_numeric($anyDate)) {
            return (int)Date('w', $anyDate);
        }
        return (int)(new DateTime($anyDate))->format('w');
    }

    /**
     *
     *
     * @param string|integer|DateTime|DateTimeImmutable $anyDate
     * @return string date formatead as Y-m-d
     */
    public static function toYmd($anyDate) {
        if(is_string($anyDate)) {
            if(self::validYMD($anyDate)) {
                return substr($anyDate, 0, 10);
            }
            return Date('Y-m-d', strtotime($anyDate));
        }
        if(is_numeric($anyDate)) {
            return Date('Y-m-d', $anyDate);
        }
        return $anyDate->format('Y-m-d');
    }

    /**
     * Get a DateTimeImmutable with only the date, hour, minutes & seconds = 0
     *
     * @param string|integer|DateTime|DateTimeImmutable $anyDate
     * @return DateTimeImmutable|false with only the date, hour, minutes & seconds = 0
     * @throws Exception
     */
    public static function toDateImmutable($anyDate) {
        if(is_numeric($anyDate)) {
            return  date_create_immutable(Date('Y-m-d 00:00:00', $anyDate));
        }
        if(is_string($anyDate)) {
            return (new DateTimeImmutable($anyDate))->setTime(0, 0, 0);
        }
        if($anyDate instanceOf DateTimeImmutable ) {
            return $anyDate->setTime(0, 0, 0);
        }
        return new DateTimeImmutable($anyDate->format('Y-m-d 00:00:00') );
    }

    /**
     * Get a DateTimeImmutable from a date
     *
     * @param string|integer|DateTimeInterface|DateTime $anyDate
     * @return DateTimeImmutable
     * @throws Exception
     */
    public static function toDateTimeImmutable($anyDate) {
        if($anyDate === null) {
            return new DateTimeImmutable(null);
        }
        if(is_numeric($anyDate)) {
            return new DateTimeImmutable(Date('Y-m-d H:i:s', $anyDate));
        }
        if(is_string($anyDate)) {
            return new DateTimeImmutable($anyDate);
        }
        if($anyDate instanceOf DateTimeImmutable) {
            return $anyDate;
        }
        return DateTimeImmutable::createFromMutable( $anyDate ); //
       // return new DateTimeImmutable($anyDate->format('Y-m-d H:i:s') );
    }

    /**
     * Get a DateTimeImmutable at the date with hour o'clock, minutes & seconds = 0
     *
     * @param string|integer|DateTimeInterface|DateTime $anyDate
     * @return DateTimeImmutable with minutes & seconds = 0
     * @throws Exception
     */
    public static function toDateHourImmutable($anyDate) {
        if(is_numeric($anyDate)) {
            return new DateTimeImmutable(Date('Y-m-d H:00:00', $anyDate));
        }
        if(is_string($anyDate)) {
            $dateTime = new DateTimeImmutable($anyDate);
            return $dateTime->setTime((int)$dateTime->format('G'), 0, 0);
        }
        if($anyDate instanceOf DateTimeImmutable ) {
            return $anyDate->setTime((int)$anyDate->format('G'), 0, 0);
        }
        return DateTimeImmutable::createFromMutable( $anyDate )->setTime((int)$anyDate->format('G'), 0, 0); //  DateTimeImmutable($anyDate->format('Y-m-d H:00:00') );
    }


    /**
     * Ensure $start <= $end, else swap it. Returns -1 when swapped, 1 not swapped
     *
     * @param DateTimeImmutable $start
     * @param DateTimeImmutable $end
     * @return int Returns -1 when swapped, 1 not swapped, and $start is <= $end
     */
    public static function ordered(DateTimeImmutable &$start, DateTimeImmutable &$end) {
        if($start <= $end) {
            return 1;
        }
        $swap = $start;
        $start = $end;
        $end = $swap;
        return -1;
    }
}