<?php
class Feriados {
	const LA_FECHA=0;	// espera ano-mes-dia, unicamente ese ano-mes-dia es feriado
	const CADA_ANO=1;	// espera mes-dia y justo ese mes-dia se repite anualmente
	const N_DAY=2;		// espera mes, diasem el dia semana a feriar, ndia el numero del diasem en el mes (1 a 4, 5 es ultimo), se repite anualmente

	// Asume php Date('w') 0 (for Sunday) through 6 (for Saturday)
	const DOM=0;
	const LUN=1;
	const MAR=2;
	const MIE=3;
	const JUE=4;
	const VIE=5;
	const SAB=6;

	protected $dia_habil = array( 0=>0,1=>1,2=>1,3=>1,4=>1,5=>1,6=>0);
	protected $feriados_define=array();
	protected $feriados=array();

	public function __construct($db_table=null, $sabado_habil=false, $domingo_habil=false) {
		$this->dia_habil_set(self::SAB,$sabado_habil);
		$this->dia_habil_set(self::DOM,$domingo_habil);
		if(empty($db_table))
			$this->feriados_define_default();
	}

//
	/**
	 * Feriados::es_feriado()
	 * Indica si un dia es habil (false) o feriado (true)
	 * Interface deduce si usar _timesatmp, _dateTime o _mysqlDate
	 *
	 * @param mixed $date Fecha a revisar timestamp, php DateTime, mysql Date o DateTime,
	 * @param bool $considerWeekends true considera fin de semana feriado, false unicamente dias feriados, see dia_habil_set
	 * @return bool false es dia habil, true es dia feriado
	 */
	public function es_feriado($date, $considerWeekends=true) {
		if(is_numeric($date))
			return $this->es_feriado_timestamp($date, $considerWeekends);
		elseif($date instanceof DateTime)
			return $this->es_feriado_dateTime($date, $considerWeekends);
		return $this->es_feriado_mysqlDate($date, $considerWeekends);
	}

		public function es_feriado_timestamp($timestamp, $considerWeekends=true) {
			if( $considerWeekends && empty($this->dia_habil[Date('w',$timestamp)])  )
				return true;
			$year = (int)Date('Y',$timestamp);
			if(!array_key_exists($year,$this->feriados))
				$this->feriados_fill($year);
			return array_key_exists(Date('Y-m-d',$timestamp),$this->feriados[$year]);
		}

		public function es_feriado_mysqlDate($mysqlDate, $considerWeekends=true) {
			if( $considerWeekends && empty($this->dia_habil[Date('w',strtotime($mysqlDate) )])  )
				return true;
			if(strlen($mysqlDate)>10)
				$mysqlDate = substr($mysqlDate,0,10);
			$year = (int)substr($mysqlDate,0,4);
			if(!array_key_exists($year,$this->feriados))
				$this->feriados_fill($year);
			return array_key_exists($mysqlDate,$this->feriados[$year]);
		}

		public function es_feriado_dateTime($dateTime, $considerWeekends=true) {
			return $this->es_feriado_mysqlDate( $dateTime->format('Y-m-d') );
		}


//

	public function habil_siguiente($date, $iDays=1) {
		if(is_numeric($date))
			return $this->habil_siguiente_timestamp($date, $iDays);
		elseif($date instanceof DateTime)
			return $this->habil_siguiente_DateTime($date, $iDays);
		else
			return $this->habil_siguiente_mysqlDate($date, $iDays);
	}

	public function habil_anterior($date,$iDays=1) {
		if(is_numeric($date))
			return $this->habil_anterior_timestamp($date, $iDays);
		elseif($date instanceof DateTime)
			return $this->habil_anterior_DateTime($date, $iDays);
		else
			return $this->habil_anterior_mysqlDate($date, $iDays);
	}

		public function habil_siguiente_timestamp($timestamp, $iDays=1) {
			while($iDays-->0)
				for($i=1;$i<=30;$i++) {
					$timestamp += 86400;
					if( !$this->es_feriado_timestamp($timestamp) )
						break;
				}
			return $timestamp;
		}

		public function habil_anterior_timestamp($timestamp, $iDays=1) {
			while($iDays-->0)
				for($i=1;$i<=30;$i++) {
					$timestamp -= 86400;
					if( !$this->es_feriado_timestamp($timestamp) )
						break;
				}
			return $timestamp;
		}


		public function habil_siguiente_mysqlDate($mysqlDate, $iDays=1) {
			return Date('Y-m-d', $this->habil_siguiente_timestamp( strtotime($mysqlDate),$iDays ) );
		}

		public function habil_anterior_mysqlDate($mysqlDate, $iDays=1) {
			return Date('Y-m-d', $this->habil_anterior_timestamp( strtotime($mysqlDate),$iDays ) );
		}


		public function habil_siguiente_DateTime($dateTime, $iDays=1) {
			return new DateTime( Date('Y-m-d G:i:s', $this->habil_siguiente_timestamp( $dateTime-getTimestamp(),$iDays ) ) );
		}

		public function habil_anterior_DateTime($dateTime, $iDays=1) {
			return new DateTime( Date('Y-m-d G:i:s', $this->habil_anterior_timestamp( $dateTime-getTimestamp(),$iDays ) ) );
		}

// Dia habil por dia de la semana
	public function dia_habil_set($dia,$habil) { $this->dia_habil[$dia] =  $habil; }
	public function dia_habil_get($dia) { return $this->dia_habil[$dia]; }
	public function dias_semana() {return $this->dia_habil; }

// Dias feriados, definicion

	protected function feriados_define_default() {
		$this->feriados_define = array(
				array( 'date'=>'01-01', 'type'=>self::CADA_ANO ),
				array( 'date'=>'02-05', 'type'=>self::N_DAY,'diasem'=>self::LUN, 'nday'=>1 ),
				array( 'date'=>'03-21', 'type'=>self::N_DAY,'diasem'=>self::LUN, 'nday'=>3 ),
				array( 'date'=>'05-01', 'type'=>self::CADA_ANO ),
				array( 'date'=>'09-16', 'type'=>self::CADA_ANO ),
				array( 'date'=>'11-20', 'type'=>self::N_DAY,'diasem'=>self::LUN, 'nday'=>3 ),
				array( 'date'=>'12-24', 'type'=>self::CADA_ANO ),
				array( 'date'=>'12-25', 'type'=>self::CADA_ANO ),
		);
	}

	public function feriados_define_set($define_feriados) {
		$this->feriados_define = $define_feriados;
		$this->feriados_clear();
	}

	public function feriados_define_add_array($add_feriados) {
 		$this->feriados_define = array_merge($this->feriados_define, $add_feriados);
 		$this->feriados_clear();
	}

	public function feriados_define_add_day($date,$type,$diasem=1,$nday=1) {
		$this->feriados_define[] = array('date'=>$date, 'type'=>$type,'diasem'=>$diasem, 'nday'=>$nday);
		$this->feriados_clear();
	}

	protected function feriados_define_clear() {
		$this->feriados_define = array();
		$this->feriados_clear();
	}

// Dias feriados por ano

	public function feriados_get($year=null) {
		if(empty($year))
			$year = Date('Y');
		if(!array_key_exists($year,$this->feriados))
			$this->feriados_fill($year);
		ksort($this->feriados[$year]);
		return $this->feriados[$year];
	}

	/**
	 * Feriados::feriados_clear()
	 * @access private
	 * @return void
	 */
	protected function feriados_clear() {$this->feriados=array();}

	/**
	 * Feriados::feriados_fill()
	 *
	 * @param mixed $year
	 * @return void
	 */
	protected function feriados_fill($year) {
		$year = (int)$year;
		if(!array_key_exists($year,$this->feriados))
			$this->feriados[$year]=array();
		foreach($this->feriados_define as $d) {
			//$iSlashes = substr_count($d['date'],'-');
			if($d['type'] == self::CADA_ANO)
				$this->feriados[$year][$year.'-'.$d['date']]=$d;
			elseif($d['type'] == self::N_DAY) {
				$weekday = array_val('diasem',$d,self::LUN);
				$num  = array_val('nday',$d,1);
				$tmp = explode('-',$d['date']);
				if(sizeof($tmp)==3)
					$best = $fecha = mktime(0,0,0,(int)$tmp[1],1,$year);
				else
					$best = $fecha = mktime(0,0,0,(int)$tmp[0],1,$year);
				$found=0;
				$iSanity=31;
				while( $found<$num && --$iSanity>0 ) {
					if(Date('w',$fecha)==$weekday) {
						$best=$fecha;
						if(++$found==$num)
							break;
					}
					$fecha += 86400;
				}
				$this->feriados[$year][Date('Y-m-d',$best)]=$d;
			} elseif($d['type'] == self::LA_FECHA && substr($d['date'],0,4)==$year)
				$this->feriados[$year][$d['date']]=$d;
		} // end foreach $this->feriados_define
	} // end function feriados_fill



}
?>