<?php declare(strict_types=1);

use Iac\inc\sql\IacSqlBuilder;
use JetBrains\PhpStorm\ArrayShape;


/**
 * Class Historian
 *
 * @usage
 *   $history = new Historian($gSqlClass, 'tabla');
 *
 *   $history->set('update', $values, $userNick);
 *
 *  $fieldChanged = $history->lastChange($valuesOrPrimaryKey);
 *
 *
 */
class Historian {
     protected array $ingoreDifferenceForFields = [
         'ultimo_cambio', 'ultimo_cambio_por',
         'last_changed', 'last_changed_by',
         'last_change', 'last_change_by',
         'inconsistencia_nota_bodega_id', 'alta_db', 'nota_bodega_item_id'
     ];
     protected string $jsonSqlType = 'JSON';
     protected string $table;
     protected string $tableHistory;
     protected array $pk;


    /**
     * Historian constructor.
     *
     * @param string $table
     * @param array $primaryKeyFieldNames
     */
     public function __construct(string $table, array $primaryKeyFieldNames = [] ) {

         $this->table = $table;
         $this->tableHistory = $table . '_hist';
         if(count($primaryKeyFieldNames) > 0)
             $this->pk = $primaryKeyFieldNames;
         else
             $this->pk = [$table . '_id'];
     }

////////////////////////////////////
     public function set(string $action, array $pk, array $values,  string $motive = '', string $user_nick = '') {
         $insertValues = [
             'action' => $action,
             'motive' => $motive,
             'pk' => $this->primaryKeyEncode($pk),
             'record' => json_encode($values, JSON_OPTIONS_FOR_MYSQL),
             'user_nick' => empty($user_nick) ? $_SESSION['usuario'] : $user_nick,
             'date' => 'NOW(6)'
         ];
         $builder = new IacSqlBuilder();
         $insertHistorySql = $builder->insert($this->tableHistory, $insertValues);
         try {
             if(!ia_query($insertHistorySql))
                return;
         } catch (Exception) { }
         $this-> historyTableCreate();
         try {
             ia_query($insertHistorySql);
         } catch (Exception) {}
     }

     #[ArrayShape(['insert' => "string", 'create_table' => "string"])]
     public function getInsertHistorian(string $action, array $pk, array $values, string $motive = '', bool $onlyInsert = true, string $user_nick = ''): array
     {
         $insertValues = [
             'action' => $action,
             'motive' => $motive,
             'pk' => $this->primaryKeyEncode($pk),
             'record' => json_encode($values, JSON_OPTIONS_FOR_MYSQL),
             'user_nick' => empty($user_nick) ? $_SESSION['usuario'] : $user_nick,
             'date' => 'NOW(6)'
         ];
         $builder = new IacSqlBuilder();
         $queries = ['insert' => $builder->insert($this->tableHistory, $insertValues)];
         if ($onlyInsert)
             return $queries;

         $queries['create_table'] = $this->getHistoryTableCreate();
         return $queries;
     }

 ///////// Get record history, indexed by history_id most recent to oldest

    /**
     *
     * @param array $primaryKeyOrValues
     * @param string $where
     * @param string $limitClause
     * @return array
     */
     public function get(array $primaryKeyOrValues, string $where = '', string $limitClause = ''):array {
         $pk = $this->primaryKeyEncode($primaryKeyOrValues);
         /** @noinspection SqlResolve */
         $sql = "SELECT history_id, `action`, `date`, `user_nick`, `record`, `motive` 
            FROM $this->tableHistory 
            WHERE (`pk` = '$pk') $where
            ORDER BY `date` DESC, history_id DESC $limitClause";

//         echo "<li>$sql";
         return $this->recordHistoryJsonDataToAssoc(
             ia_sqlArray(
                 $sql,
                 'history_id'
             )
         );
     }

     /**
      *
      * @param array $primaryKeyOrValues
      * @param int $numEntries
      * @return array

      */
     public function getLastN(array $primaryKeyOrValues, int $numEntries ):array {
         return $this->get($primaryKeyOrValues, '', "LIMIT $numEntries");
     }

     public function getDateRange(array $primaryKeyOrValues, string $fromYmdH, $toYmdH, $limit = '') {
         $where = " AND `date` BETEWEEN ";
         return $this->get($primaryKeyOrValues, $where, $limit);
     }

 /////////// Get record differences, indexed by history_id most recent to oldest
     /**
      * Regresa ultimos cambios [amount=1280.00], diferencias entre la ultima y penultima historia
      *
      * @param array $primaryKeyOrValues [cheque_id=>1] o [cheque_id=>1, name=>pedro,quantity=>1234.56]
      * @return array

      */
     public function getLastChange(array $primaryKeyOrValues) {
         return  $this->diff($this->getLastN($primaryKeyOrValues, 2) );
     }

    /**
     * All keys with changed value or keys in $after not in $before
     *
     * @param array $after
     * @param array $before
     * @return array
     */
     protected function afterBefore($after, $before) {
         $changed = [];
         foreach($after as $k => $v) {
             if(in_array($k, $this->ingoreDifferenceForFields))
                 continue;
             if(isset($before[$k])) {
                 if(is_array($v) && is_array($before[$k])) {
                     $changes = $this->afterBefore($v, $before[$k]);
                     if( !empty($changes)) {
                         $changed[$k] = $changes;
                     }
                 } elseif($v != $before[$k]) {
                     $changed[$k] = $v ." - ". $before[$k];
                 }
             } else {
                 $changed[$k] = $v;
             }
         }
         return $changed;
     }


     /**
      * Diferences between each history stored
      * $history->diff( $history->get(['id'=>1]) );
      *
      * @param array $recordHistory
      * @return array
      */
     public function diff(array $recordHistory):array {
         if(count($recordHistory) < 2)
             return [];
         $diff = [];
         $vals = array_values($recordHistory);
         for($i = 0, $len = count($vals); $i < $len; ++$i) {
             if($i === $len -1)
                 break;

             $h = $vals[$i];
             $diff[$i] = [
                 'history_id' => $h['history_id'],
                 'action' => $h['action'],
                 'date' => $h['date'],
                 'user_nick' => $h['user_nick'],
                 'record' => $i < $len -1 ?
                     $this->afterBefore($h['record'], $vals[$i+1]['record']) : $h['record'],
             ];
             if(empty($diff[$i]['record']))
                 unset($diff[$i]);
         }
         return $diff;
     }

//////////////////////////////////
    /**
     * @return array ['fieldNameToIgnore',...]
     */
    public function getIngoreDifferenceForFields(): array{return $this->ingoreDifferenceForFields;}

    /**
     * @param array $ingoreDifferenceForFields ['fieldNameToIgnore',...]
     */
    public function setIngoreDifferenceForFields(array $ingoreDifferenceForFields): void{
        $this->ingoreDifferenceForFields = $ingoreDifferenceForFields;
    }

///////////// Primary key for table //////////////////////////////////////////////////////////////////////////////////

    public function primaryKeyEncode(array $values):string {
        $pkValues = [];
        //if(!isset($this->pk))
        //     $this->pk =  $this->getPrimaryKeyFieldNames($this->table);
         foreach($this->pk as $pk)
                if(array_key_exists($pk, $values))
                    $pkValues[] = $values[$pk];
        return implode("\t" ,$pkValues);
    }

    /*
    protected function getPrimaryKeyFieldNames($table):array {
        try {
            $primaryKey =  $this->primaryKeyReadFromTable($table);
            if(count($primaryKey) > 0)
                return $primaryKey;
        } catch (IacSqlException) {
            $this->tablePrimaryKeyCreate();
        }
        try {
            $primaryKey = $this->primaryKeyReadFromInformationSchema($table);
        } catch (IacSqlException) {
            return [];
        }
        if(!empty($primaryKey)) {
            $this->primaryKeyWriteToTable($table, $primaryKey);
            return $primaryKey;
        }
        return [];
    }


    protected function primaryKeyReadFromTable(string $table):array|false {
        $method = __METHOD__;
        $primaryKey = ia_singleread(
            "SELECT  $method  pk FROM ia_table_primary_key WHERE tabla = " . strit($table)
        );
        return empty($primaryKey) ? [] : explode("\t", $primaryKey);
    }


    protected function primaryKeyReadFromInformationSchema(string $table):array|false {
        $tableProtected = "'$table'"; // Str::strit($table);
        $method = __METHOD__;
        $readPrimaryKeySql =
            "SELECT /* $method  sta.COLUMN_NAME
         FROM INFORMATION_SCHEMA.TABLES AS tab
            JOIN INFORMATION_SCHEMA.STATISTICS AS sta ON
                sta.TABLE_SCHEMA = tab.TABLE_SCHEMA AND sta.TABLE_NAME = tab.TABLE_NAME AND sta.INDEX_NAME = 'PRIMARY'
        WHERE tab.TABLE_SCHEMA = DATABASE() AND tab.TABLE_NAME = $tableProtected
        ORDER BY sta.seq_in_index";
        return ia_sqlVector($readPrimaryKeySql);
    }

     protected function primaryKeyWriteToTable(string $table, array $pk) {
         $builder = new IacSqlBuilder();
         try {
             ia_query(
                 $builder->insert('ia_table_primary_key', [
                     'tabla' => $table,
                     'pk' => implode("\t",$pk),
                     'registrado_el' => 'NOW(6)'
                 ], true)
             );
         } catch (Exception) {}
     }

     protected function tablePrimaryKeyCreate():bool {
         $method = __METHOD__;
         $createIaTableIndexes = "
        CREATE /* $method  TABLE IF NOT EXISTS ia_table_primary_key (
            tabla VARCHAR(128) NOT NULL PRIMARY KEY,
            pk VARCHAR(191) NOT NULL COMMENT 'Primary key, filed names tab separated',
            registrado_el DATETIME(6) NOT NULL
        )
      ";
         try {
             ia_query($createIaTableIndexes);
             return true;
         } catch (Exception) { return false;}
     }
*/
    /////////////////////////////////////

    public function recordHistoryJsonDataToAssoc($recordHistory) {
        foreach($recordHistory as &$d) {
            //  try {

                $d['record'] = json_decode($d['record'], true, 512,
                    JSON_BIGINT_AS_STRING | JSON_INVALID_UTF8_IGNORE | JSON_INVALID_UTF8_SUBSTITUTE
                   // | JSON_THROW_ON_ERROR,
                );

            //} catch (JsonException $_) {
            //  echo "\r\n__________ da __ERROR ___";
            //    $d['record'] = null;
            //}
        }

        return $recordHistory;
    }

//////////////////////////////////////////

    protected function historyTableCreate():bool {

        try {
            ia_query($this->getHistoryTableCreate());

            return true;
        } catch (Exception) { return false; }
    }

    protected function getHistoryTableCreate(): string
    {
        $method = __METHOD__;
        return "
        CREATE /* $method */ TABLE IF NOT EXISTS $this->tableHistory (
            `history_id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
            `action` VARCHAR(32) NOT NULL,
            `pk` VARCHAR(191) NOT NULL COMMENT 'filed names tab separated, same order as in definition',
            `motive` VARCHAR(191) NOT NULL DEFAULT '' COMMENT 'Optional short description of change',
            `record` $this->jsonSqlType,
            `user_nick` VARCHAR(32) COMMENT 'change by',
            `date` DATETIME(6) NOT NULL,
            KEY perRecord(pk, `date` DESC)
        ) ";
    }

    /**
     * Historian getNth.
     *
     * @param array $primaryKeyOrValues
     * @param int $numEntry
     */

    public function getNth(array $primaryKeyOrValues, int $numEntry ):array {
        $offSet = $numEntry <= 1 ? 0 : $numEntry -1;
        $nth = $this->get($primaryKeyOrValues, '', "LIMIT $offSet,1");
        foreach($nth as $n)
            return $n;
        return [];
    }


    /**
     * Diferences between each history stored
     *
     * @param array $recordHistory
     * @param string $tipo
     * @return array
     */
    public function getdiff(array $recordHistory, string $tipo = ''):array
    {
        //if(count($recordHistory) < 2)
        //    return [];

        switch($tipo) {
            case 'vale':
            case 'pagare':
            case 'cheque':
                $campos["no_importan"] = ['ultimo_cambio', 'abierto_el', 'abierto_por', 'categoria_id', 'cuentat_mov_id', 'remarks_wot', 'html_verified'];
                $campos["to_decode"] = ['pago', 'pago_log','uso', 'uso_log'];
                break;
            case 'cliente':
                $campos["no_importan"] = [];
                $campos["to_decode"] = ['origen_bodega'];
                break;
            case 'compra':
                $campos["no_importan"] = ['ultimo_cambio', 'html_verified'];
                $campos["to_decode"] = ['pago','pago_log'];
                break;
            default:
                $campos["no_importan"] = [];
                $campos["to_decode"] = [];
        }

        $diff = [];
        foreach($recordHistory as $key => $value)
        {

            $diff[$key] = [
                'history_id' => $value['history_id'],
                'action' => $value['action'],
                'date' => $value['date'],
                'user_nick' => $value['user_nick'],
            ];

            if ($key === 0) {
                $diff[$key]['record'] = $value['action'];
                $diff[$key]['time_lapse'] = '';
                continue;
            }

            $nota_old = json_decode($recordHistory[$key - 1]["record"], true);
            $nota_nueva = json_decode($value["record"], true);

            $diff[$key]['time_lapse'] = fechaDiff($recordHistory[0]["date"], $value['date']);
            $diff[$key]['record'] = $this->antesDespues($nota_old, $nota_nueva, $campos);

        }
        return $diff;
    }


    /**
     *
     * @param array $despues
     * @param array $antes
     * @param array $campos
     * @return array
     */
    protected function antesDespues(array $antes, array $despues,  array $campos = []):array
    {
        $changed = [];
        foreach($despues as $k => $v)
        {
            if(in_array($k, $campos["no_importan"] ?? []))
                continue;

            if(in_array($k, $campos["to_decode"] ?? []))
            {
                if( empty($antes[$k]) && !empty($v) )
                    $changed[$k]["agregado"] = json_decode($v, true);

                if( !empty($antes[$k]) && !empty($v) )
                {
                    $prev_items = json_decode($antes[$k], true);
                    $next_items = json_decode($v, true);

                    foreach( $next_items as $key => $value )
                        if( !array_key_exists($key, $prev_items))
                            $changed[$k]["agregado"][$key] = $value;

                    foreach( $prev_items as $llave => $valor )
                        if( !array_key_exists($llave, $next_items ) )
                            $changed[$k]["eliminado"][$llave] = $valor;

                }
                continue;
            }

            if(array_key_exists($k, $antes) && $v != $antes[$k] ?? '')
                $changed[$k] = strip_tags($antes[$k] ?? '') ." - ". strip_tags($v ?? '');
        }

        return $changed;
    }



 }
