<?php
/**
 * IaColor - PHP implementation of iaColor JavaScript utility
 * 
 * This class provides color manipulation utilities, particularly for determining
 * appropriate text colors based on background colors to ensure readability.
 */
class IaColor {
    /**
     * Creates a CSS gradient string
     *
     * @param array $colors Colors for the gradient ['FF0000', '444', '#999 20%', '10%', '#00FF33']
     * @param string $gradient_type Type of gradient (linear-gradient, radial-gradient, etc.)
     * @param string $direction Direction of the gradient (to bottom, to right, etc.)
     * @return string CSS gradient string
     */
    public function gradient($colors, $gradient_type = 'linear-gradient', $direction = '') {
        if (empty($direction)) {
            $direction = $gradient_type === 'linear-gradient' ? 'to bottom' : 'circle';
        }
        return $gradient_type . '(' . (empty($direction) ? '' : $direction . ',') . implode(',', $colors) . ');';
    }

    /**
     * Recommends a text color based on background colors
     *
     * @param array $gradientColors Array of background colors
     * @param mixed $textColor Preferred text color(s) or null to auto-select
     * @param float $threshold Minimum contrast threshold (default 2.5, recommended 4)
     * @param bool $test Whether to output test information
     * @return string Hex color code with # prefix
     */
    public function recomendTextColor($gradientColors, $textColor = null, $threshold = 2.5, $test = false) {
        $colors = $this->extractColors($gradientColors);
        $colorsLen = count($colors);
        
        if ($test) {
            // Test output would go here in a real implementation
        }
        
        // 1. If the requested color passes the threshold, use it
        if (is_string($textColor) && $textColor !== '') {
            $textColor = $this->hexFix($textColor);
            if ($this->goodContrast($textColor, $colors, $threshold, $test)) {
                return '#' . $textColor;
            }
        } elseif (is_array($textColor) && $textColor !== null) {
            foreach ($textColor as $color) {
                $color = $this->hexFix($color);
                if ($this->goodContrast($color, $colors, $threshold, $test)) {
                    return '#' . $color;
                }
            }
        }
        
        // 2. Find the best color
        $candidateColors = [
            'FFFFFF', 'B0B0B0', '808080', 'A9A9A9',
            'FFCC00', 'CCFF00', 'FFFF00', '00FF33',
            'FF0000', '660000'
        ];
        
        for ($i = 0; $i < $colorsLen; $i++) {
            $candidateColors[] = $this->invertColor($colors[$i], false);
        }
        
        if (is_string($textColor) && $textColor !== '') {
            $candidateColors[] = $textColor;
            $candidateColors[] = $this->invertColor($textColor, false);
        } elseif (is_array($textColor) && $textColor !== null) {
            foreach ($textColor as $color) {
                $candidateColors[] = $color;
                $candidateColors[] = $this->invertColor($color, false);
            }
        }
        
        $candidateColorsLen = count($candidateColors);
        $colorThreshold = [];
        
        for ($iCandidate = 0; $iCandidate < $candidateColorsLen; $iCandidate++) {
            $checkColor = $candidateColors[$iCandidate];
            $colorThreshold[$checkColor] = ['min' => 20, 'max' => 0];
            
            for ($i = 0; $i < $colorsLen; $i++) {
                $contrast = $this->contrast($checkColor, $colors[$i]);
                if ($contrast < $colorThreshold[$checkColor]['min']) {
                    $colorThreshold[$checkColor]['min'] = $contrast;
                }
                if ($contrast > $colorThreshold[$checkColor]['max']) {
                    $colorThreshold[$checkColor]['max'] = $contrast;
                }
                if ($contrast < 1.2) {
                    continue;
                }
            }
        }
        
        $bestColor = '';
        $bestContrast = 0.00;
        
        foreach ($colorThreshold as $color => $values) {
            if ($bestContrast <= $values['min']) {
                $bestContrast = $values['min'];
                $bestColor = $color;
            }
        }
        
        return '#' . $bestColor;
    }
    
    /**
     * Extracts colors from gradient definitions
     *
     * @param array $colors Array of colors, possibly with percentages
     * @return array Array of standardized hex colors
     */
    private function extractColors($colors) {
        if ($colors === null) {
            return [$this->hexFix('#FFFFFF')];
        }
        
        $useColors = [];
        foreach ($colors as $color) {
            $c = explode(' ', $color);
            if (strpos($c[0], '%') !== false) {
                continue;
            }
            $useColors[] = $this->hexFix($c[0]);
        }
        
        return $useColors;
    }
    
    /**
     * Checks if a color has good contrast with all background colors
     *
     * @param string $checkColor Color to check
     * @param array $colors Background colors to check against
     * @param float $threshold Minimum contrast threshold
     * @param bool $test Whether to output test information
     * @return bool True if all contrasts are above threshold
     */
    private function goodContrast($checkColor, $colors, $threshold, $test = false) {
        foreach ($colors as $color) {
            if ($this->contrast($checkColor, $color) < $threshold) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Normalizes hex color format
     *
     * @param string $hex Hex color in various formats
     * @return string Standardized hex color without # prefix
     * @throws Exception If invalid hex color
     */
    public function hexFix($hex) {
        $hex = trim($hex);
        
        if (strpos($hex, 'r') === 0) {
            $hex = $this->rgb2hex($hex);
        }
        
        if (strpos($hex, '#') === 0) {
            $hex = substr($hex, 1);
        }
        
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        
        if (strlen($hex) !== 6 && strlen($hex) !== 8) {
            $hex = "000000";
            // throw new Exception('Invalid HEX color: ' . $hex);
        }
        
        return $hex;
    }
    
    /**
     * Converts hex color to RGB array
     *
     * @param string $hex Hex color
     * @return array RGB values [r, g, b]
     */
    public function hex2rgb($hex) {
        $hex = $this->hexFix($hex);
        return [
            hexdec(substr($hex, 0, 2)),
            hexdec(substr($hex, 2, 2)),
            hexdec(substr($hex, 4, 2))
        ];
    }
    
    /**
     * Converts RGB color to hex
     *
     * @param string $cssRgb CSS RGB string like "rgb(0, 255, 0)"
     * @return string Hex color with # prefix
     */
    public function rgb2hex($cssRgb) {
        preg_match('/^rgba?\((\d+),\s*(\d+),\s*(\d+)(?:,\s*([^,\s\)]+))?\)/i', $cssRgb, $rgb);
        
        if (!$rgb) {
            return $cssRgb;
        }
        
        $hex = sprintf("%02x%02x%02x", $rgb[1], $rgb[2], $rgb[3]);
        return '#' . $hex;
    }
    
    /**
     * Calculates color luminance
     *
     * @param int $r Red component (0-255)
     * @param int $g Green component (0-255)
     * @param int $b Blue component (0-255)
     * @return float Luminance value
     */
    public function luminanace($r, $g, $b) {
        $a = [$r, $g, $b];
        
        for ($i = 0; $i < 3; $i++) {
            $a[$i] /= 255;
            $a[$i] = $a[$i] <= 0.03928 
                ? $a[$i] / 12.92 
                : pow(($a[$i] + 0.055) / 1.055, 2.4);
        }
        
        return $a[0] * 0.2126 + $a[1] * 0.7152 + $a[2] * 0.0722;
    }
    
    /**
     * Calculates contrast ratio between two colors
     *
     * @param mixed $rgb1 RGB array or hex string
     * @param mixed $rgb2 RGB array or hex string
     * @return float Contrast ratio
     */
    public function contrast($rgb1, $rgb2) {
        if (is_string($rgb1)) {
            $rgb1 = $this->hex2rgb($rgb1);
        }
        
        if (is_string($rgb2)) {
            $rgb2 = $this->hex2rgb($rgb2);
        }
        
        $lum1 = $this->luminanace($rgb1[0], $rgb1[1], $rgb1[2]);
        $lum2 = $this->luminanace($rgb2[0], $rgb2[1], $rgb2[2]);
        $brightest = max($lum1, $lum2);
        $darkest = min($lum1, $lum2);
        
        return ($brightest + 0.05) / ($darkest + 0.05);
    }
    
    /**
     * Inverts a color or chooses black/white based on brightness
     *
     * @param string $hex Hex color
     * @param bool $bw Whether to choose black/white (true) or invert (false)
     * @return string Inverted hex color without # prefix
     */
    public function invertColor($hex, $bw = false) {
        $hex = $this->hexFix($hex);
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        
        if ($bw) {
            return ($r * 0.299 + $g * 0.587 + $b * 0.114) > 186 
                ? '000000' 
                : 'FFFFFF';
        }
        
        // Invert color components
        $r = (255 - $r);
        $g = (255 - $g);
        $b = (255 - $b);
        
        // Convert to hex and pad with zeros
        return $this->padZero(dechex($r)) . $this->padZero(dechex($g)) . $this->padZero(dechex($b));
    }
    
    /**
     * Pads a string with zeros
     *
     * @param string $str String to pad
     * @param int $len Length to pad to
     * @return string Zero-padded string
     */
    public function padZero($str, $len = 2) {
        return str_pad($str, $len, '0', STR_PAD_LEFT);
    }
    
    /**
     * Gets the recommended text color for a background color
     * 
     * @param string|array $backgroundColor Background color(s)
     * @param mixed $textColor Preferred text color(s) or null
     * @param float $threshold Minimum contrast threshold
     * @return string Hex color with # prefix
     */
    public function getTextColor($backgroundColor, $textColor = null, $threshold = 4) {
        if (!is_array($backgroundColor)) {
            $backgroundColor = [$backgroundColor];
        }
        
        return $this->recomendTextColor($backgroundColor, $textColor, $threshold);
    }
}
?>