<?php

final /*static*/ class InitialsFilter {

    public static function filter(array &$data, array $initials, string|int $campo = 'label'):array {
        foreach($initials as &$t)
            $t = strtoupper(self::unAccent(trim($t)));
        unset($t);
        self::matchPriority($data, $initials, $campo);
        uasort($data, function($rowA, $rowB) use ($campo, $initials) {
            $a = strtoupper(self::unAccent($rowA[$campo]));
            $b = strtoupper(self::unAccent($rowB[$campo]));

            $starts =  reset($initials);
            $aStarts = str_starts_with($a, $starts);
            $bStarts = str_starts_with($b, $starts);

            if($aStarts && $bStarts) {
                $matchPriority = $rowB['matchPriority'] - $rowA['matchPriority'];
                if($matchPriority)
                    return $matchPriority;
                $len = strlen($a) <=> strlen($b);
                if($len)
                    return $len;
                return $a <=> $b;
            }
            if($aStarts)
                return -1;
            if($bStarts)
                return 1;
            $matchesCount = $rowB['matches'] - $rowA['matches'];
            if($matchesCount)
                return $matchesCount;
            $matchPriority = $rowB['matchPriority'] - $rowA['matchPriority'];
            if($matchPriority)
                return $matchPriority;
            return $a <=> $b;
        });
        return $data;
    }

    protected static function filterRegExp($initials):string {
        $re = [];
        foreach($initials as $t)
            $re[] = "\\b" . preg_quote($t, "/");
        return '/' . implode('|', $re) . '/iu';
    }


    protected static function matchPriority(array &$data, array $initials, string|int $campo = 'label'):void {
        $regExp = self::filterRegExp($initials);
        foreach($data as &$d) {
            try {$d['matches']= preg_match_all($regExp, self::unAccent($d[$campo]), $matches);} catch(Throwable) {continue;}
            if(isset($matches[0])) {
                $d['matchPriority'] = 0;
                $wds = preg_split("/\b/", $d[$campo]);
                foreach($wds as $i => $w)
                    if($w !== '' && $w !== '-')
                        if(preg_match($regExp, $w)){
                        $d['matchPriority'] += 1000 - $i;
                    }
            } else
                $d['matchPriority'] = 0;
        }
    }

    /**
     * Removes diacritics leaving the plain letter
     *
     * @param string $text The input text which needs to have diacritics removed.
     * @return string The processed text with diacritics removed.
     */
    protected static function unAccent(string $text):string {
        $transliterationMap = [
            // Common Latin characters
          'À' => 'A', 'Á' => 'A', 'Â' => 'A', 'Ã' => 'A', 'Ä' => 'A', 'Å' => 'A', 'Ā' => 'A', 'Ă' => 'A', 'Ą' => 'A',
          'à' => 'a', 'á' => 'a', 'â' => 'a', 'ã' => 'a', 'ä' => 'a', 'å' => 'a', 'ā' => 'a', 'ă' => 'a', 'ą' => 'a',
          'Ç' => 'C', 'Ć' => 'C', 'Ĉ' => 'C', 'Ċ' => 'C', 'Č' => 'C', 'ç' => 'c', 'ć' => 'c', 'ĉ' => 'c', 'ċ' => 'c', 'č' => 'c',
          'Ð' => 'D', 'Ď' => 'D', 'Đ' => 'D', 'ð' => 'd', 'ď' => 'd', 'đ' => 'd',
          'È' => 'E', 'É' => 'E', 'Ê' => 'E', 'Ë' => 'E', 'Ē' => 'E', 'Ĕ' => 'E', 'Ė' => 'E', 'Ę' => 'E', 'Ě' => 'E',
          'è' => 'e', 'é' => 'e', 'ê' => 'e', 'ë' => 'e', 'ē' => 'e', 'ĕ' => 'e', 'ė' => 'e', 'ę' => 'e', 'ě' => 'e',
          'Ĝ' => 'G', 'Ğ' => 'G', 'Ġ' => 'G', 'Ģ' => 'G', 'ĝ' => 'g', 'ğ' => 'g', 'ġ' => 'g', 'ģ' => 'g',
          'Ĥ' => 'H', 'Ħ' => 'H', 'ĥ' => 'h', 'ħ' => 'h',
          'Ì' => 'I', 'Í' => 'I', 'Î' => 'I', 'Ï' => 'I', 'Ĩ' => 'I', 'Ī' => 'I', 'Ĭ' => 'I', 'Į' => 'I', 'İ' => 'I',
          'ì' => 'i', 'í' => 'i', 'î' => 'i', 'ï' => 'i', 'ĩ' => 'i', 'ī' => 'i', 'ĭ' => 'i', 'į' => 'i', 'ı' => 'i',
          'Ĵ' => 'J', 'ĵ' => 'j',
          'Ķ' => 'K', 'ķ' => 'k',
          'Ĺ' => 'L', 'Ļ' => 'L', 'Ľ' => 'L', 'Ŀ' => 'L', 'Ł' => 'L', 'ĺ' => 'l', 'ļ' => 'l', 'ľ' => 'l', 'ŀ' => 'l', 'ł' => 'l',
          'Ñ' => 'N', 'Ń' => 'N', 'Ņ' => 'N', 'Ň' => 'N', 'ñ' => 'n', 'ń' => 'n', 'ņ' => 'n', 'ň' => 'n',
          'Ò' => 'O', 'Ó' => 'O', 'Ô' => 'O', 'Õ' => 'O', 'Ö' => 'O', 'Ø' => 'O', 'Ō' => 'O', 'Ŏ' => 'O', 'Ő' => 'O',
          'ò' => 'o', 'ó' => 'o', 'ô' => 'o', 'õ' => 'o', 'ö' => 'o', 'ø' => 'o', 'ō' => 'o', 'ŏ' => 'o', 'ő' => 'o',
          'Ŕ' => 'R', 'Ŗ' => 'R', 'Ř' => 'R', 'ŕ' => 'r', 'ŗ' => 'r', 'ř' => 'r',
          'Ş' => 'S', 'Ș' => 'S', 'Ś' => 'S', 'Š' => 'S', 'ş' => 's', 'ș' => 's', 'ś' => 's', 'š' => 's',
          'Þ' => 'TH', 'þ' => 'th', // Icelandic
          'Ù' => 'U', 'Ú' => 'U', 'Û' => 'U', 'Ü' => 'U', 'Ũ' => 'U', 'Ū' => 'U', 'Ŭ' => 'U', 'Ů' => 'U', 'Ű' => 'U', 'Ų' => 'U',
          'ù' => 'u', 'ú' => 'u', 'û' => 'u', 'ü' => 'u', 'ũ' => 'u', 'ū' => 'u', 'ŭ' => 'u', 'ů' => 'u', 'ű' => 'u', 'ų' => 'u',
          'Ý' => 'Y', 'Ÿ' => 'Y', 'Ŷ' => 'Y', 'ý' => 'y', 'ÿ' => 'y', 'ŷ' => 'y',
          'Ź' => 'Z', 'Ż' => 'Z', 'Ž' => 'Z', 'ź' => 'z', 'ż' => 'z', 'ž' => 'z',

          'ß' => 'ss', // German sharp S
          'Œ' => 'OE', 'œ' => 'oe', // French ligature
          'Æ' => 'AE', 'æ' => 'ae', // Danish/Norwegian ligature


            // Additional special cases for B and b
          'ƀ' => 'b', 'Ɓ' => 'B', 'Ƃ' => 'B', 'ƃ' => 'b',
          'ʙ' => 'B' // IPA symbol
        ];

        return strtr($text, $transliterationMap);
    }

    public static function sqlWhere(array $initials):string {
        if(empty($initials))
            return '';
        foreach($initials as &$t)
            $t = strtoupper(self::unAccent(trim($t)));
        unset($t);
        $wheres = [];
        $regExp = self::toDiacriticsRegExp($initials);
        foreach($initials as $k => $t) {
            if(count($initials) === 1 and strlen($t) < 3)
                $wheres[] = "nombre REGEXP " . strit("^" . $regExp[$k]);
            else
                $wheres[] = "nombre REGEXP " . strit("\b" . $regExp[$k]);
        }

        return empty($wheres) ? '' : '(' . join(' AND ', $wheres) . ')';
    }

    protected static function toDiacriticsRegExp(array $initials):array {
        $regExp = [];
        foreach($initials as $t)
            $regExp[] = str_replace(
              ['A','E','I','O','U'],
              ['[AÁÄÂÀ]','[EÉËÊÈ]','[IÍÏÎÌ]','[OÓÖÔÒ]','[UÚÜÛÙ]'],
              preg_quote( strtoupper(self::unAccent($t)))
            );
        return $regExp;
    }

}
