<?php

class TimeSeriesGraph {
    private int $width;
    private int $height;
    private array $series1;
    private array $series2;
    private $image;
    private array $colors;
    private array $padding = [60, 50, 60, 60]; // left, top, right, bottom
    private bool $twoScales;
    private string $series1Label;
    private string $series2Label;

    public function __construct(
      array $series1,
      string $series1Label = "",
      array $series2 = [],
      string $series2Label = "",
      int $width = 800,
      int $height = 400,
      bool $twoScales = false
    ) {
        $this->series1Label = $series1Label;
        $this->series2Label = $series2Label;
        $this->twoScales = $twoScales;
        $this->width = $width;
        $this->height = $height;
        $this->series1 = $this->fillMissingValues($series1);
        $this->series2 = $this->fillMissingValues($series2);

        // Initialize the image
        $this->image = imagecreatetruecolor($width, $height);

        // Set background to white
        $white = imagecolorallocate($this->image, 255, 255, 255);
        imagefill($this->image, 0, 0, $white);

        // Define colors
        $this->colors = [
          'grid' => imagecolorallocate($this->image, 220, 220, 220),
          'axis' => imagecolorallocate($this->image, 50, 50, 50),
          'text' => imagecolorallocate($this->image, 0, 0, 0),
          'line1' => imagecolorallocate($this->image, 255, 0, 0),
          'line2' => imagecolorallocate($this->image, 0, 0, 255)
        ];
    }

    private function fillMissingValues(array $data): array {
        // Get all dates from both series to ensure we cover the full range
        $allDates = array_keys($data);
        if (empty($allDates)) {
            return [];
        }

        // Sort dates and get range
        sort($allDates);
        $startDate = new DateTime(reset($allDates));
        $endDate = new DateTime(end($allDates));

        // Create array with all dates in range
        $filled = [];
        $currentDate = clone $startDate;
        $lastValue = null;

        while ($currentDate <= $endDate) {
            $dateStr = $currentDate->format('Y-m-d');

            // If we have a value for this date, use it
            if (isset($data[$dateStr])) {
                if ($data[$dateStr] !== null) {
                    $lastValue = $data[$dateStr];
                    $filled[$dateStr] = $lastValue;
                } elseif ($lastValue !== null) {
                    $filled[$dateStr] = $lastValue;
                }
            } elseif ($lastValue !== null) {
                // Date completely missing from array, use last value if we have one
                $filled[$dateStr] = $lastValue;
            }

            $currentDate->modify('+1 day');
        }

        return $filled;
    }

    private function getYAxisBounds(array|null $values = null): array {
        if ($values === null) {
            $values = array_merge(
              array_values($this->series1),
              array_values($this->series2)
            );
        }

        $min = empty($values) ? 0:  min($values);
        $max = empty($values) ? 0:  max($values);

        // Add 5% padding
        $range = $max - $min;
        $padding = $range * 0.05;

        return [
          'min' => $min - $padding,
          'max' => $max + $padding
        ];
    }

    private function mapToPixel($value, $minValue, $maxValue, $pixelMin, $pixelMax): int {
        $range = $maxValue - $minValue;
        $pixelRange = $pixelMax - $pixelMin;
        if(empty($range))
            return intval($pixelMin + ($value - $minValue) * $pixelRange);
        return intval($pixelMin + ($value - $minValue) * $pixelRange / $range);
    }

    public function render(): GdImage {
        $plotWidth = $this->width - $this->padding[0] - $this->padding[2];
        $plotHeight = $this->height - $this->padding[1] - $this->padding[3];
        $this->drawTitle();
        if ($this->twoScales) {
            $bounds1 = $this->getYAxisBounds(array_values($this->series1));
            $bounds2 = $this->getYAxisBounds(array_values($this->series2));

            // Draw grid based on left axis only
            $this->drawGrid($bounds1, $plotWidth, $plotHeight, $this->twoScales);

            // Draw right axis
            $this->drawRightAxis($bounds2, $plotWidth, $plotHeight);

            // Draw data lines with their respective scales
            $this->drawDataLine($this->series1, $bounds1, $plotWidth, $plotHeight, $this->colors['line1']);
            $this->drawDataLine($this->series2, $bounds2, $plotWidth, $plotHeight, $this->colors['line2']);
        } else {
            $bounds = $this->getYAxisBounds();

            // Draw grid and axes
            $this->drawGrid($bounds, $plotWidth, $plotHeight, $this->twoScales);

            // Draw data lines
            $this->drawDataLine($this->series1, $bounds, $plotWidth, $plotHeight, $this->colors['line1']);
            $this->drawDataLine($this->series2, $bounds, $plotWidth, $plotHeight, $this->colors['line2']);
        }

        return $this->image;
    }

    private function drawTitle(): void {
        if (!empty($this->series1Label) || !empty($this->series2Label)) {
            if (!empty($this->series1Label)) {
                imagestring($this->image, 3,
                  intval($this->width * 0.3), 10,
                  $this->series1Label, $this->colors['line1']);
            }

            if (!empty($this->series2Label)) {
                imagestring($this->image, 3,
                  intval($this->width * 0.6), 10,
                  $this->series2Label, $this->colors['line2']);
            }
        }
    }

    private function drawGrid($bounds, $plotWidth, $plotHeight, bool $leftAxisOnly): void {
        // Draw Y-axis grid lines and labels
        $yStep = ($bounds['max'] - $bounds['min']) / 5;
        for ($i = 0; $i <= 5; $i++) {
            $y = $bounds['min'] + ($i * $yStep);
            $pixelY = $this->mapToPixel($y, $bounds['min'], $bounds['max'],
              $this->height - $this->padding[3], $this->padding[1]);

            // Grid line
            imageline($this->image,
              $this->padding[0], $pixelY,
              $this->width - $this->padding[2], $pixelY,
              $this->colors['grid']);

            // Label
            $label = number_format($y, 2);
            imagestring($this->image, 2,
              $this->padding[0] - 45, $pixelY - 6,
              $label, $leftAxisOnly ? $this->colors['line1'] : $this->colors['text']);
        }

        // Draw left Y-axis line
        imageline($this->image,
          $this->padding[0], $this->padding[1],
          $this->padding[0], $this->height - $this->padding[3],
          $leftAxisOnly ? $this->colors['line1'] : $this->colors['axis']);

        // Draw X-axis grid lines and labels
        $dates = array_keys($this->series1);
        $dateCount = count($dates);
        $xStep = max(1, (int)($dateCount / 5));

        for ($i = 0; $i < $dateCount; $i += $xStep) {
            $pixelX = intval($this->padding[0] + ($i * $plotWidth / ($dateCount - 1)));

            // Grid line
            imageline($this->image,
              $pixelX, $this->padding[1],
              $pixelX, $this->height - $this->padding[3],
              $this->colors['grid']);

            // Label
            $date = new DateTime($dates[$i]);
            $label = $date->format('Y-m-d');
            imageline($this->image,
              $pixelX, $this->height - $this->padding[3],
              $pixelX, $this->height - $this->padding[3] + 5,
              $this->colors['axis']);

            imagestring($this->image, 2,
              $pixelX - 20, $this->height - $this->padding[3] + 10,
              $label, $this->colors['text']);
        }
    }

    private function drawRightAxis($bounds, $plotWidth, $plotHeight): void {
        // Draw Y-axis labels on the right side
        $yStep = ($bounds['max'] - $bounds['min']) / 5;
        for ($i = 0; $i <= 5; $i++) {
            $y = $bounds['min'] + ($i * $yStep);
            $pixelY = $this->mapToPixel($y, $bounds['min'], $bounds['max'],
              $this->height - $this->padding[3], $this->padding[1]);

            // Label on right side
            $label = number_format($y, 2);
            imagestring($this->image, 2,
              $this->width - $this->padding[2] + 5, $pixelY - 6,
              $label, $this->colors['line2']);
        }

        // Draw right Y-axis line
        imageline($this->image,
          $this->width - $this->padding[2], $this->padding[1],
          $this->width - $this->padding[2], $this->height - $this->padding[3],
          $this->colors['line2']);
    }

    private function drawDataLine(array $data, array $bounds, int $plotWidth, int $plotHeight, int $color): void {
        $dates = array_keys($data);
        $dateCount = count($dates);
        $lastX = null;
        $lastY = null;

        for ($i = 0; $i < $dateCount; $i++) {
            $value = $data[$dates[$i]];
            if ($value === null) continue;

            $pixelX = intval($this->padding[0] + ($i * $plotWidth / ($dateCount - 1)));
            $pixelY = $this->mapToPixel($value, $bounds['min'], $bounds['max'],
              $this->height - $this->padding[3], $this->padding[1]);

            if ($lastX !== null && $lastY !== null) {
                imageline($this->image, $lastX, $lastY, $pixelX, $pixelY, $color);
            }

            $lastX = $pixelX;
            $lastY = $pixelY;
        }
    }

    public function __destruct() {
        if ($this->image) {
            imagedestroy($this->image);
        }
    }
}