<?php
/**
 * JqGridColModel.php generate a jqGrid colmodel from columns as generated by ia\Sql\Mysql\Columns
 * by default uses col templates from iaJqGrid.colTemplate
 * Output with method colModel2Json
 *
 * @package ia\JqGrid
 */

/** @noinspection SqlNoDataSourceInspection */
/** @noinspection SqlResolve */

namespace ia\JqGrid;

use ia\Lib\iaPalabra;
use ia\Sql\Mysql\Columns;
use ia\Sql\Mysql\IacSqlException;
use ia\Util\iaArray;
use \ia\Util\Str;
use Zend\Json\Expr;
use Zend\Json\Json;

/**
 * Class JqGridColModel generate a jqGrid colmodel from columns as generated by ia\Sql\Mysql\Columns
 * by default uses col templates from iaJqGrid.colTemplate
 * Output with method colModel2Json
 *
 * $modeler = new JqGridColModel();
 * $colModelJson = $modeler->colModelJsonFromTable($gSqlClass, 'tableName');
 * o
 * $colModelJson = $modeler->colModelJsonFromQuery($gSqlClass, 'SELECT ...');
 *
 * manually:
 *
 * $columnGet = new \ia\Sql\Mysql\Columns($gSqlClass);
 * $columns = $columnGet->columnsForTable('tableName'); // or $columnGet->columnsForQuery('SELECT ...')
 * $modeler = new JqGridColModel();
 * $colModelArray = $modeler->colModelArray($columns, JqGridColModel::TABLE_ALIAS);
 * $colModelJson = $modeler->colModelArray2Json($colModelArray);
 *
 * @package ia\JqGrid
 */
class JqGridColModel {
    /** @var int for colModel.index property add table (table) alias if available if not original table (orgtable) */
    const TABLE_ALIAS = 1;

    /** @var int for colModel.index property add original table (orgtable) name if available */
    const TABLE_NAME = 2;

    /** @var int for colModel.index property don't add table name */
    const TABLE_NONE = 3;

    protected $colTemplates = [
        'int' => 'iaJqGrid.colTemplate.ct_int',
        'float' => 'iaJqGrid.colTemplate.ct_float',

        'date' => 'iaJqGrid.colTemplate.ct_date',
        'datetime' => 'iaJqGrid.colTemplate.ct_dateTime',
        'timestamp' => 'iaJqGrid.colTemplate.ct_timestamp',
        'year' => 'iaJqGrid.colTemplate.ct_year',
        'yearmonth' => 'iaJqGrid.colTemplate.ct_yearMonth',
        'daymonth' => 'iaJqGrid.colTemplate.ct_dayMonth',

        'varchar' => 'iaJqGrid.colTemplate.ct_varchar',
        'charsmall' => 'iaJqGrid.colTemplate.ct_charSmall',
        'char' => 'iaJqGrid.colTemplate.ct_char',
        'text' => 'iaJqGrid.colTemplate.ct_text',

        'enumSetAuto' => 'iaJqGrid.colTemplate.ct_enumSetAuto',
        'enum' => 'iaJqGrid.colTemplate.ct_enum',
        'set' => 'iaJqGrid.colTemplate.ct_set',

        'money' => 'iaJqGrid.colTemplate.ct_money',
        'tipoCambio' => 'iaJqGrid.colTemplate.ct_exchangeRate',
        'exchangeRate' => 'iaJqGrid.colTemplate.ct_exchangeRate',
        'percent' => 'iaJqGrid.colTemplate.ct_percent',
        'percent_times_100' => 'iaJqGrid.colTemplate.ct_percent_times_100',

        'email' => 'iaJqGrid.colTemplate.ct_email',
        'link' => 'iaJqGrid.colTemplate.link',
        'showlink' => 'iaJqGrid.colTemplate.showlink',
    ];

    protected $actionsCol = [
        'name' => 'act',
        'label' => '&nbsp;',
        'formatter' => 'actions',
        'wdith' => 45,
        'editable' => false,
        'editrules' => ['edithidden'=>true],
        'editoptions' => ['readonly'=>null, 'display' => 'none'],
        'formoptions' => ['display'=>'none'],
        'formatoptions' => [],
        'search' => false,
        'sortable' => false,
    ];

    /**
     * 
     * 
     * JqGridColModel constructor.
     * @param array? $colTemplates Names for colModel column templates
     */
    public function __construct($colTemplates = null) {
        if(!empty($colTemplates)) {
            $this->colTemplates = $colTemplates;
        }
    }

    /** functionality */

    /**
     * A json colModel for $table
     *
     * @param $db
     * @param string $table
     * @param string $alias
     * @param int $indexUseTable JqGridColModel::TABLE_NAME o TABLE_ALIAS o TABLE_NONE
     * @param bool $forceReadonly
     * @return string json colModel for $table
     * @throws IacSqlException
     * @noinspection PhpUnused
     */
    public function colModelJsonFromTable($db, $table, $alias = '',  $indexUseTable = self::TABLE_ALIAS, $forceReadonly = false) {
        $colInfo = new Columns($db);
        return $this->colModelArray2Json(
            $this->colModelArray( $colInfo->columnsForTable($table, $alias), 
              $indexUseTable, $forceReadonly)
        );
    }

    /**
     * A json colModel for $query
     *
     * @param $db
     * @param $query
     * @param int $indexUseTable JqGridColModel::TABLE_NAME o TABLE_ALIAS o TABLE_NONE
     * @param bool $forceReadonly
     * @return string json colModel for $query
     * @throws IacSqlException
     * @noinspection PhpUnused
     */
    public function colModelJsonFromForQuery($db, $query,  $indexUseTable = self::TABLE_ALIAS, $forceReadonly = false) {
        $colInfo = new Columns($db);
       return $this->colModelArray2Json(
           $this->colModelArray( $colInfo->columnsForQuery($query), 
             $indexUseTable, $forceReadonly)
       );
    }

    /**
     * Returns a jqGrid colModel php Array from columns as generated by ia\Sql\Mysql\Columns
     * If $column[]['colModel'] exists it is merged, preserving its values.
     * by default uses col templates from iaJqGrid.colTemplate output with method colModel2Json
     *
     * @param array $columns an array of columns as generated by ia\Sql\Mysql\Columns
     * @param int $indexUseTable JqGridColModel::TABLE_NAME o TABLE_ALIAS o TABLE_NONE
     * @param bool $forceReadonly
     * @return array
     * @noinspection PhpUnused
     */
    public function colModelArray($columns, $indexUseTable = self::TABLE_NAME, $forceReadonly = false) {
        $colModel = [ 'act' => $this->actionsCol ];

        foreach($columns['columns'] as $column) {
            $col = [
                'name' =>  $column['name'],
                'label' => isset($column['label']) ? $column['label'] : iaPalabra::toLabel($column['name']),
                'index' => $this->colModelIndex($column, $indexUseTable),
                'template' => $this->columnTemplate($column),
            ];
            if(!empty($column['primaryKey'])) {
                $col['key'] = true;
                $col['hidden'] = true;
                $col['editoptions']['readonly'] = 'readonly';
            }
            if(!empty($column['readOnly']) || $forceReadonly) {
                $col['editoptions']['readonly'] = 'readonly';
            }
            if(isset($column['default'])) {
                $col['editoptions']['defaultValue'] = $column['default'];
            }
            if(!empty($column['null'])) {
                $col['editoptions']['NullIfEmpty'] = true;
            }
            if(isset($column['required'])) {
                $col['editrules']['required'] = $column['required'];
            }

            if(isset($column['min'])) {
                $col['editrules']['minValue'] = $column['min'];
            }
            if(isset($column['max'])) {
                $col['editrules']['maxValue'] = $column['max'];
            }

            if(isset($column['decimals'])) {
                $col['formatoptions']['decimalPlaces'] = $column['decimals'];
            }

            if(isset($column['values'])) {
                $options = [];
                $cssClassByValue = [];
                foreach($column['values'] as $key => $values) {
                    $options[$key] = is_array($values) && !empty($values['label']) ?
                        $values['label'] : $key;
                    $cssClassByValue[$key] = is_array($values) && !empty($values['class']) ?
                        $values['class'] : "$column[orgtable]_$column[orgname]_" .
                            str_replace(' ', '_', $key);
                }
                $arguments =  Str::jsStrIt( $col['name'] ) . ", " . Str::jsStrIt( $col['index'] ) . ", " . "'$column[type]'";
                $arguments .= ',' . json_encode($options) . ',' . json_encode($cssClassByValue);
                $template =  $this->colTemplates['enumSetAuto'];
                $col['template'] =  new Expr("$template($arguments)");
                $col['cellClassByValue'] = $cssClassByValue;
            }

            //  max_length, input width?
            // bit(1) sea checkbox
            // classes:"jqgRZ" wraps over content
            // special fields? email, link, image
            // -- comboBoxAutoComplete:{dataUrl:'./json/cheque_tienda.html'},

            if(isset($column['colModel'])) {
                $col = array_merge($col, $column['colModel']); // $column['colModel'] values override $col values
            }
            $colModel[$column['name']] = $col;
        }
        return $colModel;
    }

    /**
     * Returns the colModel json, use this method since it may contain function calls or js code
     *
     * @param array $colModel
     * @param bool $JSON_PRETTY_PRINT
     * @return string a json for the colModel
     * @noinspection PhpUnused
     */
    public function colModelArray2Json($colModel, $JSON_PRETTY_PRINT = true) {
        // JSON_PRESERVE_ZERO_FRACTION
        Json::$useBuiltinEncoderDecoder = false;
        $json = Json::encode(array_values($colModel), false, ['enableJsonExprFinder' => true]);
        return $JSON_PRETTY_PRINT ? Json::prettyPrint($json, array('indent' => '  ')) : $json;
    }

    /** configure */

    /**
     * get column templates names by mysql type
     *
     * @return array ['int'=>'columnTemplate',...]
     * @noinspection PhpUnused
     */
    public function get_colTemplates() {
        return $this->colTemplates;
    }

    /**
     * Extend add/replace column templates to use by type
     *
     * @param array $colTemplates ['int'=>'columnTemplate',...]
     * @noinspection PhpUnused
     */
    public function merge_colTemplates($colTemplates) {
        $this->colTemplates = array_merge($this->colTemplates, $colTemplates);
    }

    /**
     * Set column templates to use Type=>colTemplate
     *
     * @param array|null $colTemplates ['int'=>'columnTemplate',...]
     * @noinspection PhpUnused
     */
    public function setColTemplates($colTemplates)  {
        $this->colTemplates = $colTemplates;
    }

    /**
     * @return array actions colModel ['name'=>'act', 'label'=>' ',...]
     */
    public function getActionsCol()
    {
        return $this->actionsCol;
    }

    /**
     * Extend add/replace actions column
     *
     * @param array $actionsCol actions colModel ['name'=>'act', 'label'=>' ',...]
     * @noinspection PhpUnused
     */
    public function merge_actionsCol($actionsCol) {
        $this->actionsCol = array_merge($this->actionsCol, $actionsCol);
    }

    /**
     * @param array $actionsCol actions colModel ['name'=>'act', 'label'=>' ',...]
     */
    public function setActionsCol(array $actionsCol)
    {
        $this->actionsCol = $actionsCol;
    }



    /** helpers */

    /**
     * Determine the col Template to use
     *
     * @param array $column
     * @return string
     */
    protected function columnTemplate($column) {

        // by *_id not *=orgtable
        // by name general

        if(strcasecmp('timestamp', $column['type']) === 0) {
            return new Expr($this->colTemplates['timestamp']);
        }

        if(stripos($column['type'], 'int') !== false) {
            return new Expr($this->colTemplates['int']);
        }
        if(!empty($column['numeric'])) {
            return new Expr($this->colTemplates['float']);
        }
        if(stripos($column['type'], 'text') !== false || stripos($column['type'], 'blob') !== false) {
            return new Expr($this->colTemplates['text']);
        }

        return iaArray::array_key_exists_insensitive($column['type'], $this->colTemplates) ?
              new Expr($this->colTemplates[$column['type']])
            : new Expr('iaJqGrid.colTemplate.ct_'.strtolower($column['type']));
    }

    /**
     * For the index property, how the field name is sent prefix it with table name, alias or none
     *
     * @param array $column
     * @param int $indexUseTable self::TABLE_ALIAS, self::TABLE_NAME or self::TABLE_NONE
     * @return mixed|string
     */
    protected function colModelIndex($column, $indexUseTable) {
        // orgname or alias for table!
        // orgname para column
        $colName = isset($column['orgname']) ? $column['orgname'] : $column['name'];
        if($indexUseTable === self::TABLE_NONE) {
            return $colName;
        }
        if($indexUseTable === self::TABLE_NAME) {
            if(isset($column['orgtable'])) {
                return $column['orgtable'] . '.' . $colName;
            }
            if(isset($column['table'])) {
                return $column['table'] . '.' . $colName;
            }
            return $colName;
        }
        if($indexUseTable === self::TABLE_ALIAS) {
            if(isset($column['table'])) {
                return $column['table'] . '.' . $colName;
            }
            if(isset($column['orgtable'])) {
                return $column['orgtable'] . '.' . $colName;
            }
            return $colName;
        }
        return $colName;
    }
}
