<?php
/**
 * JqGridRead Helps respond to jqGrid from the database
 *
 */


/** @noinspection SqlNoDataSourceInspection */
/** @noinspection SqlResolve */
/** @noinspection PhpUnused */

namespace ia\JqGrid;

use ia\Lib\iaPalabra;
use ia\Lib\iaTableIt;
use ia\Sql\Mysql\IacSqlException;
use \ia\Sql\Mysql\IaMysqli;
use ia\Sql\Mysql\SqlBuilder;
use \ia\Util\Str;

///// Example usage //////////////////////////////////////////////////////////////////////////////////////////////////
    /**
     * @example Simple mode for all columns in table or view.

        $grid = new JqGridRead($gSqlClass);
        // defaults:
            // $grid->andWhere([]); // add a fixed and where clause (string or array) to the user's filter
            // $grid->setOrderBy("ORDER BY a.name"); // null or not set uses user's selection
            // $grid->setUserMayList(true);
            //  $grid->setUserMayExportList(true);
            //  $grid->setExportFileName('tableName');
            //  $grid->setExportPrimaryKey(true);
            //  $grid->setPrimaryKeyFieldName('table_id');
            //  $grid->setExtraSelectColumns(0);
            //  $grid->maxRowsAskToSend(2000);
            //  $grid->setmaxRowsAskToSend(1000);

        // process it
            switch($grid->getOperation()) {
                case '':
                case 'read':
                case 'count': // sends ['rowsToSend'=>$rowsToSend, 'maxRowsAsk'=>$grid->maxRowsAskToSend, 'maxRowsSend'=>$grid->maxRowsToSend];
                case 'csv':
                case 'xlsx':
                case 'pdf':
                    if($grid->processReadForTable('tableName', MYSQLI_ASSOC)) {
                        // json response sent. success log it
                    } else {
                        // error sent like no permission, not implemented, db error
                    }
                    break;

                default:
                    $grid->sendError($grid::ERROR_NOT_IMPLEMENTED);
            }

        // log errors
            logErrors(false);
     */

    /**
     * @example Simple mode query.
 *
* $grid = new JqGridRead($gSqlClass);
        * // defaults:
            * //         echo "<pre>".print_r($roleName,true);
     * //        die("<pre>has=$has sql=".$sql);; // add a fixed and where clause (string or array) to the user's filter
            * // $grid->setOrderBy("ORDER BY a.name"); // null or not set uses user's selection
            * // $grid->setUserMayList(true);
            * //  $grid->setUserMayExportList(true);
            * //  $grid->setExportFileName('listado');
            * //  $grid->setExportPrimaryKey(true);
            * //  $grid->setPrimaryKeyFieldName('id');
            * //  $grid->setExtraSelectColumns(0);
            * //  $grid->setAskOnSendMoreThanRows(1000);
            * //  $grid->maxRowsAskToSend(2000);
            * //  $grid->setmaxRowsAskToSend(1000);
 *
* // process it
            * switch($grid->getOperation()) {
                * case '':
                * case 'count': // sends ['rowsToSend'=>$rowsToSend, 'maxRowsAsk'=>$grid->maxRowsAskToSend, 'maxRowsSend'=>$grid->maxRowsToSend];
                * case 'read':
                * case 'csv':
                * case 'xlsx':
                * case 'pdf':
                    * // set queries
                        * $select = "SELECT a, b ,c FROM table";
                        * $selectCount = "SELECT COUNT(*) FROM table"; // or null
                        * $selectExport = "SELECT a, b ,c FROM table";
                        * $addExtraColumns = false; // add columns actions, rn, ... for using MYSQLI_NUM
                        * // returns ['readSelect','countSelect'] adding the where but no order by nor limit
                        * $queries = $grid->getQueriesForSql($select, $selectCount, $addExtraColumns);
                    * // process
                        * if($grid->processRead($queries['readSelect'], $queries['countSelect'], $selectExport, MYSQLI_ASSOC)) {
                            * // json response sent. success log it
                        * } else {
                            * // error sent like no permission, not implemented, db error
                        * }
                    * break;
 *
* default:
                    * $grid->sendError($grid::ERROR_NOT_IMPLEMENTED);
            * }
 *
* // log errors
            * logErrors(false);
     */

    /**
     * @example  flexible jqgrid example con makeResponse

        // Tip: set colmodel index to table.fieldName or tableAlias.fieldName

        // param array $ruleSolved=['field'=>'fieldName', 'op'=>'eq', 'data'=>'value', 'clause'=>'(`table`.`a`=2)']
        // returns string clause modified or '' clause is ignored
        // see Filter2Where::getOpOperator() Array Operators with simple substitution to sql [eq=>'=',...]
        // see Filter2Where::getOpToStandard() Array of valid operators

        // change filter behaviour for a specific field
        function overrideWhereClause($ruleSolved) {
            if($ruleSolved['field']==='table.soloIgual') // asume no es array $ruleSolved['data']
                return Str::fieldit($ruleSolved['field']).'='.Str::strit($ruleSolved['data']);
            return $ruleSolved['clause']; // default Filter2Where generated clause
        }

        global $gSqlClass;
        $grid = new JqGridRead($gSqlClass, overrideWhereClause);

        // where has been created calling Filter2where.php
        // $grid->andWhere(['tienda_id'=>3]);
        // $where = $grid->getWhereClause(); $where = $where === '' ? '' : " WHERE $where ";
        // or $where = $grid->getWhereWithKeyword(); // includes WHERE if required
        // or get where as an array $grid->getWhereClauseArray()

        // $grid->setOrderBy("ORDER BY a.name"); // null or not set uses user's selection
        // for MYSQLI_NUM add '', '', for extra columns (actions, rn,...) Tip: '','',table.*
           // $grid->setExtraSelectColumns(2);
           // $extraColumns = $grid->getSelectExtraColumns();

        $selectRecords = "SELECT ... " . $where;

        $selectMatchedRecords = "SELECT COUNT (*) FROM " . $where . ' ' . $grid->getOrderBy();
        // or $selectMatchedRecords = $grid->getCountSelectFromReadSelect($selectRecords);

        $selectExport = "SELECT ... FROM ... JOIN .. " . $where . ' ' ;

        $oper = $grid->getOperation(); // oper '', read, add, ...
        switch($oper) {
            case '':
            case 'read':
                $ok = $grid->makeReadResponse(
                    $selectRecords . ' ' . $grid->getOrderBy() . ' ' . $grid->getLimit(),
                    $selectMatchedRecords,
                    MYSQLI_ASSOC
                );
                if($ok) {
                    $response = $grid->getResponse();
                    echo json_encode($response);
                } else {
                    $errorNumber = $grid->getLastError(); // one of $grid::ERROR_*
                    $errorMessage = '';
                    $grid->sendError($errorNumber, $errorMessage);
                }
                break;

            case 'count':
                $rowsToSend = $grid->getRowsToSend($countSelect); // ['rowsToSend'=>$rowsToSend, 'maxRowsAsk'=>$grid->maxRowsAskToSend, 'maxRowsSend'=>$grid->maxRowsToSend];
                if($rowsToSend === false) {
                    $this->sendError($grid::ERROR_DB_ERROR);
                } else {
                    echo json_encode($rowsToSend);
                }
                break;

            case 'csv':
                $ok = $grid->csvExport(
                    $selectExport,
                    "listado",
                    false, // don't send primary key
                    'tabla_id' // column not to send
                );
                if(!$ok) {
                    $errorNumber = $grid->getLastError();
                }
                break;

            default:
                $grid->sendError($grid::ERROR_NOT_IMPLEMENTED);
        }
        // log errors
            logErrors(false);
        */

///// Class code ////////////////////////////////////////////////////////////////////////////////////////////////////
/**
 * Class JqGridRead Helps respond to jqGrid
 *  Tip: set colmodel index to table.fieldName or tableAlias.fieldName
 *
 * @package ia\JqGrid
 */
class JqGridRead {
    protected $userMayList = true;
    protected $userMayExportList = true;

    protected $maxRowsToSend = 2000;
    protected $maxRowsAskToSend = 1000;


    protected $exportFileName = 'listado';
    protected $exportPrimaryKey = true;
    protected $primaryKeyFieldName = null;
    protected $lastError = null;
    protected $orderBy = null;

    protected $db;
    protected $params;
    protected $selectExtraColumns = '';
    protected $whereClause;
    protected $response = [];

    const ERROR_NOT_IMPLEMENTED = 1;
    const ERROR_DB_ERROR = 2;
    const ERROR_PERMISSION = 3;

    /**
     * JqGridRead constructor.
     *
     * @param IaMysqli $db
     * @param null|callable $filterFieldOverride function($ruleSolved):string
     *      param array ruleSolved ['field'=>'fieldName', 'op'=>'eq', 'data'=>'value', 'clause'=>'(`table`.`a`=2)']
     *      return string sql clause or '' = don't filter
     *      example rule2sql($r, 'overrideClause'); function overrideClause($ruleSolved);
     *      example rule2sql($r, [$reFilter, "overrideClause"]); class reFilter { public function overrideClause($ruleSolved);}
     *      example rule2sql($r, "reFilter::overrideClause"); class reFilter { public static function overrideClause($ruleSolved);}
     */
    public function __construct($db, $filterFieldOverride = null){
        $this->db = $db;
        $this->read_params();
        $this->whereClause = $this->setWhereClause($filterFieldOverride);

    }

    /**
     * @param string $idFieldName
     * @param array $fieldNames
     * @return array
     */
    public function readParams($idFieldName, $fieldNames) {
        $params = [];
        foreach($fieldNames as $key) {
            $params[$key] = isset($_POST[$key]) ? trim($_POST[$key]) : '';
        }
        if(empty($params[$idFieldName]) && isset($_POST['id'])) {
            $params[$idFieldName] = $_POST['id'];
        }
        return $params;
    }
///// Configure list and export mode /////////////////////////////////////////////////////////////////////////////////
    /**
     * Adds an and where clause to the user's filter
     *
     * @param array|string $whereClause
     * @return void
     */
    public function andWhere($whereClause) {
        if(empty($whereClause)) {
            return;
        }
        if(is_array($whereClause)) {
            $builder = new SqlBuilder();
            $whereClause = $builder->where($whereClause);
        }
        $this->whereClause .= ($this->whereClause === '' ? '' : ' AND ') . "($whereClause)";
    }

    /**
     * For simple mode (processRead) set user permission to list records
     *
     * @param bool $userMayList
     */
    public function setUserMayList(bool $userMayList) {
        $this->userMayList = $userMayList;
    }

    /**
     * For simple mode (processRead) set user permission to export records
     *
     * @param bool $userMayExportList
     */
    public function setUserMayExportList(bool $userMayExportList) {
        $this->userMayExportList = $userMayExportList;
    }

    /**
     * @param int $maxRowsToSend
     */
    public function setMaxRowsToSend(int $maxRowsToSend) {
        $this->maxRowsToSend = $maxRowsToSend;
    }



    /** For MYSQLI_NUM set the additional special columns (act, rn, ...).
     *
     * @param mixed $extraColumns int:Number, array['act','rn']
     * @return string columns to add to the select ie "'',''" no trailing coma
     */
    public function setExtraSelectColumns($extraColumns) {
        if(empty($extraColumns)) {
            $this->selectExtraColumns = '';
            return '';
        }
        if(is_numeric($extraColumns)) {
            $this->selectExtraColumns = str_repeat("'',", $extraColumns);
            return $this->selectExtraColumns;
        }
        if(is_array($extraColumns)) {
            $this->selectExtraColumns = str_repeat("'',", count($extraColumns));
            return $this->selectExtraColumns;
        }
        $this->selectExtraColumns = $extraColumns;
        return $extraColumns;
    }

    /**
     * @return string
     */
    public function getSelectExtraColumns() {
        return $this->selectExtraColumns;
    }

    /**
     * Replaces user's selected ORDER BY, null or unset uses user's selection
     *
     * @param null|string $orderBy ORDER BY a.id,a.name on null or not set uses user selection
     */
    public function setOrderBy($orderBy) {
        $this->orderBy = trim($orderBy);
    }

    /**
     * Filename for export, no extension
     *
     * @param string $exportFileName name of download export no extension
     */
    public function setExportFileName(string $exportFileName) {
        $this->exportFileName = $exportFileName;
    }

    /**
     * On export send primary key? also set setPrimaryKeyFieldName()
     *
     * @param bool $exportPrimaryKey on true primary key is exported, false field Named ->setPrimaryKeyFieldName() not exported
     */
    public function setExportPrimaryKey(bool $exportPrimaryKey) {
        $this->exportPrimaryKey = $exportPrimaryKey;
    }

    /**
     * Name of the primary key field for export rows see setExportPrimaryKey()
     *
     * @param string $primaryKeyFieldName name of the primary key field, usefull when ->setExportPrimaryKey(false)
     */
    public function setPrimaryKeyFieldName($primaryKeyFieldName) {
        $this->primaryKeyFieldName = $primaryKeyFieldName;
    }


/////// Functionality //////////////////////////////////////////////////////////////////////////////////////////////////////////

    public function getOperation() {
        return $this->params['oper'];
    }

    public function get_oper() {
        return $this->params['oper'];
    }

    /**
     * Returns queries for table or view ['readSelect','countSelect'] with where but no order by nor limit,
     * adds extraSelectColumns if set except for export.
     *
     * @param string $table table or view name
     * @return array ['readSelect','countSelect'] with where no order by nor limit
     */
    public function getQueriesForTable($table) {
        $tableName = Str::fieldit($table);
        $where = $this->getWhereWithKeyword();
        if($this->selectExtraColumns !== '' && ($this->getOperation() === 'read' || $this->getOperation()==='')) {
            $fields = $this->selectExtraColumns . ", $tableName.* ";
        } else {
            $fields = '*';
        }
        if($this->exportFileName === 'listado') {
            $this->exportFileName = $table;
        }
        if($this->primaryKeyFieldName === null) {
            $this->primaryKeyFieldName = $table.'_id';
        }

        return [
            'readSelect' => "SELECT $fields FROM $tableName $where /*" . __METHOD__ ."*/",
            'countSelect' => "SELECT COUNT(*) FROM $tableName $where  /*" . __METHOD__ ."*/"
        ];
    }

    /**
     * Returns queries ['readSelect','countSelect'] adding the where no order by nor limit
     *
     * @param string $readSelectNoWhere without the where nor order by nor limit
     * @param null|string|int $countSelectNoWhere string SELECT count(*), null creates query, int uses it as total rows that match where
     * @param bool $addExtraColumnsOnRead false leave select as is, true add extra columns if any. Using MYSQI_ASSOC leave false
     * @return array ['readSelect','countSelect'] with where no order by nor limit
     */
    public function getQueriesForSql($readSelectNoWhere, $countSelectNoWhere = null, $addExtraColumnsOnRead = false) {
        $where = $this->getWhereWithKeyword();
        $read = $readSelectNoWhere . ' ' . $where . " /*" . __METHOD__ . "*/";
        if($addExtraColumnsOnRead && $this->selectExtraColumns !== '' && ($this->getOperation() === 'read' || $this->getOperation()==='')) {
            $read = Str::replaceFirstString('SELECT', $this->selectExtraColumns . ',', $readSelectNoWhere);
        }
        return [
            'readSelect' => $read,
            'countSelect' => $countSelectNoWhere === null || $countSelectNoWhere === '' ?
                $this->getCountSelectFromReadSelect($readSelectNoWhere . ' ' . $where) :
                $countSelectNoWhere . ' ' . $where
        ];
    }

    /**
     * Sends jqGrid json read response or file export for $table. Sends header on db error or no permission
     *
     * @param string $table table or view name
     * @param int $assocMethod  MYSQLI_ASSOC or MYSQLI_NUM
     * @return bool true ok, false error
     */
    public function processReadForTable($table, $assocMethod = MYSQLI_ASSOC) {
        $queries = $this->getQueriesForTable($table);
        return $this->processRead($queries['readSelect'], $queries['countSelect'], $queries['readSelect'], $assocMethod);
    }

    /**
     * Sends jqGrid json read response or file export for $readSelect. Sends header on db error or no permission
     *
     * @param string $readSelect with where no order by nor limit
     * @param string $countSelect with where
     * @param int $assocMethod MYSQLI_ASSOC or MYSQLI_NUM
     * @return bool true ok, false error
     */
    public function processRead($readSelect, $countSelect, $exportSelect, $assocMethod = MYSQLI_ASSOC) {
        switch($this->getOperation()) {
            case '':
            case 'read':
                if (!$this->userMayList) {
                    $this->sendError(self::ERROR_PERMISSION);
                    return false;
                }
                return $this->sendReadResponse($readSelect, $countSelect, $assocMethod);
                break;

            case 'count':
                $rowsToSend = $this->getRowsToSend($countSelect);
                if($rowsToSend === false) {
                    $this->sendError(self::ERROR_DB_ERROR);
                    return false;
                }
                echo json_encode($rowsToSend);
                return true;
                break;

            case 'csv':
                if(!$this->userMayExportList) {
                    $this->sendError(self::ERROR_PERMISSION);
                    return false;
                }
                return $this->csvExport($exportSelect . $this->getWhereWithKeyword() . ' '. $this->getOrderBy() );
                break;

          case 'xlsx':
            if(!$this->userMayExportList) {
              $this->sendError(self::ERROR_PERMISSION);
              return false;
            }
            $this->xlsxExport($exportSelect . $this->getWhereWithKeyword() . ' '. $this->getOrderBy() );
            return true;
            break;

          case 'pdf':
            if(!$this->userMayExportList) {
              $this->sendError(self::ERROR_PERMISSION);
              return false;
            }
            $this->pdfExport($exportSelect . $this->getWhereWithKeyword() . ' '. $this->getOrderBy());
            return true;
            break;

          default:
              $this->sendError(self::ERROR_NOT_IMPLEMENTED);
              return false;
        }
    }

    public function getRowsToSend($countSelect) {
        $rowsToSend = $this->countRowsToSend($countSelect);
        if($rowsToSend === false) {
            return false;
        }
        return ['rowsToSend'=>$rowsToSend, 'maxRowsAsk'=>$this->maxRowsAskToSend, 'maxRowsSend'=>$this->maxRowsToSend];
    }

    /**
     * Sends jqGrid json read response $readSelect. Sends header on db error or no permission
     *
     * @param string $readSelect with where no order by nor limit
     * @param string $countSelect with where no order by nor limit
     * @param int $assocMethod  MYSQLI_ASSOC or MYSQLI_NUM
     * @return bool true ok json is sent, false error header is sent
     */
    public function sendReadResponse($readSelect, $countSelect, $assocMethod = MYSQLI_ASSOC) {
        if($this->makeReadResponse( // takes care of max rows to send
            $readSelect.' '.$this->getOrderBy() . ' ' . $this->getLimit(),
            $countSelect,
            $assocMethod
        )) {
            $this->response['gw'] = $this->getWhereWithKeyword();
            $this->response['cw'] = $readSelect.' '.$this->getOrderBy() . ' ' . $this->getLimit();
            $this->response['re'] = $_REQUEST;
            echo json_encode($this->response);
            return true;
        }
        $this->sendError(self::ERROR_DB_ERROR);
        return false;
    }

    /**
     * Prepares the response for jqGrid read, get it with ->getResponse() then send it with echo json_encode($this->response);
     *
     * @param string $selectRecords with where order by and limit
     * @param string $selectCountRecords with where
     * @param int $assocMethod  MYSQLI_ASSOC or MYSQLI_NUM
     * @return bool true ok, false error a
     */
    public function makeReadResponse($selectRecords, $selectCountRecords, $assocMethod = MYSQLI_ASSOC) {
        if(is_numeric($selectCountRecords)) {
            $totalMatchedRecords = $selectCountRecords;
        } else {
            try {
                $totalMatchedRecords = $this->db->single_read($selectCountRecords, 0);
            } catch (IacSqlException $e) {
                return false;
            }
        }
        if($totalMatchedRecords === false) {
            return false;
        }

        $this->setTotalRecordsPages($totalMatchedRecords);

        try {
            $this->response['rows'] = $this->db->selectArrayIndex($selectRecords, [], $assocMethod);
        } catch (IacSqlException $e) {
            return false;
        }
        if($this->response['rows'] === false) {
            return false;
        }
        return true;
    }



////// Export functions ////////////////////////////////////////////////////////////////////////////////////////////////////////
  public function pdfExport($selectToExport, $fileName = null, $conPK = null, $pkFieldName = null) {
      if($fileName === null) {
        $fileName = $this->exportFileName;
      }
      if($conPK === null) {
        $conPK = $this->exportPrimaryKey;
      }
      if($pkFieldName === null) {
        $pkFieldName = $this->primaryKeyFieldName;
      }
      try {
        $exporta = $this->db->selectArrayIndex($selectToExport);
      } catch (IacSqlException $e) {
        return false;
      }
      if($exporta === false) {
        return false;
      }
      if(!$conPK) {
        $this->delete_col($exporta, $pkFieldName);
      }

      iaTableIt::pdfExport($exporta, $fileName);
  }

  public function xlsxExport($selectToExport, $fileName = null, $conPK = null, $pkFieldName = null) {
      if($fileName === null) {
        $fileName = $this->exportFileName;
      }
      if($conPK === null) {
        $conPK = $this->exportPrimaryKey;
      }
      if($pkFieldName === null) {
        $pkFieldName = $this->primaryKeyFieldName;
      }
      try {
        $exporta = $this->db->selectArrayIndex($selectToExport);
      } catch (IacSqlException $e) {
        return false;
      }
      if($exporta === false) {
        return false;
      }
      if(!$conPK) {
        $this->delete_col($exporta, $pkFieldName);
      }
      //@TODO deduce xlsx column type
      $firstRow = reset($exporta);
      $headers = array_combine(
        array_keys($firstRow),
        array_fill(0, count($firstRow), "string")
      );
      $writer = new \XLSXWriter();
      $writer->writeSheetHeader('Sheet1', $headers );
      foreach($exporta as $row) {
        $writer->writeSheetRow('Sheet1', $row );
      }

      header('Content-Disposition: attachment; filename="'. $fileName .'.xlsx"');
      header("Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");

      header('Content-Transfer-Encoding: binary');
      header('Cache-Control: must-revalidate');
      header('Pragma: public');

      $writer->writeToStdOut();
    }

    public function csvExport($selectToExport, $fileName = null, $conPK = null, $pkFieldName = null) {
        if($fileName === null) {
            $fileName = $this->exportFileName;
        }
        if($conPK === null) {
            $conPK = $this->exportPrimaryKey;
        }
        if($pkFieldName === null) {
            $pkFieldName = $this->primaryKeyFieldName;
        }

        header("Content-Type: text/html; charset=utf-8");
        header("Cache-Control: no-store, no-cache");
        header("Content-Disposition: attachment; filename*=UTF-8''".$fileName.".csv");

        try {
            $exporta = $this->db->selectArrayIndex($selectToExport);
        } catch (IacSqlException $e) {
            return false;
        }
        if($exporta === false) {
            return false;
        }

        echo "\xEF\xBB\xBF";
        // header row
        foreach($exporta as $rec) {
            $row = [];
            foreach($rec as $fieldName=>$v) {
                if($conPK || (!$conPK && $fieldName !== $pkFieldName)) {
                    $row[] = Str::csvIt(iaPalabra::toLabel($fieldName));
                }
            }
            echo implode(',', $row)."\r\n";
            break; // only process the first row
        }
        // data rows
        foreach($exporta as $rec) {
            $row = [];
            foreach($rec as $fieldName=>$v) if($conPK || (!$conPK && $fieldName!=$pkFieldName)) {
                if( is_bool($v)) {
                    $row[] = $v ? 'true' : 'false';
                } elseif( is_numeric($v) ) {
                    $row[] = $v;
                } else {
                    $row[] = Str::csvIt($v);
                }
            }
            echo implode(',', $row)."\r\n";
        }

        return true;
    }

    protected function delete_col(&$array, $key) {
      foreach($array as &$d) {
        unset($d[$key]);
      }
    }

/////// Advanced usage functions ////////////////////////////////////////////////////////////////////////////////////////////

    public function getResponse() {
        return $this->response;
    }

    public function setTotalRecordsPages($totalMatchedRecrods) {
        $this->response['records'] = $totalMatchedRecrods;
        $this->response['page'] = $this->params['page'];
        if(empty($this->params['rows']) || !is_numeric($this->params['rows']) || $this->params['rows'] <= 0 ) {
            $this->response['total'] = 1;
            $this->response['page'] = 1;
            return;
        }
        $this->response['total'] = $this->response['records'] <= 0 ? 0 :
            $this->response['total'] = max( ceil($this->response['records']/$this->params['rows']),1);
        if ($this->params['page'] > $this->response['total']) {
            $this->response['page'] = $this->response['total'];
        }
    }

    public function getCountSelectFromReadSelect($readSelect) {
        return "with `a u t o-c o u n t` as ($readSelect) select count(*) from `a u t o-c o u n t`";
    }

    /**
     * @param string $selectCount
     * @return bool|int Number of rows to send or false on error
     */
    public function countRowsToSend($selectCount) {
        if(is_numeric($selectCount)) {
            $totalRecords = $selectCount;
        } else {
            try {
                $totalRecords = $this->db->single_read($selectCount, 0);
            } catch (IacSqlException $e) {
                $totalRecords = false;
            }
            if($totalRecords === false) {
                return false;
            }
        }
        if($this->params['rows'] === '') {
            return $totalRecords;
        }
        return max($totalRecords, $this->params['rows']);
    }

    /**
     * @return string where clause or '', without the WHERE keyword
     */
    public function getWhereClause() {
        return $this->whereClause;
    }

    /**
     * @return string WHERE ... or '' if no filter. includes the WHERE keyword
     */
    public function getWhereWithKeyword() {
        return empty(trim($this->whereClause)) ? ' WHERE 1=1' : " WHERE ".$this->whereClause;
    }

    /**
     * @param null|callable $filterFieldOverride function($ruleSolved):string
     *      param array ruleSolved ['field'=>'fieldName', 'op'=>'eq', 'data'=>'value', 'clause'=>'(`table`.`a`=2)']
     *      return string sql clause or '' = don't filter
     *      example rule2sql($r, 'overrideClause'); function overrideClause($ruleSolved);
     *      example rule2sql($r, [$reFilter, "overrideClause"]); class reFilter { public function overrideClause($ruleSolved);}
     *      example rule2sql($r, "reFilter::overrideClause"); class reFilter { public static function overrideClause($ruleSolved);}
     * @return array
     */
    public function getWhereClauseArray($filterFieldOverride = null) {

        $whereClause = [];
        if(!empty($this->params['searchField']) && !empty($this->params['searchOper']) && !empty($this->params['searchString'])) {
            $where = Filter2where::rule2sql(
                [
                'field'=>$this->params['searchField'],
                'op'=>$this->params['searchOper'],
                'data'=>$this->params['searchString']
                ],
                $filterFieldOverride
            );
            if(!empty($where)) {
                $whereClause[] = "($where)";
            }
        }
        if(!empty($this->params['filters'])) {

            $where = Filter2where::filter2where($this->params['filters'], 'AND', $filterFieldOverride);

            if(!empty($where)) {
                $whereClause[] = $where;
            }
        }

        return $whereClause;
    }

    public function getOrderBy() {
        if(!empty($this->orderBy)) {
            return $this->orderBy;
        }
        if(empty($this->params['sidx'])) {
            return '';
        }
        $orderBy = [];
        foreach(explode(',',  $this->params['sidx'].' '.$this->params['sord']) as $clause) {
            $part = [];
            foreach(explode(' ', $clause) as $d) {
                $q0 = Str::strim($d);
                if(empty($q0)) {
                    continue;
                }
                if(strcasecmp('ASC',$q0) === 0 || strcasecmp('DESC',$q0) === 0 || strcasecmp('RAND()',$q0) === 0 ) {
                    $part[] = $q0;
                    continue;
                }
                $part[] = is_numeric($q0) ? $q0 : Str::fieldit($q0);
            }
            $orderBy[] = implode(' ', $part);

        }
        return ' ORDER BY ' . ( empty($orderBy) ? '1' : implode(', ', $orderBy)) . ' ';
    }

    public function getLimit() {
        if(empty($this->params['rows']) || !is_numeric($this->params['rows']) )
            return '';
        if($this->params['rows']<=0)
            $this->params['rows']=10;
        if( empty($this->params['page']) || !is_numeric($this->params['page']) || $this->params['page']<=0 )
            $this->params['page']=1;
        $start = $this->params['rows'] * $this->params['page'] - $this->params['rows'];
        if($start <0) {
            $start = 0;
        }
        return " LIMIT $start,".$this->params['rows'].' ';
    }

    /**
     * Sends error message to the grid
     *
     * @param null|int $errorNumber null|int null no error int one of self::ERROR_ or 400 - 599 HTTP codes
     * @param string $message on empty sends appropriate message
     * @return void
     */
    public function sendError($errorNumber, $message = '') {
        if($errorNumber === null || $errorNumber === 0) { // no error
            return;
        }
        switch($errorNumber) {
            case self::ERROR_PERMISSION:
                $errorNumber = 401;
                $messageDefault = 'Sin permiso';
                break;
            case self::ERROR_NOT_IMPLEMENTED;
                $errorNumber = 501;
                $messageDefault = 'Operacion no reconocida';
                break;
            case self::ERROR_DB_ERROR;
                $errorNumber = 501;
                $messageDefault = 'Error de la base de datos intente mas tarde';
                break;
            default:
                if($errorNumber < 400 || $errorNumber > 599) {
                    $errorNumber = 501;
                }
                $messageDefault = 'Error al leer datos, intente mas tarde';
        }
        $msgSend = empty($message) ? $messageDefault : $message;
        header($_SERVER["SERVER_PROTOCOL"]." $errorNumber $msgSend");
    }

    /**
     * @return null|int null no error int one of self::ERROR_* or 400 - 599 HTTP codes
     */
    public function getLastError() {
        return $this->lastError;
    }


//////// Helpers ///////////////////////////////////////////////////////////////////////////////////////////////////////////////

    protected function read_params() {
        $paramsToRead = [
            'oper' => '',
            '_search' => false,
            'searchField' => '',
            'searchOper' => '',
            'searchString' => '',
            'filters' => '',
            'sidx' => '',
            'sord' => '',
            'rows' => null,
            'page' => 1,
        ];
        foreach($paramsToRead as $key => $default) {
            $this->params[$key] = isset($_REQUEST[$key]) ? $_REQUEST[$key] : $default;
        }
    //  echo "<pre>params IN =".print_r($this->params,true);
        if(isset($this->params['filters']) && is_string($this->params['filters'])) {
          $this->params['filters'] = json_decode($this->params['filters'],true );
        }
     //   echo "<pre>params=".print_r($this->params,true);
    }

    /**
     *
     * @param null|callable $filterFieldOverride
     * @return string
     */
    protected function setWhereClause($filterFieldOverride) {
        return trim(implode(' AND ', $this->getWhereClauseArray($filterFieldOverride)));
    }

}
