<?php
/** @noinspection PhpMissingParamTypeInspection */
/** @noinspection PhpMissingReturnTypeInspection */

/**
 * IacSqlBuilder.php
 * Helps build sql statements like insert, update, where clauses
 *
 * @package sql
 * @author Informatica Asociada
 * @copyright 2015
 * @version 1.0.0
 */
namespace Iac\inc\sql;
require_once( dirname(__FILE__) . "/IacStringer.php");

/**
 * IacSqlBuilder
 * Helps build sql statements like insert, update, where clausesactualiza_banco_cuenta_fiduciarios
 *
 * @author Informatica Asociada
 * @copyright 2015
 * @version 1.0.0
 */
class IacSqlBuilder {
    use IacStringer;

    public $dontQuoteValue = array(
        'IA_UUID()' => 1,
        'CURDATE()'=>1,'CURRENT_DATE()'=>1,'CURRENT_DATE'=>1,'SYSDATE()'=>1,'UTC_DATE()'=>1,
        'CURRENT_DATETIME'=>1,'NOW()'=>1, 'NOW(6)' => 1,
        'CURRENT_TIME()'=>1,'CURRENT_TIME'=>1,'CURTIME()'=>1,'UTC_TIME()'=>1,
        'CURRENT_TIMESTAMP()'=>1,'CURRENT_TIMESTAMP'=>1,'LOCALTIMESTAMP()'=>1,'LOCALTIMESTAMP'=>1,'UNIX_TIMESTAMP()'=>1,'UTC_TIMESTAMP()'=>1
    );

    public $dontQuoteFieldName = array(
    );

    public $dontOnUpdateFieldName = array(
        'alta_db'=>1
    );

    /**
     * Build a where clause from an array or string
     *
     * @param mixed $where array('col_1'=>3,...)  col_1 = '3' AND ...  or string col_1=1
     * @param string $extraStringWhere
     * @return string
     *
     */
    public function where($where,  $extraStringWhere = '', $booleanOperator = 'AND', $fieldNameDontQuote = []) {
        if(!is_array($where) || empty($where))
            if(empty($where))
                return $extraStringWhere;
            else
                return empty($extraStringWhere) ? $where : " $where ($extraStringWhere)";

        $excludeQuote = array_merge($this->dontQuoteFieldName, array_flip($fieldNameDontQuote));
        $w = [];
        foreach($where as $fieldName=>$value){
            if($value === null) {
                $w[] = $this->fieldit($fieldName) . " IS NULL";
            }
            elseif(is_array($value)) {
                $w[] = $this->fieldit($fieldName) . " IN " . $this->arrayToSqlIn($value);
            }
            else {
                $w[] = $this->fieldit($fieldName) . "=" .
                    (array_key_exists(strtoupper($value ?? ''), $this->dontQuoteValue)
                || array_key_exists($fieldName, $excludeQuote) ?
                    $value :
                    $this->strit($value));
            }
        }

        return '(' . implode(" $booleanOperator ", $w) . ')' .
            (empty($extraStringWhere) ? '' : " $booleanOperator ($extraStringWhere)");
    }

    public function arrayToSqlIn(array $array):string {
        $inArray = [];
        foreach($array as $v) {
            $inArray[] = array_key_exists(strtoupper($v ?? ''), $this->dontQuoteValue) ? $v : strit($v);
        }
        if(empty($inArray)) {
            return "('\t')";
        }
        return '(' . implode(',', $inArray) . ')';
    }

    /**
     * Protect with ` quotes a: column name to `column name` respecting . table.column to `table`.`column`
     *
     * @param string $fieldName
     * @return string
     */
    public function fieldit($fieldName) {
        if(str_contains($fieldName,"->", ))
            return $fieldName;
        $protected = [];
        $n = explode('.',$fieldName);
        foreach($n as $field) {
            $protected[]= '`'.str_replace('`', '', self::strim($field ?? '') ).'`';
        }
        return implode('.', $protected);
    }

    /**
     * update()
     *
     *
     * @example
     *
     *
     * @param string $table
     * @param array $values array('col_1'=>'value', ...)
     * @param mixed $where array('col_1'=>'3',...)  col_1 = 3 AND ...  or string col_1=1
     * @param array $fieldNameDontQuote  array('col_2',...) col_2's value wont be quoted in the set clause
     * @param array $whereDontQuoteFieldName array('col_1',...) col_1's value wont be quoted in the where clause
     * @param string $comment comentario para el query
     * @return string sql update command
     *
     * @see $dontQuoteValue
     * @see $dontQuoteFieldName
     */
    public function update($table, $values, $where=[], $fieldNameDontQuote=[], $whereDontQuoteFieldName=[], $comment='', $table_alias = '' ) {
        $excludeQuote = array_merge($this->dontQuoteFieldName, array_flip($fieldNameDontQuote));
        $set = '';
        if (empty($comment))
            $comment = "/* ACTUALIZA $table */";
        else
            $comment = str_starts_with($comment, '/*') ? $comment :  "/* $comment */";
        foreach($values as $fieldName => $value)
            if($value === null || strcasecmp($value, 'null') === 0) {
                $set .= "," . $this->fieldit($fieldName) . "=NULL";
            }
            elseif(array_key_exists($fieldName,$excludeQuote) || array_key_exists(strtoupper($value ?? ''),$this->dontQuoteValue) )
                $set .= "," . $this->fieldit($fieldName) . "=$value";
            else
                $set .= "," . $this->fieldit($fieldName) . "=".$this->strit($value);

        if (!empty($table_alias)) {
            $table_alias = "AS $table_alias";
        }
        return "UPDATE $comment " . $this->fieldit($table) . " $table_alias SET ".substr($set ?? '',1)." WHERE " .
            $this->where($where, '', 'AND', $whereDontQuoteFieldName);
    }

    /**
     * insert()
     *
     * @example
     *
     *
     * @param string $table
     * @param array $values array('col_1'=>'value', ...)
     * @param bool $autoOnUpdate true: an on duplicate key update clause will be added with $onUpdate concateanted
     *        with all fields not in $fieldNameDontQuote nor in $this->dontOnUpdateFieldName
     * @param string $onUpdate add to ON DUPLICATE KEY UPDATE  ie set last_insert_id: primary_key=LAST_INSERT_ID(primary_key)
     * @param array $fieldNameDontOnUpdate array('col_1',...) col_1's value wont be quoted in the values clause
     * @param array $fieldNameDontQuote  array('col_1',...) col_1's value wont be quoted in the values clause
     * @return string sql insert command
     *
     * @see $dontQuoteValue
     * @see $dontQuoteFieldName
     * @see $dontOnUpdateFieldName
     */
    function insert($table, $values, $autoOnUpdate=false, $onUpdate='', $fieldNameDontOnUpdate=[], $fieldNameDontQuote=[], $comment='', $table_alias = '') {

        $onDup = trim($onUpdate);
        if(!empty($onDup)) {
            if(!str_starts_with($onDup ?? '', ','))
                $onDup = ','.$onDup;
            if(str_ends_with($onDup ?? '', ','))
                $onDup = substr($onDup ?? '',0,-1);
        }

        // $comment = "/* INSERTA $table **/";

        if (empty($comment))
            $comment = "/* INSERTA $table **/";
        else
            $comment = "/* $comment **/";

        $dontOnDup = array_merge($this->dontOnUpdateFieldName, array_flip($fieldNameDontOnUpdate));
        $excludeQuote = array_merge($this->dontQuoteFieldName, array_flip($fieldNameDontQuote));

        $fields = '';
        $vals = '';
        foreach($values as $fieldName => $value) {
            $fieldProtected = $this->fieldit($fieldName);
            $fields .= ",$fieldProtected";
            if($autoOnUpdate && !array_key_exists($fieldName,$dontOnDup) && stripos($value ?? '','DEFAULT(') === false )
                $onDup .= ",$fieldProtected=VALUES($fieldProtected)";
            if($value === null)
                $vals .= ',NULL';
            elseif(array_key_exists($fieldName,$excludeQuote) || array_key_exists(strtoupper($value ?? ''),$this->dontQuoteValue) )
                $vals .= ",$value";
            else
                $vals .= ','.$this->strit($value);
        }
        if(!empty($onDup)) {
            $onDup = ' ON DUPLICATE KEY UPDATE '.substr($onDup ?? '',1);
        }
        return "INSERT $comment INTO $table(".substr($fields ?? '',1).") VALUES(".substr($vals ?? '',1).")$onDup";
    }

    /**
     * IacSqlBuilder::insertInto()
     *
     * @example
     *
     *
     * @param mixed $table
     * @param mixed $values array('col_1'=>'value', ...)
     * @return string
     *
     * @see insertValues()
     * @see insertOnDuplicateKey()
     *
     */
    public function insertInto($table,$values) {
        $fields = '';
        foreach($values as $fieldName => $_)
            $fields .= ',' . $this->fieldit($fieldName);
        return "INSERT INTO " . $this->fieldit($table) . "(".substr($fields ?? '',1).") ";
    }

    /**
     * IacSqlBuilder::insertValues()
     *
     *
     * @example
     *
     *
     * @param array $values array('col_1'=>'value', ...)
     * @param array $fieldNameDontQuote
     * @return string
     *
     * @see $dontQuoteValue
     * @see $dontQuoteFieldName
     * @see insertInto()
     * @see insertOnDuplicateKey()
     */
    public function insertValues($values, $fieldNameDontQuote=[]) {
        $excludeQuote = array_merge($this->dontQuoteFieldName, array_flip($fieldNameDontQuote));
        $vals = '';
        foreach($values as $fieldName => $value) {
//            if($value === null)
//                $vals .= ',NULL';
//            elseif(array_key_exists($fieldName,$excludeQuote) || array_key_exists(strtoupper($value ?? ''),$this->dontQuoteValue) )
//                $vals .= ",$value";
//            else
//                $vals .= ','.$this->strit($value);
            $vals .= empty($value) ?
                ', NULL' :
                (array_key_exists($fieldName,$excludeQuote) || array_key_exists(strtoupper($value ?? ''),$this->dontQuoteValue) ?
                    ",  $value" :
                    ','.$this->strit($value)
                );
        }
        if(empty($vals))
            return '';
        return '('.substr($vals ?? '',1).')';
    }

    /**
     * IacSqlBuilder::insertOnDuplicateKey()
     *
     *
     * @example
     *
     *
     *
     * @param array $values array('col_1'=>'value', ...)
     * @param string $onUpdate  add to ON DUPLICATE KEY UPDATE  ie set last_insert_id: primary_key=LAST_INSERT_ID(primary_key)
     * @param array $fieldNameDontOnUpdate ['colDontUpdate']
     * @return string
     *
     * @see $dontOnUpdateFieldName
     * @see insertInto()
     * @see insertValues()
     */
    public function insertOnDuplicateKey($values, $onUpdate='', $fieldNameDontOnUpdate=[]) {
        $onDup = trim($onUpdate);
        if(!empty($onDup)) {
            if(!str_starts_with($onDup ?? '', ','))
                $onDup = ','.$onDup;
            if(str_ends_with($onDup ?? '', ','))
                $onDup = substr($onDup ?? '',0,-1);
        }

        $dontOnDup = array_merge($this->dontOnUpdateFieldName, array_flip($fieldNameDontOnUpdate));

        foreach($values as $fieldName => $value)
            if (!array_key_exists($fieldName, $dontOnDup)) {
                $f = $this->fieldit($fieldName);
                $onDup .= ",$f=VALUES($f)";
            }

        if(empty($onDup))
            return '';
        return ' ON DUPLICATE KEY UPDATE '.substr($onDup ?? '',1);
    }

}
