<?php
/** @noinspection PhpComposerExtensionStubsInspection */

/**
 * Simula preload opCache subiendo al opcache los archivos encontrados bajo el directorio indicado
 * Correrlo desde el url o con get_async
 *
 * Si se desea un poco mas rapido correr 2
 *  uno con WarmUp->fill_opcache(SCANDIR_SORT_ASCENDING) y el otro WarmUp->fill_opcache(SCANDIR_SORT_DESCENDING)
 *
 */
namespace vitex\warmup;

use Throwable;

final class WarmUp {

    private int $phpFilesFound = 0;
    private int $phpFilesWithErrors = 0;
    private array $filesWithErrors = [];

    /**
     * Pone en el opcache los archivos .php bajo $path
     *
     * @param int $sortingOrder SCANDIR_SORT_ASCENDING,  SCANDIR_SORT_DESCENDING
     * @param string|null $path to find php files, null lo deduce
     * @param bool $lintOnly
     */
    public function fill_opcache($sortingOrder = SCANDIR_SORT_ASCENDING, $path = null, $lintOnly = false) {
        $this->phpFilesFound = 0;
        $this->phpFilesWithErrors = 0;
        if($path === null)
            $path = dirname(__DIR__ . '/../../');
        $path = strtr($path, "\\", '/');
        if (substr($path, -1) !== '/')
            $path .= '/';
        $this->get_files($sortingOrder, $path, $lintOnly);
//        $this->get_files($sortingOrder, "C:/wamp/apps/phpmyadmin4.1.14/", $lintOnly);
    }

    /**
     * Recorre $path y sube al opcache los archivos .php
     *
     * @param int $sortingOrder SCANDIR_SORT_ASCENDING,  SCANDIR_SORT_DESCENDING
     * @param string $path
     * @param bool $lintOnly en true no pone en opcache
     */
    private function get_files($sortingOrder, $path, $lintOnly)
    {
        $included = array_flip( get_included_files() );
        $exclude = array_flip([
            'css', 'css2', 'saas', 'demo', 'font', 'fonts', 'htmltest', 'js', 'js2', 'js_ia', 'locale',
            'test', 'testie', 'tests', 'example', 'examples', 'tutorial', 'doing',
            'img', 'image', 'images', 'jquery-ui-themes', 'PHPExcel'
        ]);
        $excludeFiles = array_flip([
            'limpiar_prog.php', 'warmUp.php', 'lintme.php',

        ]);
        foreach (scandir($path, $sortingOrder) as $file) {
            if (empty($file) || $file[0] === '.' ||
                array_key_exists(strtolower($file), $exclude) ||
                array_key_exists($file, $excludeFiles) ||
                strcasecmp($file, 'PHPExcel') === 0
            )
                continue;
            if (is_dir($path . $file)) {
                $this->get_files($sortingOrder,$path . $file . '/', $lintOnly);
                continue;
            }
            if(strstr($file, '.php') !== '.php' ||
                stripos($file, "old.php") !== false ||
                stripos($file, " Copy") !== false
            )
                continue;

            $this->phpFilesFound++;
            $filePath = $path . $file;

            if (!$lintOnly && opcache_is_script_cached($filePath))
                continue;
            exec('C:\wamp\bin\php\php5.6.16\php.exe -l ' . $filePath, $output, $lint);
            if ($lint == 0)
                try {
                    if(!opcache_is_script_cached($filePath) && !array_key_exists(realpath($filePath), $included))
                        opcache_compile_file($filePath);
                } catch(Throwable $e) {
                    $this->phpFilesWithErrors++;
                    $this->filesWithErrors[$filePath] = $filePath . " <i>al opcachear</i>";
                    if(!$lintOnly) {
                        echo "<li>Error al cachear: " . $e->getMessage() . "</li>";
                    }
                }
            else {
                $this->filesWithErrors[$filePath] = $filePath;
                $this->phpFilesWithErrors++;
                if(!$lintOnly)
                    echo "<li>Error en php -l $filePath</li>";
            }
            unset($output, $lint);
            echo "<li>$filePath";
            $blanks = str_pad('', 4096, '&nbsp;');
            echo $blanks;
            ob_flush();
            flush();
        }
    }

    /** Renglón resumen del resultado de correr fill_opcache*/
    public function filled_summary():string {
        $timing = isset($_SERVER['REQUEST_TIME_FLOAT']) ? round(microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'], 4) . " sec" : '';
        $opCacheStatus = opcache_get_status(false);
        return "\r\n<p>\r\n" .
            " Php files: " .
            number_format($this->getPhpFilesFound(), 0, '', ',') .
            " (not excluded). Cached Files: " .
            number_format($opCacheStatus['opcache_statistics']['num_cached_scripts'], 0, '', ',') .
            " Can't cache:   " .
            number_format($this->getPhpFilesWithErrors(), 0, '', ',') .
            $timing .
            "\r\n";
    }

    public function lint_summary():string {
        $timing = isset($_SERVER['REQUEST_TIME_FLOAT']) ? round(microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'], 4) : 0;
        $phpFilesFound = number_format($this->getPhpFilesFound(), 0, '', ',');
        $lintNumErrors = count($this->filesWithErrors);
        if($lintNumErrors === 0) {
            $lintErrors = "<li style='color:darkgreen'>.php files lint Ok!</li>";
        } else {
            $lintErrors = "<li style='color:red'>$lintNumErrors .php files con lint error:<ol><li>" .
              implode('<li>', $this->filesWithErrors) .  "</li></ol></li>";
        }
        return <<< LINT
        <ul>
            $lintErrors
            <li>Php Files Processed: <b>$phpFilesFound</b></li>
        </ul>
LINT;

    }

    /**
     * @return int
     */
    public function getPhpFilesFound(): int{return $this->phpFilesFound;}

    /**
     * @return int
     */
    public function getPhpFilesWithErrors(): int{return $this->phpFilesWithErrors;}

    /** revisa el opcache este enabled y settings del opcache */
    public function check_opcache():bool {
        $errors = [];
        if (!empty(ini_get('opcache.opcache_enabled')))
            $errors[] = "Error: Fix php.ini: opcache is disabled, set set [opcache] opcache_enabled=1 ";
        if (empty(ini_get('opcache.use_cwd')))
            $errors[] = "Error: Fix php.ini: use_cwd is off, set set [opcache] use_cwd=1 ";
        if (empty(ini_get('opcache.save_comments')))
            $errors[] = "Error: Fix php.ini: save_comments is off, set set [opcache] save_comments=1 ";
        if (empty(ini_get('opcache.revalidate_path')))
            $errors[] = "Error: Fix php.ini: revalidate_path is off, set set [opcache] revalidate_path=1 ";
        if (PHP_SAPI === 'cli' && empty(ini_get('opcache.enable_cli'))) {
            echo "\r\nError: Fix php.ini: set [opcache] enable_cli=1" .
                "\r\n  php.ini loaded: " . php_ini_loaded_file() . " \r\n";
            exit(1);
        }
        if(count($errors) === 0)
            return true;
        echo "<h2>opCache settings:</h2> <ul><li>" . implode("<li>", $errors) ."</ul>" .
            "<div><b>php.ini loaded: " . php_ini_loaded_file() . "</b></div>";
        return false;
    }

}
