/**
 * dateUtils
 * Format Date using php's format string.  Date to/From ymd. Get Today's date as ymd or Date, with no time.
 *
 * @type {{todayYMD: (function(): string), ymdToDate: (function(*): Date), is_ymd: ((function(*): boolean)|*), today: (function(): Date), format: ((function((Date|string|number), string=, string=): string)|*), dateToYMD: (function(*): string), dateToDay: (function(*): Date), setLanguage(*): ({shortMonth, shortDay, longMonth, longDay})}}
 */
const dateUtils = {
    version: "2024-03-16",

    /**
     * A date in any sets hours, minutes & seconds to 0 leaving the same Y-m-d ignoring timezones
     *
     * @param {string|int|Date} date as Date, timestamp a string (ymd) or any string accepted by Date
     * @returns {Date} at midnight, hours, minutes, seconds,... set to 0
     */
    dateToDay: function(date){
        if(typeof date === 'undefined')
            date = new Date();
        return this.is_ymd(date) ? new Date(date + " 00:00:00") :
            new Date((new Date(date)).setHours(0, 0, 0, 0) );
    },

    dateToYMD: function date(date) {
        if(typeof date === 'undefined')
            date = new Date();
        return this.dateToDay(date).toISOString().split('T')[0];
    },

    today: function() {return new Date((new Date()).setHours(0, 0, 0, 0) );},

    todayYMD: function() {return this.today().toISOString().split('T')[0];},

    ymdToDate: function(ymd) {return this.is_ymd(ymd) ? new Date(ymd + " 00:00:00") : new Date(ymd); },

    is_ymd: function(value, dateOnly = true) {
        if(value === null || !isNaN(value) || typeof value !== 'string')
            return false;
        if(dateOnly) {
            if(value.length !== 10)
                return false;
            const regex = /^\d\d\d\d[-.\\\/_](0[1-9]|1[0-2])[-.\\\/_](0[1-9]|[1-2][0-9]|3[0|1])$/gm;
            return value.match(regex) !== null;
        }
        if(value.length < 10 || value.length > 27)
            return false;
        const regex =
           /^\d\d\d\d[-.\\\/_](0[1-9]|1[0-2])[-.\\\/_](0[1-9]|[1-2][0-9]|3[0|1])($|([ T]([0-1]\d|2[0-4]):[0-5]\d:[0-5]\d))/gm;
        return value.match(regex) !== null;
    },

    /**
     * Formats a date according to the given format string in PHP style
     *
     * @param {Date|string|number} inputDate - The date to be formatted. Can be a Date object, a date string, or a timestamp.
     * @param {string} [formatPhpStyle="d/M/y"] - The format string to be used for formatting the date. Defaults to "d/M/y"
     *  https://www.php.net/manual/en/datetime.format.php.
     * @param {string} [language="en"] - Englinsh en, Spanish es
     * @returns {string} - The formatted date string.
     * @throws {Error} - If an error occurs during the formatting process.
     */
    format: function(inputDate, formatPhpStyle = "d/M/y", language = "en")  {
        let lang = this.setLanguage(language);
        if(inputDate === null)
            return "";
        try {
            function padZero(value) {return value < 10  ? `0${value}` : `${value}`;}
            let date;
            if(inputDate instanceof Date)
                date = inputDate;
            else if(typeof inputDate === 'object')
                return "[object]";
            else if(isNaN(inputDate))
                date = this.is_ymd(inputDate) ?
                    new Date(`${inputDate}T00:00:00`) : new Date(inputDate);
            else
                date = new Date(inputDate);

            const parts = {
                d: padZero(date.getDate()),
                j: date.getDate(),
                D: lang.shortDay[date.getDay()],
                l: lang.longDay[date.getDay()],
                w: date.getDay(),

                m: padZero(date.getMonth() + 1),
                n: date.getMonth() + 1,
                M: lang.shortMonth[date.getMonth()],
                F: lang.longMonth[date.getMonth()],

                Y: date.getFullYear(),
                y: date.getFullYear().toString().slice(-2),

                H: padZero(date.getHours()),
                G: date.getHours(),
                h: padZero(date.getHours() > 12 ? date.getHours() - 12 : date.getHours()),
                g: date.getHours() > 12 ? date.getHours() - 12 : date.getHours(),
                i: padZero(date.getMinutes()),
                s: padZero(date.getSeconds()),

                a: date.getHours() < 12 ? 'am' : 'pm',
                A: date.getHours() < 12 ? 'AM' : 'PM',
            };

            let skip = false;
            let ret = [];
            for(let i = 0, len = formatPhpStyle.length; i < len; ++i) {
                let c = formatPhpStyle[i];
                if(c === "\\") {
                    skip = true;
                    continue;
                }
                if(skip) {
                    skip = false;
                    ret.push(c);
                    continue;
                }
                ret.push(parts.hasOwnProperty(c) ? parts[c] : c);
            }
            return ret.join("");
        } catch(error) {
            console.log("ERROR: fmt.date arguments:", arguments);
            console.log("       fmt.date error:", error);
            return inputDate;
        }
    },

    setLanguage(language) {
        switch(language.toLowerCase()) {
            case 'es':
                return {
                    longMonth: ['Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio', 'Julio', 'Agosto',
                        'Septiembre', 'Octubre', 'Noviembre', 'Diciembre'],
                    shortMonth: ['Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun', 'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'],
                    longDay: ['Domingo', 'Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes', 'Sábado'],
                    shortDay: ['Do', 'Lu', 'Ma', 'Mi', 'Ju', 'Vi', 'Sa'],
                };
            case 'en':
            default:
                return {
                    longMonth: ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September',
                        'October', 'November', 'December'],
                    shortMonth: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                    longDay: ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'],
                    shortDay: ['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa'],
                };
        }
    },

};
