// noinspection ES6ConvertVarToLetConst,JSUnusedGlobalSymbols

/**
 * Filters a dataset based on provided initials and sorts the results. The matching is done using a specific property of the objects in the dataset.
 *
 * @param {Array<Object>} data - The dataset to be filtered. Each object in the array should contain the property to be matched.
 * @param {Array<String>} initials - The initials used to filter the dataset. These initials will be converted to uppercase and un-accented before matching.
 * @param {String} [property="label"] - The property of each object in the dataset that will be used for matching against the initials.
 * @return {Array<Object>} - The filtered and sorted dataset. Each object in the result will contain additional properties `_score` and `_normalized` used for sorting.
 */
function initialsFilter(data, initials, property) {
    if(initials.length === 0 || (initials.length === 1 && initials[0].trim().length === 0) || !Array.isArray(initials))
        return data;
    initials = initials.map(function(t) {return strip_tags(unAccent(t)).toUpperCase();});
    property = property || "label";
    for(let d of data) {
        if(d.hasOwnProperty("_normalized"))
            break;
        d._normalized = strip_tags(unAccent(d[property])).toUpperCase();
    }

    return filter(data, initials).sort(function(rowA, rowB) {
        if(rowA._starts && rowB._starts) {
            let matchPriority = rowB.matchPriority - rowA.matchPriority;
            if(matchPriority)
                return matchPriority;
            let len = rowA._normalized.length - rowB._normalized.length;
            if(len)
                return len;
            return rowA._normalized < rowB._normalized ? -1 : +(rowA._normalized > rowB._normalized);
        }
        if(rowA._starts)
            return -1;
        if(rowB._starts)
            return 1;
        let matchesCount = rowB['matches'] - rowA['matches'];
        if(matchesCount)
            return matchesCount;
        let matchPriority = rowB.matchPriority - rowA.matchPriority;
        if(matchPriority)
            return matchPriority;
        return rowA._normalized < rowB._normalized ? -1 : +(rowA._normalized > rowB._normalized);
    });

    function filter(data, initials) {
        var ret = [];
        var starts = initials[0];
        var regExp = buildRegExp(initials);

        for (let da of data) {
            try {
                var matches = da._normalized.match(regExp);
                if (matches === null)
                    continue;
                da.matches = matches.length;
            } catch (e) {}

            if (da.matches > 0) {
                var d = { ...da };

                // ✅ Match: do all initials exist somewhere in the label?
                d._starts = initials.every(i => d._normalized.indexOf(i) !== -1);

                d.matchPriority = 0;
                let words = d._normalized.split(/\b/);
                let currentIndex = -1;
                let matchInOrder = true;

                // ✅ Iterate over initials to check if they match words in correct order
                for (let i = 0; i < initials.length; i++) {
                    let found = false;
                    for (let j = 0; j < words.length; j++) {
                        if (j > currentIndex && words[j].startsWith(initials[i])) {
                            currentIndex = j;
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        matchInOrder = false;
                        break;
                    }
                }

                // ✅ Score per word position
                for (let i = 0; i < words.length; i++) {
                    let w = words[i];
                    if (w !== '' && w !== '-') {
                        if (regExp.test(w)) {
                            d.matchPriority += 1000 - i;
                        }
                    }
                }

                // ✅ Bonus if initials match words in exact sequence
                if (matchInOrder && initials.length > 1) {
                    d.matchPriority += 1200; // 🏆 Presence Combo Bonus
                }

// ✅ Penalty if initials match *more* words than intended (e.g., SP hitting SPORT & SPANDEX)
                if (d.matches > initials.length) {
                    d.matchPriority -= 900; // ✂️ Noise penalty
                }

// ✅ Bonus if match count exactly equals number of initials (clean mapping)
                if (d.matches === initials.length) {
                    d.matchPriority += 800; // 👑 Clean precision bonus
                }


                ret.push(d);
            }
        }

        return ret;
    }



    function buildRegExp(initials) {
        let re = [];
        for(let c of initials)
            re.push("\\b" + pregQuote(c));
        return new RegExp(re.join("|"), "giu");
    }

    function pregQuote(str) {
        var term =  strip_tags(str.normalize("NFD").replace(/[\u0300-\u036f]/g, ""));
        // return $.ui.autocomplete.escapeRegex(term)
        // return (str + '').replace(/([\\\.\+\*\?\[\^\]\$\(\)\{\}\=\!\<\>\|\:\/])/g, '\\$1');
        return term.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

    }

    function unAccent(text) {return text.normalize("NFD").replace(/[\u0300-\u036f]/g, "");}

}

