<?php

//use JetBrains\PhpStorm\Language; // Activar attribute #[ExpectedValues

/**
 * Cacher: a file cache:
 *    Register keys, with Cacher::init, and their callback that generates the contet that will be generated when
 *      the required key:  file dosen't exists or its ttl has expired,
 *      use same key to json_encode or var_export for use in php, and or cache as it is received, get it with the desired extension
 *      a call to Cacher:: generateIfNeeded, generate, generateAllIfNeeded or generateAll
 *
 *
 * @usage
 *   Cacher::init(
 *      __DIR__ . '/../cached', // defaults to __DIR__ . '/cached/'
 *      // predefined cache keys with content generator callable to be called if not exists or ttl has expired
 *     // este param podria ser include('cacher_definitons.php'),
 *      [
 *          'assocArray' => Cacher::def('getArray', ENCODE_JSON | ENCODE_PHP, 60 * 60 *2 ],
 *          'coso.txt' => Cacher::def(['className', 'staticMethod']),
 *          'coso.txt' => Cacher::def([(new ClassName()), 'method']),
 *      ],
 *      // time to live in seconds, defaults to 1 day: 60 * 60 * 24
 *      60 *60 *24
 *  );
 *
 *      $assocArray = Cacher::get('assocArray.php');
 *      $json = Cacher::get('assocArray.json');
 *      Cacher::readfile('coso.txt');
 *
 *  // cambio contenido
 *      Cacher::set('assocArray.php', $newValues);
 *
 * // elimina que cache, es decir que siempre regenere el contenido
 *   Cacher::setForceGenerate(true);
 *
 * // llaves no definidas en init
 *      Cacher::set('notRegistered.txt', 'manual key');
 *      $notRegistered = Cacher::get('notRegistered.txt', 'default value');
 *
 * @example example/exampleCacher.php
 *
 * @version 1.0.1 2021-04-29
 *
 */

final class Cacher {
    const ENCODE_NONE = 1;
    const ENCODE_JSON = 2;
    const ENCODE_PHP = 4;

    /** @var string $path To store and read cache files ends in / */
    /* final */ private static string $path = __DIR__ . '/cached/';
    /** @var int $ttlInSeconds default time to live in seconds, older files are generated */
    private static int $ttlInSeconds = 60 * 60 * 24; // 1 dia = sec * minutos * horas
    /** @var array $registered ['key' => [ 'call' => ?callable, 'encode' => Cacher::ENCODE_, 'ttlInSeconds' => ?int, ] */
    /* final */ private static array $registered = [];
    /** @var string $jsonExtension used when encoding has Cacher::ENCODE_JSON */
    /* final */ private static string $jsonExtension = '.json';

    private static bool $forceGenerate = false;


//// Initializers /////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Initialise settings, define registered cache keys with callback and change default settings
     *
     * @param string|null $path defaults to  __DIR__ . '/cached/'
     * @param array $registerd [key=>Cacher::def(), ...]
     * @param int|null $defaultTtlInSeconds defaults to 60 * 60 * 24; // 1 dia = sec * minutos * horas
     * @param string $jsonExtension defaults to .json
     */
    public static function init(?string $path = null, array $registerd = [], ?int $defaultTtlInSeconds = null,
                                string $jsonExtension = '.json'): void
    {
      if ($path !== null)
          self::$path = rtrim( str_replace("\\", '/', $path), '/' ) . '/';
      self::$registered = $registerd;
      if ($defaultTtlInSeconds !== null)
        self::$ttlInSeconds = $defaultTtlInSeconds;
      self::$jsonExtension = $jsonExtension;
    }

    /**
     *  Registered hepler: use ['key' => Cacher::def(...), ]
     *
     * @param callable|null $call 'function_name' o ['className', 'staticMethod'] o [(new Class()), 'methodName'] o [$classInstance,'methodName']
     * @param int $encode Cacher::ENCODE_NONE, Cacher::ENCODE_JSON, Cacher::ECNODE_PHP pueden combinarse con |
     * @param int|null $ttlInSeconds defaults vitex estandard; solo en casos raros cambiarlo aqui //  60 * 60 * 24 = 1 dia  = sec * minutos * horas,
     * @param string|null $exceptionPath
     * @return array
     */
    public static function def(
        ?callable $call = null,
       #[ExpectedValues([Cacher::ENCODE_NONE, Cacher::ENCODE_JSON, Cacher::ECNODE_PHP])]
       $encode = Cacher::ENCODE_NONE,
       ?int $ttlInSeconds = null,
       ?string $exceptionPath = null
    ):array {
        return ['call' => $call, 'encode' => $encode, 'ttlInSeconds' => $ttlInSeconds, 'exceptionPath' => $exceptionPath];
    }


    /**
     * @return bool true no hace cache genera cada vez. Default es false es decir cachea
     */
    public static function isForceGenerate(): bool{return self::$forceGenerate;}

    /**
     *
     * @param bool $forceGenerate true no hace cache genera cada vez. Default es false es decir cachea
     */
    public static function setForceGenerate(bool $forceGenerate): void{self::$forceGenerate = $forceGenerate;}


//// Leer   /////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Manda el key cacheado directo al browser con php's readfile, genera de no existir o estar expirado
     *
     * @param string $key
     * @param string $default
     * @return false|int false en error, int bytes sent
     */
    public static function readFile(string $key, $default = '') {
        try {
            $fullPath = self::fullPath($key);
            $r = self::getKeyDefinition($key);
            if(!empty($r))
                self::generateIfNeeded($key);
            if (ob_get_level())
                ob_end_flush();
            flush();
            if( is_readable($fullPath) && is_file($fullPath))
                return readfile($fullPath);
        } catch(Throwable $_) {}
        echo $default;
        return false;
  }

    /**
     * Regresa el key cacheado, $default en no existe o error, genera de no existir o estar expirado
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function get(string $key, $default = '') {

        try {
            $fullPath = self::fullPath($key);
            $r = self::getKeyDefinition($key);
            if(empty($r))
                return is_readable($fullPath) && is_file($fullPath) ?  file_get_contents($fullPath) : $default;
            self::generateIfNeeded($key);
            if(!is_readable($fullPath) || !is_file($fullPath)) {
                return $default;
            }

            if( substr($key, -4) === '.php' && (self::ENCODE_PHP & $r['encode']) === self::ENCODE_PHP ) {
                /** @noinspection PhpIncludeInspection */
                return (include($fullPath));
            }
            return file_get_contents($fullPath);
        } catch(Throwable $_) {
            return $default;
        }
    }


//// Escribir ///////////////////////////////////////////////////////////////////////////////////////////////

    public static function set(string $key, $content):bool {
        $r = self::getKeyDefinition($key);
        if(empty($r) || empty($r['encode']) || $r['encode'] === self::ENCODE_NONE)
            return self::write($key, $content);
        $ok = true;
        if( (self::ENCODE_NONE & $r['encode']) === self::ENCODE_NONE )
            $ok &= self::write($key, $content);
        $keyNoExtension = Cacher::getKeyNoExtension($key);
        if( (self::ENCODE_JSON & $r['encode']) === self::ENCODE_JSON )
            $ok &= self::write($keyNoExtension . self::$jsonExtension,
                json_encode(
                $content,
                JSON_INVALID_UTF8_IGNORE | JSON_INVALID_UTF8_SUBSTITUTE |JSON_NUMERIC_CHECK |
                JSON_PARTIAL_OUTPUT_ON_ERROR | JSON_PRESERVE_ZERO_FRACTION
                )
            );
        if( (self::ENCODE_PHP & $r['encode']) === self::ENCODE_PHP )
            $ok &= self::write($keyNoExtension . '.php', '<?php return ' . var_export($content, true) . ";\r\n");
        return $ok;
    }

    public static function generate(string $key):bool {
        $r = self::getKeyDefinition($key);
        if(empty($r['call']))
            return true;
        try {
            return self::set($key, call_user_func($r['call']));
        } catch(Exception $_) {}
        return false;
    }

    public static function generateAll() : bool {
        $ok = true;
        foreach(self::$registered as $key => $_)
            $ok &= self::generate($key);
        return $ok;
    }

    public static function generateIfNeeded(string $key):bool {
        if(self::$forceGenerate || self::isCachedExpired($key))
            return self::generate($key);
        return true;
    }

    public static function generateAllIfNeeded() : bool {
        $ok = true;
        foreach(self::$registered as $key => $_)
            $ok &= self::generateIfNeeded($key);
        return $ok;
    }


//// Informacción de keys/cache //////////////////////////////////////////////////////////////////////////////////////

    /**
     * return true if the cached file for the key dosen't exists o has expired,
     *
     * @param string $key
     * @return bool true the key has expired or not exists, ie: the cache should be generated
     */
    public static function isCachedExpired(string $key):bool {
        $r = self::getKeyDefinition($key);
        if(empty($r))
            return self::keyExpiredOrNotExists($key);
        try {
            $encode = $r['encode'];
            $files = [$key];
            if(self::ENCODE_JSON === (self::ENCODE_JSON & $encode))
                $files[$key] = self::getKeyNoExtension($key) . self::$jsonExtension;
            if(self::ENCODE_PHP === (self::ENCODE_PHP & $encode))
                $files[$key] = self::getKeyNoExtension($key) . '.php';
            foreach($files as $k)
                if(self::keyExpiredOrNotExists($k))
                    return true;
        } catch(Throwable $_) {return true;}
        return false;
    }

    public static function has(string $key):bool {
        $fullPath = self::fullPath($key);
        clearstatcache(true, $fullPath);
        return is_file($fullPath);
    }

    public static function getRegistered():array {return self::$registered;}

    public static function getPath():string {return self::$path;}

    public static function getJsonExtension():string {return self::$jsonExtension;}


//// helpers ////////////////////////////////////////////////////////////////////////////////////////////////

    private static function keyExpiredOrNotExists(string $key):bool {
        $fullPath = self::fullPath($key);
        clearstatcache(true, $fullPath);
        if(!is_file($fullPath))
            return true;
        $ttlInSeconds = $r['ttlInSeconds'] ?? self::$ttlInSeconds;
        $lastModified = filemtime($fullPath);
        return $lastModified === false || $ttlInSeconds === null || !is_numeric($ttlInSeconds) ||
            time() > ($lastModified + $ttlInSeconds);
    }

    private static function getKeyNoExtension($key) {
        $rpos = strrpos($key, '.');
        if($rpos === false)
            return $key;
        return substr($key, 0, $rpos);
    }

    private static function getKeyDefinition(string $key):array {
        if(array_key_exists($key, self::$registered))
            return self::$registered[$key];
        $keyNoExtension = Cacher::getKeyNoExtension($key);
        if(array_key_exists($keyNoExtension, self::$registered))
            return self::$registered[$keyNoExtension];
        return [];
    }

    private static function write(string $key, string $content):bool {
        $fullPath = self::fullPath($key);

        if(is_dir($fullPath))
          return false;
        $tempName = tempnam(self::$path, 'tmp');
        if($tempName === false) {
          self::delete($key);
          return false;
        }
        if(file_put_contents($tempName, $content) === false) {
          self::delete($key);
          return false;
        }
        if(rename($tempName, $fullPath)) return true;
        self::delete($key);
        return false;
    }



    private static function fullPath(string $key):string {
      return empty(self::$registered[$key]['exceptionPath']) ?
        self::$path . $key : rtrim(self::$registered[$key]['exceptionPath'], "/\\") . '/' . $key;
  }

//// Eliminar ///////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Borra un key o varios
     *
     * @param string|array $keys 'file1.json' o ['file1.json', 'file3.php', ...]
     */
    private static function delete( $keys):void {
        if(is_string($keys)) {
            $keys = [$keys];
        }
        foreach($keys as $key) {
            $fullPath = self::fullPath($key);
            if (is_file($fullPath))
                unlink($fullPath);
            $r = self::getKeyDefinition($key);
            if(empty($r) || empty($r['encode']))
                continue;
            if(self::ENCODE_JSON === (self::ENCODE_JSON & $r['encode'])) {
                $fullPath = self::fullPath(Cacher::getKeyNoExtension($key) . self::$jsonExtension);
                if (is_file($fullPath))
                    unlink($fullPath);
            }
            if(self::ENCODE_PHP === (self::ENCODE_PHP & $r['encode'])) {
                $fullPath = self::fullPath(Cacher::getKeyNoExtension($key) . '.php');
                if (is_file($fullPath))
                    unlink($fullPath);
            }
        }
    }


//// force singleton ////////////////////////////////////////////////////////////////////////////////////////

    /** Prevent new of the instance */
    private function __construct() {}

    /** Prevent cloning of the instance */
    private function __clone() { }

    /**
     * Prevent serialization of the instance
     *
     * @noinspection PhpUnusedPrivateMethodInspection
     */
    public function __sleep() { }

    /**
     * Prevent deserialization of the instance
     *
     * @noinspection PhpUnusedPrivateMethodInspection
    */
    public function __wakeup() { }

}
