# Plan: merge_alumnos_becados.php

## Purpose
Create a helper script to merge duplicate scholarship student records in the `alumnos_becados` table. Records are considered duplicates if they share the same `estudiante_id` and `beca_tipo`.

## Merge Logic

### Duplicate Detection
- Group records by `estudiante_id` + `beca_tipo` combination
- If a group has more than 1 record, merge them

### Merge Strategy
For each group of duplicates:
1. Keep the first record as the "survivor"
2. For `porcentaje_beca_colegiatura`: Take the non-NULL/non-zero value from any record
3. For `porcentaje_beca_inscripcion`: Take the non-NULL/non-zero value from any record
4. Delete the duplicate records (keep only the survivor)

### Example
Before merge:
| estudiante_id | beca_tipo | porcentaje_beca_colegiatura | porcentaje_beca_inscripcion |
|--------------|-----------|----------------------------|----------------------------|
| 0201839 | Campaña Financiera | 80 | NULL |
| 0201839 | Campaña Financiera | NULL | 80 |

After merge:
| estudiante_id | beca_tipo | porcentaje_beca_colegiatura | porcentaje_beca_inscripcion |
|--------------|-----------|----------------------------|----------------------------|
| 0201839 | Campaña Financiera | 80 | 80 |

## Implementation Steps

### Step 1: Create the PHP Script
Location: `/backoffice/helper/merge_alumnos_becados.php`

Structure:
1. **Include config.php** - For database connection and framework functions
2. **Query duplicates** - Find all groups with duplicates
3. **Process each group**:
   - Identify the survivor record (first by alta_db or existing non-NULL percentages)
   - Copy non-NULL percentage values from duplicates to survivor
   - Delete duplicate records
4. **Display results** - Show statistics and confirmation

### Step 2: Query to Find Duplicates
```sql
SELECT estudiante_id, beca_tipo, COUNT(*) as cnt
FROM alumnos_becados
GROUP BY estudiante_id, beca_tipo
HAVING COUNT(*) > 1
```

### Step 3: Merge Process
For each duplicate group:
1. Get all records in the group
2. Determine survivor (first record or one with most data)
3. Merge percentage fields: COALESCE(survivor.value, duplicate.value)
4. Delete duplicates by ID

### Step 4: Safety Features
- Preview mode: Show what will be merged without executing
- Transaction: Wrap in transaction for rollback capability
- Logging: Log each merge operation

## Script Structure

```php
<?php
require_once("../../inc/config.php");

// Step 1: Find duplicates
$duplicates = ia_sqlArrayIndx("SELECT estudiante_id, beca_tipo, COUNT(*) as cnt 
    FROM alumnos_becados 
    GROUP BY estudiante_id, beca_tipo 
    HAVING COUNT(*) > 1");

// Step 2: Process each duplicate group
foreach ($duplicates as $row) {
    // Get all records for this estudiante_id + beca_tipo
    $records = ia_sqlArrayIndx("SELECT * FROM alumnos_becados 
        WHERE estudiante_id = '{$row['estudiante_id']}' 
        AND beca_tipo = '{$row['beca_tipo']}'");
    
    // Identify survivor and duplicates
    $survivor = null;
    $duplicates_to_delete = array();
    
    foreach ($records as $record) {
        if ($survivor === null) {
            $survivor = $record;
        } else {
            $duplicates_to_delete[] = $record['alumno_becado_id'];
        }
    }
    
    // Merge percentage values
    foreach ($records as $record) {
        if ($record['alumno_becado_id'] !== $survivor['alumno_becado_id']) {
            if (empty($survivor['porcentaje_beca_colegiatura']) && !empty($record['porcentaje_beca_colegiatura'])) {
                $survivor['porcentaje_beca_colegiatura'] = $record['porcentaje_beca_colegiatura'];
            }
            if (empty($survivor['porcentaje_beca_inscripcion']) && !empty($record['porcentaje_beca_inscripcion'])) {
                $survivor['porcentaje_beca_inscripcion'] = $record['porcentaje_beca_inscripcion'];
            }
        }
    }
    
    // Update survivor
    $update_sql = "UPDATE alumnos_becados SET 
        porcentaje_beca_colegiatura = " . ($survivor['porcentaje_beca_colegiatura'] ?: 'NULL') . ",
        porcentaje_beca_inscripcion = " . ($survivor['porcentaje_beca_inscripcion'] ?: 'NULL') . "
        WHERE alumno_becado_id = '{$survivor['alumno_becado_id']}'";
    ia_query($update_sql);
    
    // Delete duplicates
    foreach ($duplicates_to_delete as $dup_id) {
        ia_query("DELETE FROM alumnos_becados WHERE alumno_becado_id = '{$dup_id}'");
    }
}
?>
```

## Expected Output
- Number of duplicate groups found
- Number of records merged/deleted
- Confirmation of successful operation

## Rollback Consideration
- Keep a backup of deleted IDs
- Can re-insert if needed