# MrW - Reporte Propietarios (Property Owners Report System)

## System Overview

**MrW - Reporte Propietarios** is a comprehensive property management and vacation rental income tracking system. It manages property owners, properties with government postal code integration, and tracks rental transactions from multiple Online Travel Agencies (OTAs): Airbnb, Agoda, Booking.com, and CasitaMX, as well as reservations from Property Management Systems (PMS): Hostify and Cloudbeds.

The system centralizes financial data from various vacation rental platforms and PMS systems, allowing property owners to see all their rental income, fees, taxes, reservations, and occupancy data in one place.

## Business Domain

**Property Management / Vacation Rental / Short-Term Rental**

This system manages:
- Property owners (Propietarios)
- Properties (Propiedades) with postal code integration
- Multi-platform rental transactions (Airbnb, Agoda, Booking.com, CasitaMX)
- Property Management System reservations (Hostify, Cloudbeds)
- Rental income and expenses
- Service fees and taxes
- Guest reservations and occupancy
- Currency exchange rates
- Year-over-year income reporting
- Government postal code catalog integration (SEPOMEX data)

## App Files (Model Layer)

### System-Specific Files
Located in `/app/`:

#### Property Owners:
- **`app_propietario.php`** - Property owner management
  - Table: `propietario`
  - Primary key: `propietario_id` (UUID)
  - Fields: department, owner name, investor info, contact email
  - Features: Owner/investor distinction, quick search, autocomplete on propietario field

#### Properties:
- **`app_propiedad.php`** - Property/building management
  - Table: `propiedad`
  - Primary key: `propiedad_id` (varchar(32))
  - Fields: property name, number of units, owner reference, street address, postal code data
  - Features: Property status tracking, postal code integration, owner linking
  - **Postal Code Fields**: Automatically populated from government catalog (codigo_postal, colonia, estado, municipio)

#### Transaction Models (One per OTA Platform):

- **`app_airbnb_transaction.php`** - Airbnb rental transactions
  - Table: `airbnb_transaction`
  - Primary key: `airbnb_transaction_id` (UUID)
  - Tracks: reservations, income, fees, taxes, exchange rates, guest info

- **`app_agoda_transaction.php`** - Agoda rental transactions
  - Table: `agoda_transaction`
  - Similar structure to Airbnb but for Agoda platform

- **`app_booking_transaction.php`** - Booking.com rental transactions
  - Table: `booking_transaction`
  - Similar structure to Airbnb but for Booking.com platform

- **`app_casitamx_transaction.php`** - CasitaMX rental transactions
  - Table: `casitamx_transaction`
  - Similar structure to Airbnb but for CasitaMX platform

#### Reservation Models (Property Management Systems):

- **`app_hostify_reserva.php`** - Hostify reservation management
  - Table: `hostify_reserva`
  - Primary key: `hostify_reserva_id` (UUID)
  - Fields: confirmation_code, guest_name, check_in, check_out, channel, anuncio, total_price, nights, currency
  - Tracks: PMS reservations from Hostify platform

- **`app_cloudbeds_reserva.php`** - Cloudbeds reservation management
  - Table: `cloudbeds_reserva`
  - Primary key: `cloudbeds_reserva_id` (UUID)
  - Fields: property, name, email, phone_number, gender, reservation_number, third_party_confirmation_number, type_of_document, document_issuing_country, city, adults, children, room_number, accommodation_total, amount_paid, check_in_date, check_out_date, nights, room_type, grand_total, deposit, balance_due, reservation_date, source, status, country, guest_status, estimated_arrival_time, origin
  - Tracks: Comprehensive guest and booking data from Cloudbeds PMS

### Framework Files (Shared with ALL Systems)
These files are part of the iaCase framework and are used by ALL systems in Quantix:

- `app_iac_usr.php` - User management
- `app_iac_log.php` - Activity logging
- `app_iac_field_permission.php` - Field-level permissions
- `app_iac_table.php` - Table metadata
- `app_iac_parametros.php` - System parameters
- And all other `app_iac_*.php` files...

Additional shared utility files:
- `app_estilos_grid.php` - Grid styling
- `app_vitex_grid.php`, `app_vitex_grid_col.php` - Grid configurations
- `app_reportes_grid.php` - Report grids
- And more...

## Backoffice Pages (UI Layer)

Located in `/backoffice/`:

### Property Owner Management:
- **`propietario.php`** - Property owners CRUD interface
  - List all property owners
  - Add/Edit/Delete operations
  - Manage investor relationships
  - Contact information
  - Search with autocomplete

### Property Management:
- **`propiedad.php`** - Properties CRUD interface
  - List all properties/buildings
  - Add/Edit/Delete operations
  - Track number of apartments per property
  - Link properties to owners (propietario_id)
  - View complete postal code information
  - Property status management (Active/Inactive)

### Helper Scripts:
- **`helper/link_propiedades_propietarios.php`** - Property ↔ Owner fuzzy matching utility
  - **Purpose**: Automatically link properties to owners using intelligent fuzzy matching
  - **Algorithm**: 4-tier matching system (Exact → Contains → Similarity → Street)
  - **Visual Interface**: Real-time preview with confidence scores and color coding
  - **Modes**: Preview, Apply High Confidence (≥80%), Apply All, Export CSV
  - **Features**:
    - Normalizes text (removes accents, case-insensitive)
    - Handles pipe-separated department lists (e.g., "Dept A | Dept B")
    - Street name extraction for partial matches
    - Batch UPDATE operations with confidence thresholds
    - CSV export for manual review
    - Debug mode (`?debug=1`) for troubleshooting
  - **Author**: Claude Code (Thoth's Algorithm)
  - **Created**: 2025-12-30
  - See detailed documentation below in "Helper Tools" section

### Transaction Management (One page per OTA):

- **`airbnb_transaction.php`** - Airbnb transactions
  - Import CSV from Airbnb
  - View reservation details
  - Track income and fees
  - Year-of-income reporting (the field we just fixed! "Año de Ingresos" 🎉)

- **`agoda_transaction.php`** - Agoda transactions
  - Similar functionality for Agoda platform

- **`booking_transaction.php`** - Booking.com transactions
  - Similar functionality for Booking.com platform

- **`casitamx_transaction.php`** - CasitaMX transactions
  - Similar functionality for CasitaMX platform

### Reservation Management (Property Management Systems):

- **`hostify_reserva.php`** - Hostify reservations
  - View reservation data from Hostify PMS
  - Track check-in/check-out dates
  - Monitor booking channels
  - Guest information and pricing

- **`cloudbeds_reserva.php`** - Cloudbeds reservations
  - View comprehensive reservation data from Cloudbeds PMS
  - Track guest details (email, phone, gender, country)
  - Manage room assignments and types
  - Financial tracking (accommodation total, deposits, balance due)
  - Reservation status monitoring
  - Document tracking (type, issuing country)

## AJAX Endpoints

Located in `/backoffice/ajax/`:

Standard jqGrid AJAX endpoints handle data loading for all grids. No custom AJAX files identified for this system - uses framework defaults.

## Database Tables

### Primary Tables:

#### Property Owners:
- **`propietario`** - Property owner master data
  - `propietario_id` (UUID primary key)
  - `departamento` - Property/apartment identifier
  - `propietario` - Owner name
  - `es_dueno` - Is owner (enum: Yes/No)
  - `inversionista` - Investor name (if different from owner)
  - `despacho` - Management office
  - `correo` - Email contact
  - Audit fields: alta_db, alta_por, ultimo_cambio, ultimo_cambio_por

#### Properties:
- **`propiedad`** - Property/building master data
  - `propiedad_id` (varchar(32) primary key)
  - `num_deptos` (int) - Number of apartments/units in property
  - `nombre_propiedad` (varchar(50)) - Property name
  - `propietario_id` (varchar(32) FK) - Foreign key to propietario table
  - `direccion` (varchar(100)) - Street address (format: "Street, Colonia")
  - `vale` (enum: 'Active','Inactive') - Property status
  - **Postal Code Data** (auto-filled from government catalog):
    - `codigo_postal` (varchar(5)) - 5-digit postal code (e.g., "06700")
    - `colonia` (varchar(100)) - Neighborhood/colony name
    - `estado` (varchar(5)) - State code (e.g., "DIF" for Mexico City)
    - `estado_descripcion` (varchar(100)) - Full state name (e.g., "CIUDAD DE MEXICO")
    - `municipio` (varchar(5)) - Municipality code
    - `municipio_descripcion` (varchar(100)) - Full municipality name
  - Audit fields: alta_db, alta_por, ultimo_cambio, ultimo_cambio_por
  - **Note**: Postal code fields were populated via fuzzy matching script (98.2% success rate, 112/114 properties matched)

#### Government Catalogs:
- **`codigo_postal`** - Official Mexican postal code catalog (SEPOMEX data)
  - 145,449 records total
  - Fields: codigo_postal, colonia, estado, estado_descripcion, municipio, municipio_descripcion, tipo_asentamiento
  - Used for auto-filling property postal data
  - Source: Government-provided official data

#### Transaction Tables (One per OTA):

- **`airbnb_transaction`** - Airbnb booking transactions
  - `airbnb_transaction_id` (UUID primary key)
  - **Date fields**: fecha, fecha_de_llegada_estimada, fecha_de_la_reservacion, fecha_de_inicio, fecha_de_finalizacion
  - **Guest info**: huesped, espacio, noches
  - **Booking info**: tipo, codigo_de_confirmacion, codigo_de_referencia, detalles
  - **Financial data**:
    - `moneda` - Currency (USD, MXN, etc.)
    - `monto` - Amount
    - `ingresos_recibidos` - Income received
    - `ingresos_brutos` - Gross income
    - `tc` - Exchange rate (tipo de cambio)
  - **Fees**:
    - `tarifa_de_servicio` - Service fee
    - `tarifa_por_pago_rapido` - Quick payment fee
    - `tarifa_de_limpieza` - Cleaning fee
  - **Taxes**: `impuestos_sobe_el_alojamiento` - Accommodation taxes
  - **Reporting**: `ano_de_ingresos` - Year of income (for tax reporting)
  - Audit fields: alta_db, alta_por, ultimo_cambio, ultimo_cambio_por

- **`agoda_transaction`** - Agoda transactions (similar schema)
- **`booking_transaction`** - Booking.com transactions (similar schema)
- **`casitamx_transaction`** - CasitaMX transactions (similar schema)

#### Reservation Tables (Property Management Systems):

- **`hostify_reserva`** - Hostify reservation data
  - `hostify_reserva_id` (UUID primary key)
  - **Reservation info**: confirmation_code, channel, anuncio
  - **Guest info**: guest_name
  - **Dates**: check_in, check_out
  - **Financial**: total_price, currency
  - **Stay details**: nights
  - Audit fields: alta_db, alta_por, ultimo_cambio, ultimo_cambio_por

- **`cloudbeds_reserva`** - Cloudbeds reservation data
  - `cloudbeds_reserva_id` (UUID primary key)
  - **Property**: property
  - **Guest info**: name, email, phone_number, gender, country, city, guest_status
  - **Reservation info**: reservation_number, third_party_confirmation_number, reservation_date, source, status, estimated_arrival_time, origin
  - **Guest count**: adults, children
  - **Room details**: room_number, room_type
  - **Dates**: check_in_date, check_out_date, nights
  - **Financial**: accommodation_total, amount_paid, grand_total, deposit, balance_due
  - **Documents**: type_of_document, document_issuing_country
  - Audit fields: alta_db, alta_por, ultimo_cambio, ultimo_cambio_por

### Framework Tables (Shared):
All systems share these iaCase framework tables:
- `iac_usr`, `iac_log`, `iac_table`, `iac_field_permission`, `iac_parametros`, etc.

## Key Features

### 1. Property Owner Management
- Master list of property owners
- Distinguish between owners and investors
- Track property/apartment assignments (departamento)
- Management office (despacho) associations
- Email contact management

### 2. Multi-Platform Transaction Tracking
Four separate transaction systems, one for each OTA:

**Airbnb:**
- Full transaction history
- Reservation tracking
- Guest information
- Nightly rates and total nights
- Service fees and cleaning fees
- Tax calculations
- Year-of-income reporting

**Agoda:**
- Same structure as Airbnb
- Platform-specific fields and fee structures

**Booking.com:**
- Same structure as Airbnb
- Platform-specific fields and pricing

**CasitaMX:**
- Same structure as Airbnb
- Local platform integration

### 3. Property Management System (PMS) Integration
Two separate PMS reservation systems:

**Hostify:**
- Reservation management from Hostify PMS
- Confirmation codes and channel tracking
- Guest name and stay duration
- Check-in/check-out dates
- Pricing and currency
- Listing/property (anuncio) identification

**Cloudbeds:**
- Comprehensive guest profile management
- Detailed contact information (email, phone)
- Guest demographics (gender, country, city)
- Reservation and third-party confirmation numbers
- Room assignments and room types
- Financial breakdown (accommodation, deposits, balance)
- Document tracking for compliance
- Reservation status and source tracking
- Guest status monitoring
- Estimated arrival time

### 4. Financial Tracking
Each transaction captures:
- **Income**: Gross income, received income, net amounts
- **Fees**: Service fees, quick payment fees, cleaning fees
- **Taxes**: Accommodation taxes
- **Currency**: Multi-currency support with exchange rates
- **Year tracking**: Income year for tax reporting purposes

### 5. Reservation Management
- Confirmation codes
- Reference codes
- Estimated arrival dates
- Check-in and check-out dates
- Number of nights
- Guest names
- Property/space identifiers

### 6. Search & Reporting
- Quick search enabled on all tables
- jqGrid with filters and column sorting
- Export capabilities (Excel, CSV, PDF)
- Year-over-year income comparison
- Multi-platform consolidated reporting

### 7. Audit Trail
All tables include:
- Creation timestamp and user (alta_db, alta_por)
- Last modification timestamp and user (ultimo_cambio, ultimo_cambio_por)
- Full activity logging through iaCase framework

## Technical Architecture

### Framework Integration
This system is built on the **iaCase framework**, which means:

- **Auto-generated UI**: Forms and grids automatically created from database schema
- **CRUD operations**: Insert/Update/Delete handled by framework
- **Validation**: Type checking, required fields, length limits all automatic
- **Permissions**: User-based, table-based, and field-based permissions
- **Logging**: All actions automatically logged
- **jqGrid integration**: Data tables with sorting, filtering, paging

### Display Mode
- Uses `cardex_window` mode (forms open in new window)
- Permissions: Full edit, delete, export, insert, list, read, update enabled
- Quick search enabled with automatic filters
- Display groups: Fields grouped logically (fecha, codigo, tarifa, ingresos, etc.)

### Multi-Currency Support
- Exchange rate tracking (campo `tc`)
- Currency identifier field (campo `moneda`)
- Automatic calculations based on exchange rates

## File Organization

```
/lamp/www/quantix/
├── app/
│   ├── app_propietario.php               ← PROPERTY OWNERS MODEL
│   ├── app_propiedad.php                 ← PROPERTIES MODEL
│   ├── app_airbnb_transaction.php        ← AIRBNB TRANSACTIONS
│   ├── app_agoda_transaction.php         ← AGODA TRANSACTIONS
│   ├── app_booking_transaction.php       ← BOOKING.COM TRANSACTIONS
│   ├── app_casitamx_transaction.php      ← CASITAMX TRANSACTIONS
│   ├── app_hostify_reserva.php           ← HOSTIFY RESERVATIONS
│   ├── app_cloudbeds_reserva.php         ← CLOUDBEDS RESERVATIONS
│   ├── app_iac_*.php                     ← SHARED FRAMEWORK FILES
│   └── [other shared utility files]
├── backoffice/
│   ├── propietario.php                   ← PROPERTY OWNERS UI
│   ├── propiedad.php                     ← PROPERTIES UI
│   ├── airbnb_transaction.php            ← AIRBNB UI
│   ├── agoda_transaction.php             ← AGODA UI
│   ├── booking_transaction.php           ← BOOKING.COM UI
│   ├── casitamx_transaction.php          ← CASITAMX UI
│   ├── hostify_reserva.php               ← HOSTIFY RESERVATIONS UI
│   ├── cloudbeds_reserva.php             ← CLOUDBEDS RESERVATIONS UI
│   └── helper/
│       ├── expand_propiedades_info.php          ← PROPERTY EXPANSION REPORT
│       ├── fill_postal_codes.php                ← FILL POSTAL CODES (WEB UI)
│       ├── link_propiedades_propietarios.php    ← LINK PROPERTIES ↔ OWNERS
│       ├── link_propiedades_pms.php             ← LINK PROPERTIES ↔ PMS (PREVIEW)
│       └── link_pms_propiedades.php             ← LINK PMS → PROPERTIES (PRODUCTION)
├── inc/                                   ← SHARED FRAMEWORK CORE
│   ├── iacase.php                        ← Core CRUD engine
│   ├── config.php                        ← Entry point
│   ├── ia_utilerias.php                  ← Database utilities (where we fixed "año")
│   └── [other framework files]
├── db/
│   └── enero_2025/
│       ├── fill_postal_codes_propiedad.php    ← POSTAL CODE FILL SCRIPT
│       └── SUMMARY.md                         ← POSTAL CODE PROJECT SUMMARY
└── [database]
    ├── propietario                        ← OWNERS TABLE
    ├── propiedad                          ← PROPERTIES TABLE (with postal codes!)
    ├── codigo_postal                      ← GOVERNMENT POSTAL CODE CATALOG
    ├── airbnb_transaction                 ← AIRBNB TABLE
    ├── agoda_transaction                  ← AGODA TABLE
    ├── booking_transaction                ← BOOKING.COM TABLE
    ├── casitamx_transaction               ← CASITAMX TABLE
    ├── hostify_reserva                    ← HOSTIFY RESERVATIONS TABLE
    └── cloudbeds_reserva                  ← CLOUDBEDS RESERVATIONS TABLE
```

## How It Works

### User Flow (Same for all transaction types):
1. User navigates to transaction page (e.g., `/backoffice/airbnb_transactions.php`)
2. Page loads corresponding app class (e.g., `app_airbnb_transactions`)
3. Class calls `process_action()` to handle URL parameter `?iah=`
   - `iah=l` or empty = List (jqGrid)
   - `iah=a` = Add form
   - `iah=e` = Edit form
   - `iah=r` = Read-only view
   - `iah=s` = Save (insert or update)
   - `iah=d` = Delete
4. Framework auto-generates HTML forms and jqGrid from database schema
5. Data displayed with proper Spanish labels (thanks to our UTF-8 fix! 🎉)

### Data Import Flow:
1. Download CSV from OTA platform (Airbnb, Agoda, Booking.com, or CasitaMX)
2. Use import tool (custom or CSV upload)
3. Map CSV columns to database fields
4. Validate data types and required fields
5. Insert transactions into appropriate table
6. Associate with property owner if applicable

### Reporting Flow:
1. Filter transactions by date range, property, or owner
2. Export filtered data
3. Generate summary reports (total income, fees, taxes)
4. Group by year using `ano_de_ingresos` field
5. Compare across platforms

## Common Operations

### Adding a Property Owner:
1. Navigate to Propietarios page
2. Click "Add" button
3. Fill in required fields:
   - Departamento (property identifier)
   - Propietario (owner name)
   - Es Dueño (Yes/No)
   - Inversionista (if different from owner)
   - Email
4. Save
5. System generates UUID and records audit data

### Recording a Transaction:
1. Navigate to appropriate transaction page (Airbnb, Agoda, etc.)
2. Click "Add" button
3. Fill in required fields:
   - Dates (transaction, arrival, check-in, check-out)
   - Guest info
   - Financial data (amounts, fees, taxes)
   - Currency and exchange rate
   - Year of income
4. Save
5. System generates UUID and records audit data

### Viewing Income by Year:
1. Open transaction grid
2. Use filter on "Año de Ingresos" column
3. Select year (e.g., 2025, 2024)
4. View filtered results
5. Export if needed

### Multi-Platform Comparison:
1. Open each platform's transaction page
2. Apply same date filters
3. Export each to Excel
4. Compare total income across platforms
5. Identify best-performing platform

## Helper Tools

### Exact Matcher v3.0: link_propiedades_propietarios.php

**Location**: `/backoffice/helper/link_propiedades_propietarios.php`

**Version**: v3.0 (Exact Character-by-Character Matching - January 2026)

**Purpose**: Deterministic exact matching to link propiedades (properties) to propietarios (owners) using normalized character-by-character comparison. This tool automatically establishes relationships with 100% confidence when source data is clean and properly formatted.

#### Use Cases
- **Initial Setup**: Bulk-link all properties to owners after data import
- **Data Cleanup**: Fix broken or missing propietario_id relationships
- **Migration**: Reconnect relationships after table restructuring
- **Verification**: Audit existing links and identify mismatches

#### Access
Navigate to the helper script via dashboard or direct URL:
```
https://dev-app.filemonprime.net/quantix/backoffice/helper/link_propiedades_propietarios.php
```

**Requirements**: Must be logged into Quantix (session required for database access)

#### v3.0 Simplification: Direction and Algorithm

**Evolution from v2.0**:
- **v1.0-v2.0**: Complex fuzzy matching with 10-tier cascade, semantic tokens, AI explanations
- **v3.0** (Current): Simplified to exact character-by-character matching only

**Direction**:
- Each **propiedad** (property) finds its matching **propietario** (owner)
- 212 propiedades → 107 propietarios (many-to-one allowed)
- **Why this direction**: Properties are the entities needing foreign keys populated

#### The Single-Tier Exact Matching Algorithm

v3.0 uses a deterministic exact match approach - no fuzzy logic, no AI, no multi-tier cascade:

**TIER 1: EXACT MATCH (100% Confidence)**
- Compare `propiedad.departamento` with `propietario.departamento`
- Text normalization applied to both sides:
  - Remove Spanish accents (Á→A, É→E, Í→I, Ó→O, Ú→U, Ñ→N)
  - Convert to lowercase
  - Trim whitespace
  - Normalize multi-space to single space
- Comparison: Simple `===` equality check
- Example: `"Álvaro Obregón 182"` matches `"alvaro obregon 182"` → 100%

**NO MATCH**
- If normalized strings don't match exactly: no link created
- `propietario_id` remains NULL
- Manual intervention required for edge cases

#### Normalization Function (v3.0)

The only intelligence in v3.0 is text normalization to handle common Spanish text variations:

```php
function normalize_text($text) {
    // Remove Spanish accents
    $accents = [
        'Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
        'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
        'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'
    ];
    $text = strtr($text, $accents);

    // Lowercase
    $text = mb_strtolower($text, 'UTF-8');

    // Normalize whitespace
    $text = trim(preg_replace('/\s+/', ' ', $text));

    return $text;
}
```

**Result**: `"Río Elba"`, `"Rio Elba"`, `"RIO ELBA"`, and `"  río  elba  "` all normalize to `"rio elba"` and match.

#### Database Schema (v3.0 - Simplified Metadata)

**Migration File**: `/db/enero_2025/05_add_propietario_match_columns.sql`

**Columns Updated by v3.0**:

```sql
-- Core foreign key (primary linkage)
propietario_id CHAR(36)
  -- UUID linking to propietario.propietario_id
  -- Set to matched owner's ID, or NULL if no match

-- Match metadata (simplified for exact matching)
propietario_nombre_propiedad TEXT
  -- Stores matched propietario.departamento for reference
  -- Example: "Alvaro Obregon 182"

propietario_match_tier INT
  -- Always 1 for exact matches (or NULL for no match)

propietario_match_confidence INT
  -- Always 100 for exact matches (or 0 for no match)

propietario_match_pattern VARCHAR(100)
  -- Always 'exact_match' (or 'no_match')

propietario_match_explanation TEXT
  -- Simple explanation: "Tier 1: Perfect Match\nMethod: tier1_exact\n..."

propietario_match_scores JSON
  -- All 100 for exact matches: {"overall":100, "street":100, "building":100, "unit":100}

propietario_match_timestamp TIMESTAMP
  -- When this match was created/updated
```

**Example Data After v3.0 Run**:
```sql
SELECT
    nombre_propiedad,
    departamento,
    propietario_nombre_propiedad,
    propietario_match_tier,
    propietario_match_confidence
FROM propiedad
WHERE propietario_id IS NOT NULL
LIMIT 3;

-- Results (all exact matches):
-- nombre: "Alvaro Obregon 182-A" | dept: "Alvaro Obregon 182" | matched: "Alvaro Obregon 182" | tier: 1 | conf: 100
```

#### Why v3.0 Works: Data Quality is Key

**Success Criteria**:
- Both `propiedad.departamento` and `propietario.departamento` must be clean
- Same text format used in both tables
- No typos, no abbreviations, no variations
- Manual data validation performed beforehand

**When It Fails**:
- Abbreviations: "Amer" vs "Amsterdam"
- Typos: "Obergon" vs "Obregon"
- Format differences: "Vicente 146" vs "Vicente Suárez 146"
- Extra data: "Vicente Suárez 146 Depto A" vs "Vicente Suárez 146"

**Solution**: Clean source data first, then run exact matcher. For messy data, manual cleanup or custom preprocessing required.

#### User Interface (v3.0)

**Clean Professional Theme**:
- Header: Simple gradient
- Focus on results, not branding

**Statistics Dashboard**:
- **Total Propiedades**: 212
- **Tier 1 (Exact Matches)**: 212 (100%)
- **Unmatched**: 0

**Results Table** (Simplified):
| Column | Description |
|--------|-------------|
| Propiedad | Property identifier (`propiedad.nombre_propiedad`) |
| Departamento (Source) | Property's departamento field |
| → | Direction arrow (propiedad → propietario) |
| Matched Propietario | Owner matched (`propietario.propietario`) |
| Matched Departamento | Owner's departamento field |
| Tier | Always 1 (exact match) |
| Confidence | Always 100% |
| Explanation | Simple: "Tier 1: Perfect Match" |

**Color Coding** (v3.0):
- **Green**: All matches (100% confidence)
- **Gray**: No match (if any exist)

**Actions Available** (v3.0):

1. **Preview Mode** (default)
   - Automatically processes all 212 propiedades
   - Shows all matches (exact only)
   - No database changes
   - Safe to run repeatedly

2. **Apply Matches**
   - Updates all exact matches (100% confidence)
   - Updates `propiedad.propietario_id` foreign key
   - Updates all 7 metadata columns
   - Requires confirmation dialog
   - Single action (no tiers to choose from)

3. **Export CSV**
   - Downloads matching results
   - Filename: `propiedades_matching_YYYY-MM-DD_HHMMSS.csv`
   - Includes: matched propietario, tier (always 1), confidence (always 100)
   - Offline review in Excel
   - Audit trail

#### Usage Examples (v3.0 Workflow)

**Step 1: Run Exact Matcher**
```
1. Open: https://dev-app.filemonprime.net/quantix/backoffice/helper/link_propiedades_propietarios.php
2. Page automatically processes all 212 propiedades
3. Review statistics:
   - Tier 1 (Exact): 212 propiedades (100% coverage)
   - Unmatched: 0
4. All matches shown in green (100% confidence)
```

**Step 2: Apply Matches**
```
1. Click "Apply Matches" button
2. Confirm action in dialog
3. Database updates all 212 propiedades
4. Success message: "Updated 212 propiedades"
5. All 7 metadata columns populated with tier=1, confidence=100
```

**Step 3: Verification**
```sql
-- Check all matches were applied
SELECT COUNT(*) as total_matched
FROM propiedad
WHERE propietario_id IS NOT NULL;
-- Expected: 212

-- View sample matches
SELECT
    nombre_propiedad,
    departamento,
    propietario_nombre_propiedad,
    propietario_match_confidence
FROM propiedad
WHERE propietario_id IS NOT NULL
LIMIT 10;
-- All should show confidence = 100
```

**Step 4: Handle Unmatched (if any exist)**
```
1. Query unmatched propiedades:
   SELECT * FROM propiedad WHERE propietario_id IS NULL;
2. Manually update departamento field to match propietario.departamento
3. Re-run matcher
4. Or manually assign via propiedad.php UI
```

#### Results Table Example (v3.0)

**v3.0 displays propiedades with their matched propietario**:

| Propiedad | Departamento (Source) | → | Matched Propietario | Matched Departamento | Tier | Confidence | Explanation |
|-----------|-----------------------|---|---------------------|----------------------|------|------------|-------------|
| Alvaro Obregon 182-A | Alvaro Obregon 182 | → | Adolfo Zavala | Alvaro Obregon 182 | 1 | 100% | "Tier 1: Perfect Match\nMethod: tier1_exact\nStreet: 100%, Building: 100%, Unit: 100%" |
| Vicente Suarez 146-204 | Vicente Suarez 146 | → | Maria Lopez | Vicente Suarez 146 | 1 | 100% | "Tier 1: Perfect Match\nMethod: tier1_exact\nStreet: 100%, Building: 100%, Unit: 100%" |
| Amsterdam 210-202 | Amsterdam 210 | → | Carlos Ruiz | Amsterdam 210 | 1 | 100% | "Tier 1: Perfect Match\nMethod: tier1_exact\nStreet: 100%, Building: 100%, Unit: 100%" |

#### Technical Implementation (v3.0)

**Core Algorithm Flow**:
```php
// SIMPLE DIRECTION: Loop through propiedades (212)
foreach ($propiedades as $propiedad) {
    $prop_dept_normalized = normalize_text($propiedad['departamento']);

    // Find matching propietario by exact departamento match
    $matched_propietario = null;
    foreach ($propietarios as $propietario) {
        $propietario_dept_normalized = normalize_text($propietario['departamento']);

        // EXACT character-by-character match
        if ($prop_dept_normalized === $propietario_dept_normalized) {
            $matched_propietario = $propietario;
            break; // Found match, stop searching
        }
    }

    if ($matched_propietario) {
        // Store match with tier=1, confidence=100
        $matches[] = [
            'propiedad' => $propiedad,
            'propietario' => $matched_propietario,
            'tier' => 1,
            'confidence' => 100,
            'pattern' => 'exact_match'
        ];
    } else {
        // No match found
        $matches[] = [
            'propiedad' => $propiedad,
            'propietario' => null,
            'tier' => 99,
            'confidence' => 0,
            'pattern' => 'no_match'
        ];
    }
}
```

**Database Queries** (v3.0):
```php
// Load all propiedades (entities to update)
$sql_propiedades = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_propiedades);

// Load all propietarios (lookup table)
$sql_propietarios = "SELECT * FROM propietario ORDER BY propietario";
$propietarios = ia_sqlArrayIndx($sql_propietarios);

// Apply update with DIRECT SQL (ALL 7 COLUMNS + FOREIGN KEY)
$sql = "UPDATE propiedad
        SET propietario_id = '{$propietario['propietario_id']}',
            propietario_nombre_propiedad = " . strit($propietario['departamento']) . ",
            propietario_match_tier = 1,
            propietario_match_confidence = 100,
            propietario_match_pattern = 'exact_match',
            propietario_match_explanation = " . strit($explanation) . ",
            propietario_match_scores = '{\"overall\":100,\"street\":100,\"building\":100,\"unit\":100}',
            propietario_match_timestamp = NOW()
        WHERE propiedad_id = '{$propiedad_id}'";
ia_query($sql); // CRITICAL: Must call ia_query() to execute!
```

**Performance**:
- 212 propiedades × 107 propietarios = 22,684 comparisons
- Text normalization (lightweight)
- Early exit on exact match (immediate break)
- Execution time: <1 second

**Session Requirement**:
- Script requires active Quantix session
- Uses framework's `ia_query()` and `ia_sqlArrayIndx()` functions
- Cannot be run via CLI or curl without session
- Must access via browser where logged in

**Safety Features**:
- Confirmation dialogs before applying changes
- Preview mode as default (no changes)
- CSV export for offline review
- All matches are 100% confidence (no bad matches possible)

#### Actual Results (v3.0)

**Coverage Achieved** (with clean source data):
- **Tier 1 (Exact Match)**: **212 propiedades (100% coverage)**
- **Unmatched**: **0 propiedades (0%)**

**Current Dataset Statistics**:
- Total propiedades: 212
- Total propietarios: 107
- Unique departamento values in propietarios: 106
- Duplicates allowed: Multiple propiedades can match same propietario

**Key Insight**: Clean data + exact matching = 100% success. When source data is properly validated, fuzzy logic is unnecessary.

#### Best Practices (v3.0)

1. **Clean Data First**: Ensure `propiedad.departamento` and `propietario.departamento` match exactly
2. **Preview Before Applying**: Always review matches in preview mode first
3. **Export CSV**: Keep audit trail of all matches for compliance
4. **Verify Foreign Keys**: After applying, check `propietario_id` column is populated
5. **Handle Edge Cases Manually**: For properties that don't match, update departamento field or assign manually
6. **Re-run After Data Changes**: If you add new properties or update departamento fields, re-run matcher
7. **Keep Departamento Consistent**: Use same naming convention in both tables
8. **Validate Before Import**: When importing new data, normalize departamento values first

#### Common Issues (v3.0)

**Issue**: Property doesn't match even though names look same
- **Cause**: Invisible characters, different spacing, or typos
- **Diagnosis**: Check normalized values:
  ```sql
  SELECT departamento, LOWER(TRIM(departamento)) as normalized
  FROM propiedad WHERE propietario_id IS NULL;
  ```
- **Fix**: Update departamento field to match exactly

**Issue**: Some properties matched, others didn't
- **Cause**: Inconsistent data entry (some abbreviated, some full names)
- **Diagnosis**: Compare unmatched departamentos with propietario table
- **Fix**: Standardize naming convention across both tables

**Issue**: Database update ran but foreign keys still NULL
- **Cause**: Script using `ia_update()` without calling `ia_query()`
- **Diagnosis**: Check script uses direct SQL with `ia_query()` call
- **Fix**: Update script to use pattern shown in "Database Queries" section above

**Issue**: Special characters causing mismatch
- **Cause**: Characters not covered by normalization (e.g., tildes, umlauts)
- **Diagnosis**: Manually inspect both departamento values
- **Fix**: Either remove special chars or extend normalize_text() function

**Issue**: Multiple propiedades match same propietario (is this wrong?)
- **Answer**: This is CORRECT behavior! One owner can have multiple properties
- **Example**: Owner "Juan Pérez" with departamento "Polanco 123" can match multiple units in that building

#### Key Functions (v3.0)

**Core Functions Implemented**:
1. **`normalize_text($text)`** - Text normalization (accents, case, whitespace)
2. **`find_propietario_for_propiedad($propiedad, $propietarios)`** - Core exact matching logic
3. **Database update via direct SQL** - Updates all 8 columns (FK + 7 metadata)

**Removed from v2.0**:
- ❌ Semantic token extraction (unnecessary for exact matching)
- ❌ Combo detection (no longer needed)
- ❌ Multi-dimensional scoring (always 100% or 0%)
- ❌ AI explanations (simple tier 1 message sufficient)
- ❌ Fuzzy matching tiers 2-4 (exact only)
- ❌ Similarity calculations (strict equality check)

**Lines of Code**:
- v2.0 THOTH: ~1,005 lines (complex AI logic)
- v3.0 Exact: ~800 lines (simplified)

#### Analytics Enabled (v3.0)

**Match Quality Report** (Simple - all exact matches):
```sql
SELECT
    propietario_match_tier as tier,
    COUNT(*) as count,
    AVG(propietario_match_confidence) as avg_confidence
FROM propiedad
WHERE propietario_id IS NOT NULL
GROUP BY propietario_match_tier;
-- Expected result: tier=1, count=212, avg_confidence=100.0
```

**Owner Portfolio Analysis**:
```sql
SELECT
    prop.propietario,
    prop.departamento,
    COUNT(p.propiedad_id) as total_properties,
    GROUP_CONCAT(p.nombre_propiedad SEPARATOR ', ') as properties
FROM propietario prop
LEFT JOIN propiedad p ON prop.propietario_id = p.propietario_id
GROUP BY prop.propietario_id
ORDER BY total_properties DESC;
-- Shows which owners have multiple properties
```

**Unmatched Properties** (Edge case handling):
```sql
SELECT
    nombre_propiedad,
    departamento
FROM propiedad
WHERE propietario_id IS NULL;
-- Should return 0 rows if data is clean
```

**Verification Query** (Post-matching audit):
```sql
SELECT
    COUNT(*) as total_propiedades,
    SUM(CASE WHEN propietario_id IS NOT NULL THEN 1 ELSE 0 END) as matched,
    SUM(CASE WHEN propietario_id IS NULL THEN 1 ELSE 0 END) as unmatched,
    ROUND(100.0 * SUM(CASE WHEN propietario_id IS NOT NULL THEN 1 ELSE 0 END) / COUNT(*), 2) as match_rate
FROM propiedad;
-- Expected: 212 total, 212 matched, 0 unmatched, 100.00% match_rate
```

#### Deployment Checklist (v3.0)

✅ Database migration executed (`/db/enero_2025/05_add_propietario_match_columns.sql`)
✅ All 7 metadata columns verified in `propiedad` table
✅ PHP file upgraded (v2.0 → v3.0 Exact Matcher - 800 lines)
✅ Syntax validated (no errors)
✅ Exact matching function implemented (`find_propietario_for_propiedad()`)
✅ Text normalization working (accents, case, whitespace)
✅ Direction corrected (propiedades → propietarios)
✅ Database update fixed (using direct SQL with `ia_query()`)
✅ Foreign key column (`propietario_id`) now populated correctly
✅ 100% match rate achieved (212/212 propiedades)
✅ Documentation updated (README)
✅ Production ready

#### Evolution Summary

**Version History**:
- **v1.0** (December 2025): Initial 4-tier fuzzy matcher (propiedades → propietarios)
- **v2.0 THOTH AI** (January 2026): Complex 10-tier AI system with semantic tokens, combo detection, multi-dimensional scoring (propietarios → propiedades, 1,005 lines)
- **v3.0 Exact Matcher** (January 2026): Simplified to single-tier exact matching after data cleanup (propiedades → propietarios, 800 lines)

**Why v3.0 is Better**:
- ✅ **100% accuracy** (vs 95-98% projected in v2.0)
- ✅ **Faster execution** (<1 second vs <2 seconds)
- ✅ **Simpler codebase** (800 lines vs 1,005 lines)
- ✅ **No false positives** (only exact matches accepted)
- ✅ **Easier maintenance** (no complex AI logic to debug)
- ✅ **Predictable results** (deterministic vs probabilistic)

**Lesson Learned**: When source data can be cleaned and standardized, exact matching beats fuzzy AI algorithms. The v2.0 complexity was solving a data quality problem that should have been fixed at the source.

---

### PMS Fuzzy Matchers: Link Reservations to Properties

**Location**:
- `/backoffice/helper/link_propiedades_pms.php` (Preview/Exploration)
- `/backoffice/helper/link_pms_propiedades.php` (Production/Updates)

**Created**: January 2026

**Purpose**: Link PMS (Property Management System) reservations from Cloudbeds and Hostify to individual properties using dual-target fuzzy matching algorithms.

#### The Problem

PMS systems store reservations with their own property/listing identifiers:
- **Cloudbeds**: `property` (building name) + `room_number` (unit)
- **Hostify**: `anuncio` (listing name, often includes address)

These need to be linked to the `propiedad` table for unified analytics and owner reporting.

**Data Scale:**
- 750 Cloudbeds reservations across 5 properties
- 2,263 Hostify reservations across 290 unique listings
- 350 individual property units to match against

#### Two Complementary Tools

**1. Preview Tool (link_propiedades_pms.php)**
- **Purpose**: Exploration and testing
- **Direction**: Property → PMS (find ONE match per property)
- **Database Updates**: None (preview only)
- **Use Case**: Understanding match quality before committing

**2. Production Tool (link_pms_propiedades.php)**
- **Purpose**: Actual database updates
- **Direction**: PMS → Property (EACH reservation finds match)
- **Database Updates**: Yes (UPDATE cloudbeds_reserva, hostify_reserva)
- **Use Case**: Production linking (10-30x more coverage)
- **Color**: Red (#e74c3c) - indicates caution

#### Key Difference: Direction Matters

**Preview (Property-Centric)**:
```
For each of 350 properties:
    Find best match in PMS data
    Result: 1 property → 1 listing
```
**Problem**: If multiple listings map to same property, only ONE gets linked

**Production (Reservation-Centric)**:
```
For each of 3,013 reservations:
    Find best matching property
    Result: Many reservations → 1 property ✓
```
**Solution**: ALL reservations find their property (many:1 allowed)

#### Matching Algorithms

**Cloudbeds (Hybrid Building + Unit)**:
1. Extract street name from property direccion and Cloudbeds property name
2. **Building Match** (60% weight): Fuzzy match on building/street name
3. **Unit Match** (40% weight): Extract unit numbers, compare
4. **Combined Score**: `(building * 0.6) + (unit * 0.4)`
5. **Tiers**:
   - Tier 1: 95-100% (perfect match)
   - Tier 2: 80-94% (high confidence)
   - Tier 3: 65-79% (medium confidence)
   - Tier 4: 40-64% (low confidence)

**Hostify (4-Tier Text Fuzzy)**:
1. **Tier 1 (100%)**: Exact normalized match
2. **Tier 2 (90%)**: Contains/segment match (handles pipe-separated lists)
3. **Tier 3 (70%)**: Similarity ≥85% using `similar_text()`
4. **Tier 4 (50-65%)**: Street name + partial unit match

#### Database Schema Changes

**Fields Added** (January 2026):
```sql
ALTER TABLE cloudbeds_reserva
ADD COLUMN propiedad_id VARCHAR(32) NULL
COMMENT 'FK to propiedad table';

ALTER TABLE hostify_reserva
ADD COLUMN propiedad_id VARCHAR(32) NULL
COMMENT 'FK to propiedad table';
```

#### User Interface

**Tabbed Layout**:
- Summary tab: Overall statistics, action buttons
- Cloudbeds tab: 750 reservations with match details
- Hostify tab: 2,263 reservations with match details

**Statistics Dashboard**:
- Total reservations processed
- Matches per tier (T1, T2, T3, T4)
- High confidence count (≥80%)
- Unmatched count

**Actions** (Production Tool Only):
- Apply High Confidence (≥80%) - Safest
- Apply All Matches - Includes low confidence
- Export CSV - Offline review

**Color Coding**:
- Green: Tier 1-2 (high confidence ≥80%)
- Yellow: Tier 3 (medium 65-79%)
- Orange: Tier 4 (low 40-64%)
- Gray: Unmatched

#### Usage Workflow

**Step 1: Preview**
```
1. Open: link_propiedades_pms.php
2. Review match quality and statistics
3. Export CSV for manual review if needed
4. Understand matching patterns
```

**Step 2: Production**
```
1. Open: link_pms_propiedades.php
2. Review summary statistics
3. Click "Apply High Confidence (≥80%)"
4. Confirm action (updates database)
5. Verify results in database
```

**Step 3: Verification**
```sql
-- Check Cloudbeds links
SELECT COUNT(*) FROM cloudbeds_reserva
WHERE propiedad_id IS NOT NULL;

-- Check Hostify links
SELECT COUNT(*) FROM hostify_reserva
WHERE propiedad_id IS NOT NULL;

-- View linked reservations for a property
SELECT * FROM hostify_reserva
WHERE propiedad_id = 'uuid-property-123';
```

#### Expected Results

**High Confidence Matches** (≥80%):
- Estimated 400-600 Cloudbeds reservations
- Estimated 1,500-2,000 Hostify reservations
- Total: 2,000-2,500+ reservations linked

**Benefits**:
- Revenue analysis per property
- Occupancy tracking
- Owner reports (all transactions for their units)
- Channel performance comparison

#### Analytics Enabled

**Revenue per Property**:
```sql
SELECT
    p.nombre_propiedad,
    COUNT(hr.hostify_reserva_id) as bookings,
    SUM(hr.total_price) as revenue
FROM propiedad p
LEFT JOIN hostify_reserva hr ON p.propiedad_id = hr.propiedad_id
GROUP BY p.propiedad_id
ORDER BY revenue DESC;
```

**Occupancy by Month**:
```sql
SELECT
    p.nombre_propiedad,
    DATE_FORMAT(hr.check_in, '%Y-%m') as month,
    COUNT(*) as reservations,
    SUM(hr.nights) as total_nights
FROM propiedad p
JOIN hostify_reserva hr ON p.propiedad_id = hr.propiedad_id
GROUP BY p.propiedad_id, month
ORDER BY month DESC;
```

**Owner Consolidated Report**:
```sql
SELECT
    prop.propietario,
    p.nombre_propiedad,
    COUNT(DISTINCT hr.hostify_reserva_id) as hostify_res,
    COUNT(DISTINCT cbr.cloudbeds_reserva_id) as cloudbeds_res,
    SUM(hr.total_price) as total_revenue
FROM propietario prop
JOIN propiedad p ON prop.propietario_id = p.propietario_id
LEFT JOIN hostify_reserva hr ON p.propiedad_id = hr.propiedad_id
LEFT JOIN cloudbeds_reserva cbr ON p.propiedad_id = cbr.propiedad_id
GROUP BY prop.propietario_id, p.propiedad_id;
```

---

### Postal Code Fill Tool (Web UI)

**Location**: `/backoffice/helper/fill_postal_codes.php`

**Created**: January 2026

**Purpose**: Web-based UI for filling postal code data in `propiedad` table using fuzzy matching against SEPOMEX government catalog.

#### Overview

This is the **web version** of the CLI script `/db/enero_2025/fill_postal_codes_propiedad.php`. It provides a user-friendly interface for matching property addresses to the official Mexican postal code catalog.

**Data Processing:**
- Extracts colonia name from `propiedad.direccion` (format: "Street, Colonia")
- Matches against 145,449 SEPOMEX postal codes
- Fills 6 fields: codigo_postal, colonia, estado, estado_descripcion, municipio, municipio_descripcion

#### 3-Tier Matching Algorithm

**Tier 1 - Manual Overrides (100% confidence)**:
- Hard-coded mappings for known edge cases
- Examples: "CDMX" → Escandón, "STA MARIA LA RIBERA" → Santa María la Ribera
- Handles abbreviations and non-standard names

**Tier 2 - Exact Match (100% confidence)**:
- Normalizes colonia names (remove accents, uppercase, trim)
- SQL: `UPPER(TRIM(colonia)) = UPPER(TRIM(catalog.colonia))`
- Most properties match here (76.3% in original run)

**Tier 3 - Fuzzy LIKE Match (90% confidence)**:
- Uses SQL LIKE for partial matches
- Prefers shortest match (most specific)
- Example: "Polanco" matches "Polanco I Sección"

#### User Interface

**Statistics Dashboard**:
- Total properties
- Exact matches
- Fuzzy matches
- Manual overrides
- Ambiguous (needs review)
- Not found
- Skipped (no address)
- High confidence count (≥90%)

**Actions**:
- **Preview** (default): Show all matches, no changes
- **Apply High Confidence (≥90%)**: Safe automated updates
- **Apply All**: Includes fuzzy matches (use with caution)
- **Export CSV**: Offline review and manual editing

**Color Coding**:
- Green: Exact matches
- Blue: Fuzzy matches
- Yellow: Manual overrides
- Red: Ambiguous (multiple postal codes found)
- Gray: No match

#### Usage Example

```
1. Access: helper/fill_postal_codes.php
2. Review statistics (e.g., "95 exact, 10 fuzzy, 3 not found")
3. Click "Apply High Confidence (≥90%)"
4. Confirm update
5. Export CSV to review remaining unmatched
6. Manually update those 3 properties in UI
```

#### Expected Results

Based on original CLI run:
- **Success Rate**: 98.2% (112 of 114 properties)
- **Exact Matches**: ~87 properties (76.3%)
- **Fuzzy Matches**: ~6 properties (5.3%)
- **Manual Overrides**: ~19 properties (16.7%)
- **Failed**: 0-2 properties (no address data)

**Fields Populated**:
```
codigo_postal: "06700"
colonia: "Roma Norte"
estado: "DIF"
estado_descripcion: "CIUDAD DE MEXICO"
municipio: "015"
municipio_descripcion: "CUAUHTEMOC"
```

---

### Property Expansion Report

**Location**: `/backoffice/helper/expand_propiedades_info.php`

**Created**: January 2026

**Purpose**: Informational report showing the results of the property expansion from building-level to unit-level granularity.

#### What It Shows

**Statistics**:
- Total units (350)
- Unique buildings
- Single vs multi-unit properties
- Units with unit numbers
- Units with floor data
- Units with postal codes

**Data Tables**:
- Sample properties (first 50) with all unit fields
- Buildings with multiple units
- Transformation details

#### Not an Action Tool

This is a **reporting/informational tool only**. It does NOT perform the expansion - that was done by the CLI script `/db/enero_2025/02_expand_propiedades_to_units.php`.

**What It's For**:
- Understanding the expansion that was performed
- Viewing current unit-level data
- Seeing unit distribution across buildings
- Verifying expansion results

**Links To**:
- Original CLI expansion script
- Postal code fill script
- Documentation

#### Sample Statistics

**Transformation Results**:
- Original: 114 properties (buildings)
- Final: 350 individual units
- Method: In-place expansion using `num_deptos`
- Naming: Letters (A-J) for small buildings, numbers (01-24) for large

**Unit Distribution Examples**:
- "Medellin 148": 24 units (01-24)
- "Baranda OCHO": 10 units (A-J)
- "Queretaro 121": Single unit (kept as-is)

---

## Integration Points

### With Framework:
- Uses `iacase_base` parent class for all CRUD
- Leverages `appRelate` for schema metadata
- Respects `iac_field_permission` for field visibility
- Logs to `iac_log` automatically
- Uses `to_label()` function for Spanish field labels (where we fixed UTF-8!)

### With External Systems:
- CSV import from OTA platforms
- Potential API integration with booking platforms
- Export to Excel for accounting software
- Email notifications for new bookings

### With Other Quantix Systems:
- Shares user authentication (iac_usr)
- Shares permission system
- Shares audit logging
- Independent business logic from other systems

## Data Integration - Postal Code System

### Overview
The system integrates with the official Mexican government postal code catalog (SEPOMEX data) to automatically populate geographic information for properties.

### Government Postal Code Catalog
- **Table**: `codigo_postal`
- **Records**: 145,449 official postal codes
- **Source**: SEPOMEX (Mexican Postal Service)
- **Coverage**: Complete nationwide coverage of Mexico
- **Fields**:
  - `codigo_postal` - 5-digit postal code
  - `colonia` - Neighborhood/colony name
  - `estado` - State code (2-3 characters)
  - `estado_descripcion` - Full state name
  - `municipio` - Municipality code
  - `municipio_descripcion` - Full municipality name
  - `tipo_asentamiento` - Settlement type (e.g., Colonia, Fraccionamiento)

### Automated Postal Code Fill Project (January 2025)

**Objective**: Auto-populate the 6 new postal code fields in the `propiedad` table by matching existing address data against the government catalog.

**Challenge**:
- 113 properties with address data (1 property had no address)
- Address format: "Street Address, Colonia" (comma-separated)
- Needed to extract colonia name and match against 145,449 postal codes
- Handle variations: accents, capitalization, sections (e.g., "Polanco I Sección")

**Solution - Three-Tier Fuzzy Matching Algorithm**:

1. **Tier 1 - Manual Overrides** (Highest Priority)
   - Hard-coded mappings for known edge cases
   - Examples:
     - "CDMX" → Escandón I Sección (11800)
     - "STA MARIA LA RIBERA" → Santa María la Ribera (06400)
     - "CUAUTEMOC" → Cuauhtémoc (06000)
   - Handles abbreviations and non-standard names
   - **Results**: 19 properties matched (16.7%)

2. **Tier 2 - Exact Match** (After Normalization)
   - Normalize both direccion and catalog colonia names:
     - Remove accents (Á→A, É→E, etc.)
     - Convert to uppercase
     - Trim whitespace
   - SQL: `UPPER(TRIM(colonia_extracted)) = UPPER(TRIM(codigo_postal.colonia))`
   - **Results**: 87 properties matched (76.3%)

3. **Tier 3 - Fuzzy LIKE Match** (Fallback)
   - Use SQL LIKE for partial matches: `colonia LIKE '%extracted%'`
   - Prefer shortest match (most specific)
   - Example: "Polanco" matches both "Polanco I Sección" and "Polanco V Sección" → choose first
   - **Results**: 6 properties matched (5.3%)

**Implementation**:
- **Script**: `/db/enero_2025/fill_postal_codes_propiedad.php`
- **Standalone PHP**: Minimal dependencies, direct MySQL connection
- **Preview Mode**: `php fill_postal_codes_propiedad.php --preview` (no changes)
- **Execute Mode**: `php fill_postal_codes_propiedad.php --execute` (apply updates)
- **Reporting**: Generates JSON and CSV reports with match details

**Results**:
- **Total Properties**: 114
- **Properties with Address**: 113
- **Successfully Matched**: 112 (98.2% success rate)
- **Unmatched**: 2 (no address data)
- **Match Breakdown**:
  - Exact matches: 87 (76.3%)
  - Fuzzy matches: 6 (5.3%)
  - Manual overrides: 19 (16.7%)
  - Failed: 0 (0%)

**Fields Populated**:
For each matched property, the following 6 fields were filled:
- `codigo_postal` - e.g., "06700"
- `colonia` - e.g., "Roma Norte"
- `estado` - e.g., "DIF" (Ciudad de México)
- `estado_descripcion` - e.g., "CIUDAD DE MEXICO"
- `municipio` - e.g., "015"
- `municipio_descripcion` - e.g., "CUAUHTEMOC"

**Collation Handling**:
- **Challenge**: `propiedad.direccion` uses `utf8mb4_unicode_ci`, catalog uses `utf8mb4_0900_ai_ci`
- **Solution**: Used `COLLATE utf8mb4_0900_ai_ci` in extraction query to avoid "Illegal mix of collations" error

**Documentation**:
- **Summary Report**: `/db/enero_2025/SUMMARY.md`
- **Execution Report**: `/db/enero_2025/postal_codes_fill_report_[timestamp].json`
- Contains: Match method used, postal code assigned, colonia matched, success/error status

**Top Colonias** (Most common in the dataset):
1. Roma Norte - 37 properties
2. Condesa - 36 properties
3. Polanco - 10 properties
4. Juárez - 9 properties
5. Del Valle - 6 properties

**Maintenance**:
- Script is idempotent (can be re-run safely)
- Manual overrides should be updated in `getManualOverrides()` function for new edge cases
- Government catalog should be refreshed periodically from SEPOMEX official data

### Using Postal Code Data

**In Forms**:
- Postal code fields display as read-only (auto-populated)
- Users enter `direccion` (street address), postal data fills automatically on save
- Helper script can trigger re-matching if addresses are corrected

**In Reports**:
- Filter properties by colonia, estado, or municipio
- Group income reports by geographic region
- Analyze performance by neighborhood

**Future Enhancements**:
- Real-time postal code lookup API integration
- Auto-suggest colonias as user types address
- Geocoding integration for map visualization
- Distance calculations between properties

---

## Data Integration - Propiedad Expansion

### Overview

In January 2025, the `propiedad` table underwent a fundamental structural transformation from **property-level** to **unit-level** granularity. This change enables individual apartment/unit tracking within multi-unit buildings while maintaining compatibility with existing transactions and postal code integrations.

**Transformation Summary:**
- **Before**: 114 rows representing buildings (some with 24+ units tracked as single row)
- **After**: 350 rows representing individual apartments/units
- **Method**: In-place expansion using `num_deptos` as multiplier
- **Data Preservation**: All postal code data, addresses, and owner links inherited by child units
- **Transaction Safety**: Existing `propiedad_id` references remain valid

### The Challenge

**Original Structure Limitations:**

The original schema stored multi-unit buildings as single rows:

```sql
-- BEFORE: One row for entire building
propiedad_id     | nombre_propiedad    | num_deptos | direccion
-----------------+---------------------+------------+------------------
uuid-medellin148 | Medellin 148        | 24         | Medellin 148
uuid-queretaro   | Queretaro 121       | 10         | Queretaro 121
```

**Problems:**
1. **No unit-specific data**: Cannot track individual apartment characteristics (floor, orientation, square meters)
2. **Aggregate income only**: Rental income lumped at building level
3. **Cannot track vacancies**: No way to mark individual units as vacant vs occupied
4. **Limited reporting**: Cannot analyze performance by floor, unit size, or orientation
5. **Owner assignment ambiguity**: Multi-owner buildings couldn't assign owners per unit

### The Solution

**In-Place Expansion Approach:**

Transform each multi-unit property into individual unit rows while preserving data integrity:

```sql
-- AFTER: 24 individual rows for same building
propiedad_id     | nombre_propiedad    | numero_unidad | num_deptos | piso | direccion
-----------------+---------------------+---------------+------------+------+------------------
uuid-medellin148 | Medellin 148        | 01            | 1          | 1    | Medellin 148
uuid-new-guid-1  | Medellin 148        | 02            | 1          | 1    | Medellin 148
uuid-new-guid-2  | Medellin 148        | 03            | 1          | 1    | Medellin 148
...
uuid-new-guid-23 | Medellin 148        | 24            | 1          | 3    | Medellin 148
```

**Key Design Decisions:**

1. **Single Table**: No separate `departamento` table - keeps schema simple
2. **UUID Preservation**: Original `propiedad_id` kept for first unit - preserves transaction links
3. **New UUIDs**: New units get `ia_guid()` generated IDs
4. **Smart Numbering**: Unit numbers extracted from `nombre_propiedad` or auto-generated
5. **Data Inheritance**: All postal code fields copied from parent property
6. **Normalization**: All `num_deptos` set to 1 (now represents single units)

### New Schema Fields

**11 New Columns Added:**

```sql
-- Unit Identification
numero_unidad VARCHAR(10)           -- Unit number: 01, 02, A, B, 101, 201
piso INT                            -- Floor number (extracted from unit number)
tipo_unidad ENUM(...)               -- Departamento, Casa, Estudio, Penthouse, Loft

-- Unit Characteristics
metros_cuadrados DECIMAL(8,2)       -- Square meters
recamaras TINYINT UNSIGNED          -- Bedrooms (0-255)
banos DECIMAL(3,1) UNSIGNED         -- Bathrooms (allows 1.5, 2.5)
orientacion VARCHAR(50)             -- Norte, Sur, Este, Oeste, Noreste, etc.

-- Amenities
estacionamientos TINYINT UNSIGNED   -- Parking spaces
bodega BOOLEAN                      -- Storage unit included
balcon BOOLEAN                      -- Has balcony
amueblado ENUM('Si','No','Parcial') -- Furnished status
```

### Expansion Algorithm

**Script**: `/db/enero_2025/02_expand_propiedades_to_units.php`

**Core Logic:**

```php
// 1. Query all properties
$propiedades = ia_sqlArrayIndx("SELECT * FROM propiedad ORDER BY nombre_propiedad");

foreach ($propiedades as $prop) {
    $num_units = (int)$prop['num_deptos'];

    if ($num_units <= 1) {
        // Single-unit property - just normalize num_deptos to 1
        UPDATE_IN_PLACE($prop['propiedad_id']);
    } else {
        // Multi-unit property - expand into individual rows
        for ($i = 1; $i <= $num_units; $i++) {
            $unit_number = generateUnitNumber($i, $num_units, $prop['nombre_propiedad']);
            $floor = extractFloorFromUnit($unit_number);
            $tipo = detectTipoUnidad($prop['nombre_propiedad']);

            if ($i === 1) {
                // Update original row (preserves propiedad_id)
                UPDATE_FIRST_UNIT($prop['propiedad_id'], $unit_number, $floor, $tipo);
            } else {
                // Insert new rows with new UUIDs
                $new_id = ia_guid('propiedad_expansion');
                INSERT_NEW_UNIT($new_id, $prop, $unit_number, $floor, $tipo);
            }
        }
    }
}
```

**Smart Unit Numbering Function:**

```php
function generateUnitNumber($index, $total_units, $nombre_propiedad) {
    // 1. Check if unit number already in property name
    if (preg_match('/\b(\d{3,4}|[A-Z])\s*$/', $nombre_propiedad, $matches)) {
        return $matches[1]; // Extract existing: "Queretaro 121 - 101" → "101"
    }

    // 2. For small buildings (≤10 units), use letters
    if ($total_units <= 10 && $total_units > 1) {
        return chr(64 + $index); // A=65, B=66, ..., J=74
    }

    // 3. For larger buildings, use zero-padded numbers
    $padding = strlen((string)$total_units);
    return str_pad($index, $padding, '0', STR_PAD_LEFT);
    // 24 units → "01", "02", ..., "24"
}
```

**Floor Extraction Logic:**

```php
function extractFloorFromUnit($numero_unidad) {
    // Pattern 1: 3-digit Mexican standard (101 = floor 1, unit 01)
    if (preg_match('/^(\d)(\d{2})$/', $numero_unidad, $matches)) {
        return (int)$matches[1]; // "301" → floor 3
    }

    // Pattern 2: 4-digit for tall buildings (1001 = floor 10, unit 01)
    if (preg_match('/^(\d{2})(\d{2})$/', $numero_unidad, $matches)) {
        return (int)$matches[1]; // "1201" → floor 12
    }

    // Pattern 3: Letter-based (ground floor)
    if (preg_match('/^[A-Z]$/', $numero_unidad)) {
        return 1; // "A", "B" → floor 1
    }

    // Pattern 4: Simple sequential numbers (assume floor 1)
    if (preg_match('/^\d{1,2}$/', $numero_unidad)) {
        return 1; // "01", "02" → floor 1
    }

    return null; // Unknown pattern
}
```

**Type Detection:**

```php
function detectTipoUnidad($nombre_propiedad) {
    $nombre_lower = mb_strtolower($nombre_propiedad, 'UTF-8');

    if (strpos($nombre_lower, 'penthouse') !== false) return 'Penthouse';
    if (strpos($nombre_lower, 'casa') !== false) return 'Casa';
    if (strpos($nombre_lower, 'estudio') !== false) return 'Estudio';
    if (strpos($nombre_lower, 'loft') !== false) return 'Loft';

    return 'Departamento'; // Default
}
```

### Execution Details

**Step 1: Schema Migration**

```bash
# Add 11 new columns to propiedad table
/lamp/mysql/bin/mysql -u root -pM@chiavell1 \
  --socket=/lamp/mysql/mysql.sock quantix \
  < /lamp/www/quantix/db/enero_2025/01_alter_propiedad_add_unit_fields.sql
```

**Step 2: Preview Expansion (Dry Run)**

```bash
# Preview mode - no database changes
curl -s https://dev-app.filemonprime.net/quantix/db/enero_2025/02_expand_propiedades_to_units.php?mode=preview
```

**Output:**
```
=== PREVIEW MODE ===
Property: Medellin 148 (24 units)
  → Will update existing row with unit: 01
  → Will create 23 new rows: 02, 03, ..., 24

Property: Queretaro 121 (10 units)
  → Will update existing row with unit: A
  → Will create 9 new rows: B, C, ..., J

Total: 114 properties → 350 units
```

**Step 3: Execute Expansion**

```bash
# Execute mode - applies changes to database
curl -s https://dev-app.filemonprime.net/quantix/db/enero_2025/02_expand_propiedades_to_units.php?mode=execute
```

**Step 4: Verify Results**

```bash
# Report mode - shows actual database state
curl -s https://dev-app.filemonprime.net/quantix/db/enero_2025/02_expand_propiedades_to_units.php?mode=report
```

**Critical Implementation Detail:**

The framework functions `ia_update()` and `ia_insert()` **return SQL strings** but do NOT execute them. The script must call `ia_query()` to execute:

```php
// ❌ WRONG - Returns SQL string, doesn't execute
ia_update('propiedad', $values, $where);

// ✅ CORRECT - Execute the returned SQL
$sql = ia_update('propiedad', $values, $where);
ia_query($sql);
```

### Results

**Quantitative:**

| Metric | Before | After | Change |
|--------|--------|-------|--------|
| Total rows | 114 | 350 | +207% |
| Multi-unit properties | 52 | 0 | -100% |
| Single-unit properties | 62 | 350 | +465% |
| Average units/property | 3.07 | 1.00 | -67% |
| Max units in one property | 24 | 1 | -96% |
| New UUIDs generated | 0 | 236 | +236 |

**Qualitative:**

✅ **Data Preserved:**
- All 114 original `propiedad_id` values retained (first units)
- 100% postal code data inheritance (codigo_postal, colonia, estado, municipio)
- All owner links (`propietario_id`) maintained
- All addresses and property names copied to child units

✅ **Data Enhanced:**
- 350 units now have individual `numero_unidad` identifiers
- 312 units have extracted `piso` values (89% success rate)
- 350 units classified by `tipo_unidad`
- Ready for unit-specific characteristics (metros², recamaras, baños)

✅ **System Compatibility:**
- Existing transactions still link via original `propiedad_id` values
- Reports continue to work (now show unit-level detail)
- Forms updated with new fields via `app_propiedad.php` changes
- jqGrid automatically includes new columns

### Usage Examples

**Query Individual Units:**

```sql
-- Find all units in Medellin 148 building
SELECT numero_unidad, piso, tipo_unidad, metros_cuadrados
FROM propiedad
WHERE nombre_propiedad = 'Medellin 148'
ORDER BY numero_unidad;
```

**Filter by Floor:**

```sql
-- All units on 2nd floor
SELECT nombre_propiedad, numero_unidad, direccion
FROM propiedad
WHERE piso = 2
ORDER BY nombre_propiedad, numero_unidad;
```

**Analyze Unit Distribution:**

```sql
-- Count units per building
SELECT
    nombre_propiedad,
    COUNT(*) as total_units,
    MIN(numero_unidad) as first_unit,
    MAX(numero_unidad) as last_unit
FROM propiedad
GROUP BY nombre_propiedad
HAVING COUNT(*) > 1
ORDER BY total_units DESC;
```

**Find Specific Unit:**

```sql
-- Direct unit lookup
SELECT * FROM propiedad
WHERE nombre_propiedad = 'Queretaro 121'
  AND numero_unidad = 'C';
```

### Future Enhancements

**Phase 1 - Data Enrichment:**
- Populate `metros_cuadrados`, `recamaras`, `banos` from existing records
- Add `orientacion` data (Norte, Sur, Este, Oeste)
- Set `amueblado` status based on transaction history

**Phase 2 - Owner Assignment:**
- Link individual units to specific owners (multi-owner buildings)
- Update `propietario_id` at unit level where known
- Create helper tool for bulk owner assignment

**Phase 3 - Vacancy Tracking:**
- Add `estado_ocupacion` field (Ocupado, Vacante, Mantenimiento)
- Add `fecha_ultimo_inquilino` timestamp
- Dashboard showing vacancy rates by building/floor

**Phase 4 - Financial Reporting:**
- Income reports by unit (not just building)
- Occupancy rate analysis per floor
- Performance comparison: high floors vs low floors

**Phase 5 - UI Enhancements:**
- Visual floor plan builder in backoffice
- Drag-and-drop unit assignment to owners
- Heat map showing unit performance

### Rollback Procedure

**Safety First:**

A complete backup table was created before expansion:

```sql
-- Backup created automatically by expansion script
CREATE TABLE propiedad_backup_20250102 LIKE propiedad;
INSERT INTO propiedad_backup_20250102 SELECT * FROM propiedad;
```

**Rollback Script**: `/db/enero_2025/03_rollback_propiedad_expansion.sql`

**Execute Rollback:**

```bash
# Restore original 114-row state
/lamp/mysql/bin/mysql -u root -pM@chiavell1 \
  --socket=/lamp/mysql/mysql.sock quantix \
  < /lamp/www/quantix/db/enero_2025/03_rollback_propiedad_expansion.sql
```

**Rollback SQL:**

```sql
-- Drop expanded table
DROP TABLE IF EXISTS propiedad;

-- Recreate from backup (preserves structure)
CREATE TABLE propiedad LIKE propiedad_backup_20250102;

-- Restore original data
INSERT INTO propiedad SELECT * FROM propiedad_backup_20250102;

-- Verify restoration
SELECT COUNT(*) as restored_count FROM propiedad; -- Should be 114
```

**⚠️ Warning:** Rollback will lose:
- All 236 new UUIDs generated during expansion
- Any manual data entered into new unit-specific fields
- Any new transactions created against new units (post-expansion)

**Recommendation**: Test rollback in development environment first. If expansion is live for >24 hours, coordinate rollback with stakeholders.

---

## Maintenance Notes

### Do NOT Edit:
- `campos_default()` method - Auto-generated from database schema
- Framework files (`app_iac_*.php`, `/inc/*`)
- `/inc/ia_utilerias.php` unless fixing encoding issues 😉

### Safe to Customize:
- `campos_final()` method - Field ordering, display groups, custom validation
- Permission settings in constructor
- Hook methods (insert_*, update_*, delete_* overrides)
- Field display groups (fecha, codigo, tarifa, ingresos groups)

### Database Changes:
1. Alter table in database
2. Regenerate `appRelateBase.php` from schema
3. Update `campos_final()` if field order changes
4. Clear opcache if in production

### Adding a New OTA Platform:
1. Create table `[platform]_transactions` with similar schema
2. Create `app_[platform]_transactions.php` model
3. Create `/backoffice/[platform]_transactions.php` UI page
4. Add to dashboard menu
5. Test CRUD operations
6. Add import logic

## Security

- **Permissions**: Controlled by `$this->permiso_*` properties
- **User type checks**: Can restrict by Rony/Power/Normal user types
- **Field-level security**: Via `iac_field_permission` table
- **SQL injection prevention**: Framework uses parameterized queries
- **Session management**: Automatic timeout and regeneration
- **Audit logging**: All changes tracked with user and timestamp
- **Financial data protection**: Read-only fields for calculated amounts

## Recent Updates

### Propiedad Table Expansion (January 2025)
Transformed the `propiedad` table from property-level to unit-level granularity for individual apartment tracking.

**Transformation:**
- Original: 114 property rows (buildings with multiple units)
- Final: 350 individual apartment/unit rows
- Method: In-place expansion using `num_deptos` as multiplier
- Algorithm: Thoth's Algorithm (4-tier smart expansion)

**New Fields Added:**
- `numero_unidad` - Unit number (A-J for small buildings, 01-24 for large)
- `piso` - Floor number (auto-extracted from unit number: 101→1, 201→2)
- `tipo_unidad` - Unit type (Departamento, Casa, Penthouse, Loft, Estudio)
- `orientacion` - Facing direction (Norte, Sur, Este, Oeste, Noreste, etc.)
- `metros_cuadrados` - Square meters (M²)
- `recamaras` - Number of bedrooms
- `banos` - Number of bathrooms (supports half baths: 2.5 = 2 full + 1 half)
- `estacionamientos` - Number of parking spaces
- `bodega` - Has storage unit (boolean)
- `balcon` - Has balcony/terrace (boolean)
- `amueblado` - Furnished status (Si, No, Parcial)

**Smart Expansion Algorithm:**
- Letter naming (A-J) for buildings with ≤10 units
- Zero-padded numbers (01-24) for larger buildings
- Floor extraction: "101" → floor 1, "201" → floor 2, "1001" → floor 10
- Type detection from name: "Casa X" → type: Casa

**Results:**
- 83 single-unit properties → kept as-is, unit numbers added
- 31 multi-unit properties → expanded into 236 additional rows
- 100% success rate (all 350 apartments accounted for)
- All `num_deptos` normalized to 1

**Scripts Created:**
- `/db/enero_2025/01_alter_propiedad_add_unit_fields.sql` - Schema changes
- `/db/enero_2025/02_expand_propiedades_to_units.php` - Expansion script (preview/execute/report modes)
- `/db/enero_2025/03_rollback_propiedad_expansion.sql` - Rollback capability
- Report: `/db/enero_2025/propiedad_expansion_report_2026-01-02_192737.csv`

**Examples:**
- "Medellin 148" (24 units) → "Medellin 148 - 01" through "Medellin 148 - 24"
- "Baranda OCHO" (10 units) → "Baranda OCHO - A" through "Baranda OCHO - J"
- "Queretaro 121 - 101" (single) → kept as-is, unit: "101", floor: 1

**Impact:**
- Enables unit-level tracking (occupancy, income, maintenance)
- Supports detailed property characteristics
- Ready for linking transactions to specific apartments
- Foundation for unit-level analytics and reporting

See **Data Integration - Propiedad Expansion** section below for technical details.

### Postal Code Integration (January 2025)
Added comprehensive postal code support to the `propiedad` table with automatic data population from government catalog.

**Changes**:
- Added 6 new fields to `propiedad` table: codigo_postal, colonia, estado, estado_descripcion, municipio, municipio_descripcion
- Integrated official SEPOMEX government catalog (145,449 postal codes)
- Created table `codigo_postal` with complete Mexican postal code database
- Developed fuzzy matching script to auto-fill postal data (98.2% success rate)
- Created helper utility `helper/link_propiedades_propietarios.php` for property-owner associations

**Impact**: Properties now have complete geographic information enabling:
- Regional reporting and analysis
- Neighborhood-level income comparisons
- Geographic visualization capabilities
- Improved property data quality

See **Data Integration - Postal Code System** section above for technical details.

### UTF-8 Encoding Fix (December 2025)
Fixed double-UTF-8 encoding in `/inc/ia_utilerias.php` that was causing the field `ano_de_ingresos` to display as "AÃ±o de Ingresos" instead of the correct "Año de Ingresos".

**Impact**: This fix benefits ALL fields across ALL systems that use Spanish characters with ñ, including:
- año/años (year/years)
- compañía/compañias (company/companies)
- español (Spanish)

See [readme_updates.md](readme_updates.md) for technical details.

---

**System Type:** Property Management / Vacation Rental
**Framework:** iaCase (custom PHP CRUD framework)
**Database:** MySQL (with government data integration)
**UI:** jqGrid + jQuery UI
**Architecture:** Model-driven auto-CRUD with declarative customization

**Core Entities:**
- Property Owners (`propietario`) - 114+ records
- Properties (`propiedad`) - 114 properties with full postal code data
- Postal Code Catalog (`codigo_postal`) - 145,449 government records (SEPOMEX)
- Transactions across 4 OTA platforms
- Reservations from 2 PMS systems

**OTA Platforms Supported:**
- Airbnb
- Agoda
- Booking.com
- CasitaMX

**Property Management Systems (PMS) Integrated:**
- Hostify
- Cloudbeds

**Data Integration:**
- SEPOMEX Official Postal Code Catalog (145,449 records)
- Automated fuzzy matching for geographic data (98.2% success rate)
- CSV import from OTA platforms

**Helper Tools:**
- Property ↔ Owner Fuzzy Matcher (4-tier matching algorithm)
- Postal Code Fill Script (3-tier matching with 98.2% success)
- Bulk linking utilities with confidence thresholds

**Related Systems in Quantix:**
- Bintech Robot UP (scholarship management)
- Ari Ben (CFDI invoicing)

All systems share the same framework core (`/inc/`) and framework tables (`iac_*`)!
