# OpenCode Documentation - Quantix System

## System Overview

**Quantix** is a comprehensive business application framework built on a custom PHP platform called **iaCase**. This is a production system handling real business operations across multiple domains.

### Primary System: MrW - Reporte Propietarios (Property Management)

The main business focus is **MrW - Reporte Propietarios**, a sophisticated vacation rental property management system that:
- Manages 350 individual property units across Mexico City
- Tracks 107 property owners with automated linking
- Integrates with 6 different platforms (4 OTAs + 2 PMS systems)
- Processes thousands of rental transactions and reservations
- Provides comprehensive analytics and reporting

## Technical Architecture

### Core Framework: iaCase
- **Location**: `/inc/iacase.php` (~8000 lines)
- **Pattern**: Modified MVC with declarative CRUD engine
- **Philosophy**: Database-first, reality-driven execution
- **Features**: Auto-generates forms, grids, validation from DB schema

### Database Layer
- **Connection**: MySQL via custom wrapper (`ia_utilerias.php`)
- **Key Functions**:
  - `ia_sqlArrayIndx($sql)` - Get indexed array (MOST COMMON)
  - `ia_sqlArray($sql, $key)` - Get associative array (requires key parameter)
  - `ia_singleton($sql)` - Get single value
  - `ia_query($sql)` - Execute raw SQL
  - `ia_insert($table, $values)` - Insert with auto-escaping
  - `ia_update($table, $values)` - Update with auto-escaping

### Session & Security
- **Timeout**: 70 min (before 6pm), 37 min (after 6pm)
- **User Types**: Rony (super admin), Power, Normal
- **Permissions**: Table-level + field-level via `iac_field_permission`
- **Session Regeneration**: Automatic on login

## Key Business Domains

### 1. Property Management (MrW - Reporte Propietarios)

#### Property Owners (`propietario`)
- **Table**: `propietario` (107 records)
- **PK**: `propietario_id` (UUID)
- **Fields**: departamento, propietario (owner name), es_dueno, inversionista, correo
- **Features**: Owner/investor distinction, department assignments

#### Properties (`propiedad`)
- **Table**: `propiedad` (350 individual units)
- **PK**: `propiedad_id` (varchar(32))
- **Key Fields**: nombre_propiedad, numero_unidad, piso, direccion
- **Geographic Data**: codigo_postal, colonia, estado, municipio (auto-filled from SEPOMEX)
- **Expansion**: Transformed from 114 buildings → 350 units (January 2025)

#### Transaction Tables (One per OTA)
- `airbnb_transaction` - Airbnb bookings
- `agoda_transaction` - Agoda bookings  
- `booking_transaction` - Booking.com bookings
- `casitamx_transaction` - CasitaMX bookings
- **Common Fields**: dates, guest info, financial data, currency, year_of_income

#### Reservation Tables (PMS Systems)
- `hostify_reserva` - Hostify PMS reservations (lightweight)
- `cloudbeds_reserva` - Cloudbeds PMS reservations (comprehensive)

### 2. Framework Systems (Shared)
- User management (`iac_usr`)
- Activity logging (`iac_log`)
- System parameters (`iac_parametros`)
- Field permissions (`iac_field_permission`)

## File Organization

### Application Models (`/app/`)
```
app_propietario.php              ← Property owners
app_propiedad.php                ← Properties (350 units)
app_airbnb_transaction.php       ← Airbnb transactions
app_agoda_transaction.php        ← Agoda transactions
app_booking_transaction.php      ← Booking.com transactions
app_casitamx_transaction.php     ← CasitaMX transactions
app_hostify_reserva.php          ← Hostify reservations
app_cloudbeds_reserva.php        ← Cloudbeds reservations
app_iac_*.php                    ← Shared framework models
```

### UI Pages (`/backoffice/`)
```
propietario.php                  ← Property owners CRUD
propiedad.php                    ← Properties CRUD
airbnb_transaction.php           ← Airbnb transactions UI
[other OTA transaction pages]
hostify_reserva.php              ← Hostify reservations UI
cloudbeds_reserva.php            ← Cloudbeds reservations UI
helper/                          ← Utility scripts
├── link_propiedades_propietarios.php  ← Property ↔ Owner matching
├── link_propiedades_pms.php            ← PMS preview matching
├── link_pms_propiedades.php            ← PMS production matching
└── fill_postal_codes.php               ← Postal code filling UI
```

### Framework Core (`/inc/`)
```
config.php                       ← Entry point, session, DB setup
iacase.php                       ← Core CRUD engine (~8000 lines)
ia_utilerias.php                 ← Database utilities (6000+ lines)
iaJQGrid.php                     ← jqGrid integration
appRelateBase.php                ← Auto-generated schema metadata
appRelate.php                    ← Custom schema definitions
helpers.php                      ← Business logic utilities
```

## Database Schema Highlights

### Key Tables
```sql
-- Property Owners
propietario (107 records)
├── propietario_id (UUID PK)
├── departamento (property identifier)
├── propietario (owner name)
└── correo (contact)

-- Properties (expanded to unit-level)
propiedad (350 records)
├── propiedad_id (varchar(32) PK)
├── nombre_propiedad (building name)
├── numero_unidad (unit: A-J, 01-24)
├── piso (floor: extracted from unit)
├── direccion (street address)
├── propietario_id (FK to owners)
└── [6 postal code fields from SEPOMEX]

-- Government Catalog
codigo_postal (145,449 records)
├── codigo_postal (5-digit)
├── colonia (neighborhood)
├── estado (state code)
└── municipio (municipality)

-- Transactions (example: Airbnb)
airbnb_transaction
├── airbnb_transaction_id (UUID PK)
├── fecha, fecha_de_llegada, check_in, check_out
├── huesped (guest)
├── moneda, monto, ingresos_recibidos
├── tarifa_de_servicio, impuestos
└── ano_de_ingresos (tax year)
```

## Major Technical Achievements

### 1. Property Expansion (January 2025)
- **Challenge**: 114 buildings with multiple units tracked as single rows
- **Solution**: In-place expansion to 350 individual units
- **Method**: Used `num_deptos` field as multiplier, smart unit numbering
- **Result**: 100% success, all data preserved, UUID integrity maintained

### 2. Postal Code Integration (January 2025)
- **Challenge**: Properties lacked geographic data
- **Solution**: Fuzzy matching against SEPOMEX government catalog
- **Algorithm**: 3-tier matching (manual overrides → exact → fuzzy)
- **Result**: 98.2% success rate (112/114 properties matched)

### 3. Smart Matching Systems
- **Property ↔ Owner**: Exact character matching (212 properties → 107 owners, 100% success)
- **PMS ↔ Property**: 4-tier fuzzy matching (3,013 reservations → 350 units)
- **Features**: Confidence scoring, color coding, CSV export, preview modes

### 4. UTF-8 Encoding Fix (December 2025)
- **Problem**: "AÃ±o de Ingresos" instead of "Año de Ingresos"
- **Cause**: Double-UTF-8 encoding in `to_label()` function
- **Solution**: Fixed 6 replacements in `ia_utilerias.php`
- **Impact**: Fixed ALL Spanish labels across entire system

## Development Patterns

### Adding New Entities
1. Create database table
2. Regenerate `appRelateBase.php` from schema
3. Create `/app/app_[tablename].php` extending `iacase_base`
4. Create `/backoffice/[tablename].php` UI page
5. Add custom metadata to `appRelate.php` if needed

### iaCase Class Pattern
```php
class app_propietario extends iacase_base {
    function __construct() {
        parent::__construct('r', 'propietario');  // permission, table
        
        $this->campos_default();  // Auto-generate from DB schema
        // Customizations here
        $this->campos_final();    // Finalize configuration
        
        $this->modo = 'cardex_window';
        $this->permiso_delete = true;
    }
}
```

### Database Query Patterns
```php
// Get single value
$count = ia_singleton("SELECT COUNT(*) FROM users");

// Get all rows for iteration (MOST COMMON)
$rows = ia_sqlArrayIndx("SELECT * FROM properties");
foreach ($rows as $row) {
    echo $row['name'];
}

// Get rows keyed by ID (requires second parameter!)
$users = ia_sqlArray("SELECT * FROM users", 'id');
echo $users['123']['name'];

// WRONG - Missing key parameter!
$data = ia_sqlArray("SELECT * FROM users");  // ❌ Will fail!
```

### Request Flow
```
User Request → /backoffice/[tablename].php
    ↓
require ../inc/config.php (session, auth, DB, autoloader)
    ↓
$f = new app_[tablename]()
    ↓
$f->process_action() handles ?iah parameter:
    'r' = Read/View
    'e' = Edit form  
    'a' = Add form
    's' = Save (insert/update)
    'd' = Delete
    'l' or '' = List (jqGrid)
    ↓
$f->show() renders HTML (or JSON for AJAX)
```

## Helper Tools & Utilities

### Property ↔ Owner Matcher
- **Location**: `/backoffice/helper/link_propiedades_propietarios.php`
- **Version**: v3.0 Exact Matcher (January 2026)
- **Algorithm**: Deterministic exact character matching
- **Results**: 212 properties → 107 owners (100% success)
- **Features**: Preview mode, confidence scoring, CSV export

### PMS Matching Tools
- **Preview**: `/backoffice/helper/link_propiedades_pms.php`
- **Production**: `/backoffice/helper/link_pms_propiedades.php`
- **Scope**: 3,013 reservations across Hostify + Cloudbeds
- **Algorithm**: 4-tier fuzzy matching with confidence scores

### Postal Code Filler
- **Location**: `/backoffice/helper/fill_postal_codes.php`
- **Purpose**: Web UI for filling postal code data
- **Algorithm**: 3-tier matching (manual → exact → fuzzy)
- **Results**: 98.2% success rate (112/114 properties)

## Important Technical Notes

### Critical Database Function Usage
- **✅ CORRECT**: `ia_sqlArrayIndx($sql)` for iteration
- **✅ CORRECT**: `ia_sqlArray($sql, 'id')` for keyed access (requires 2nd parameter!)
- **❌ WRONG**: `ia_sqlArray($sql)` without key parameter (will fail!)

### Framework Function Execution
- `ia_update()` and `ia_insert()` **return SQL strings** but do NOT execute
- **Must call** `ia_query($sql)` to execute the returned SQL
- Example: `$sql = ia_update('table', $values); ia_query($sql);`

### Session Requirements
- All helper scripts require authenticated Quantix session
- Cannot run via CLI or curl without session cookies
- Must access via browser where logged into system

### File Encoding
- All PHP files must be UTF-8 without BOM
- Spanish characters (ñ, á, é, í, ó, ú) must be properly encoded
- Text editor must be set to UTF-8 encoding

## Security Considerations

### SQL Injection Prevention
- Use `ia_insert()`, `ia_update()` helpers (auto-escape)
- Manual queries: Use parameterized queries or proper escaping
- Never concatenate user input directly into SQL

### Permission System
- User type checks: `usuarioTipoRony()`, `usuarioSupervisaBodega()`
- Table permissions: `$this->permiso_insert`, `$this->permiso_update`
- Field permissions: `iac_field_permission` table

### Session Security
- Auto-timeout enforcement
- Session ID regeneration on login
- SameSite=Strict cookies

## Current System State

### Production Status
- **✅ Fully operational** - Managing real business data
- **✅ MySQL stable** - Service runs indefinitely (fixed 90-second restart issue)
- **✅ All modules functional** - 6 platforms integrated
- **✅ Data integrity maintained** - UUID primary keys, audit trails

### Data Scale
- **Properties**: 350 individual units
- **Owners**: 107 property owners  
- **Transactions**: Thousands across 4 OTAs
- **Reservations**: 3,013 across 2 PMS systems
- **Geographic**: Complete coverage via SEPOMEX catalog

### Recent Updates (January 2025)
1. Property table expansion (114 → 350 units)
2. Postal code integration (98.2% success)
3. UTF-8 encoding fix (Spanish labels)
4. PMS integration (Hostify + Cloudbeds)
5. Smart matching tools (property ↔ owner, PMS ↔ property)

## Access Information

### Web Interface
- **URL**: https://dev-app.filemonprime.net/quantix/
- **Login**: Requires authenticated session
- **Dashboard**: `/backoffice/index.php`

### CLI Execution
```bash
# PHP binary location
/lamp/php/bin/php

# Execute script locally
/lamp/php/bin/php /lamp/www/quantix/script_name.php

# Web execution (remote)
curl -s https://dev-app.filemonprime.net/quantix/script_name.php
```

### Database Access
```bash
# MySQL shell
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock

# Connect to quantix database
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock quantix
```

## Testing & Development

### Test Framework
- **PHPUnit**: Located in parent directory structure
- **Bootstrap**: Sets session user_id=1 (super admin)
- **Coverage**: Reports in `/tests/reports/`

### Development Environment Detection
- Requires `/wamp/www/showErrors.vin` file for development mode
- Production mode: Errors suppressed, logged to files
- Check via: `file_exists(__DIR__ . '/../../showErrors.vin')`

## Common Operations

### Adding New Fields to iaCase Classes
1. Add column to database
2. Update `campos_default()` method (auto-generated section)
3. Update comment headers
4. Customize in `campos_final()` method
5. Update `appRelate.php` for relationships if needed

### Regenerating Schema Metadata
```bash
# After database changes, regenerate appRelateBase.php
# This is typically done manually by examining schema and updating the file
```

### Running Helper Scripts
1. Access via browser (requires login session)
2. Review preview mode first
3. Apply changes with confirmation
4. Export CSV for audit trail

## Future Enhancements (Planned)

### Phase 1 - Data Enrichment
- Populate unit characteristics (metros², recamaras, baños)
- Add orientation data (Norte, Sur, Este, Oeste)
- Set furnished status from transaction history

### Phase 2 - Advanced Analytics
- Unit-level income reporting
- Occupancy rate analysis by floor
- Performance comparison across platforms

### Phase 3 - UI Enhancements
- Visual floor plan builder
- Drag-and-drop unit assignment
- Heat map showing unit performance

## Troubleshooting Guide

### Common Issues
1. **Database query returns empty**: Check function usage (ia_sqlArray vs ia_sqlArrayIndx)
2. **Helper script shows login**: Access via browser with authenticated session
3. **UTF-8 encoding issues**: Verify file saved as UTF-8 without BOM
4. **Permission denied**: Check user type and table permissions
5. **MySQL connection issues**: Verify service status and socket path

### Debug Commands
```bash
# Check MySQL service
systemctl status lamp-mysql

# Test database connectivity
/lamp/mysql/bin/mysql -u root -pM@chiavell1 \
  --socket=/lamp/mysql/mysql.sock \
  -e "SELECT 'MySQL is alive!' as status;"

# Check PHP syntax
/lamp/php/bin/php -l /lamp/www/quantix/file.php
```

## Key Contacts & Context

### System Creator
- **Framework**: Custom iaCase (reality-driven, minimal abstraction)
- **Business Domain**: Mexican vacation rental property management
- **Data Sources**: 4 international OTAs + 2 local PMS systems
- **Geographic Focus**: Mexico City (complete SEPOMEX integration)

### Business Value
- **Centralizes fragmented data** from 6 platforms
- **Enables owner-level analytics** across all income sources
- **Automates manual processes** (postal codes, property linking)
- **Provides audit trails** for compliance and reporting
- **Scales effectively** for enterprise-level property management

---

**Documentation Created**: January 5, 2026  
**Created By**: Claude Code (OpenCode)  
**Purpose**: Self-documentation for future OpenCode instances working on Quantix system  
**Scope**: Complete system overview for effective development and maintenance

---

## Quick Reference Commands

### Database
```bash
# Connect
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock quantix

# Quick query
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock quantix -e "SELECT COUNT(*) FROM propiedad;"
```

### PHP
```bash
# Execute script
/lamp/php/bin/php /lamp/www/quantix/script_name.php

# Check syntax
/lamp/php/bin/php -l /lamp/www/quantix/file.php
```

### Web Access
```bash
# Test web endpoint
curl -s https://dev-app.filemonprime.net/quantix/index.php

# With session (requires cookie extraction)
curl -s -b cookies.txt https://dev-app.filemonprime.net/quantix/backoffice/helper/link_propiedades_propietarios.php
```

### Essential File Locations
- **Framework**: `/inc/iacase.php` (core engine)
- **DB Utils**: `/inc/ia_utilerias.php` (database functions)
- **Config**: `/inc/config.php` (entry point)
- **Models**: `/app/app_*.php` (business logic)
- **UI**: `/backoffice/*.php` (user interfaces)
- **Helpers**: `/backoffice/helper/*.php` (utility scripts)