# UTF-8 Encoding Fix - December 2025

## What Was Broken

The field `ano_de_ingresos` in the Airbnb Transactions module was displaying its label as:
```
AÃ±o de Ingresos
```

Instead of the correct Spanish text:
```
Año de Ingresos
```

The letter "ñ" (Spanish eñe) was showing up as the garbage characters "Ã±".

## Why This Happened - The Simple Explanation

Think of text encoding like a secret code book. Every letter needs a code number.

The letter "ñ" has a special code in UTF-8 encoding. But what happened here is like using the code book TWICE on the same letter. It's like:

1. First encoding: "ñ" → becomes bytes "C3 B1" (correct UTF-8)
2. Second encoding: Someone read those bytes AS IF they were Latin-1 text, which gave us "Ã±"
3. Third encoding: Then "Ã±" got encoded to UTF-8 AGAIN → becomes bytes "C3 83 C2 B1" (double-encoded mess)

So the file had the wrong bytes stored inside it.

## Technical Details (For Nerds)

### File Location
`/lamp/www/quantix/inc/ia_utilerias.php`

### Function Involved
`to_label()` - Lines 1223-1279

This function converts database field names (like `ano_de_ingresos`) into pretty Spanish labels for the user interface.

### How It Works

1. Takes field name: `ano_de_ingresos`
2. Replaces underscores with spaces: `ano de ingresos`
3. Uses regex to find Spanish words and add proper accents:
   - Pattern: `/\bano\b/i` (finds the word "ano")
   - Replacement: `'año'` (replaces with accented version)
4. Capitalizes words: `Año De Ingresos`
5. Fixes articles: `Año de Ingresos`

### The Bug

In step 3, the replacement array had BAD DATA stored in it:

**Before (BROKEN):**
```php
,array('acción','alemán','almacén','aÃ±o','aÃ±os', ... 'compaÃ±ía','compaÃ±ias', ... 'espaÃ±ol'
```

**After (FIXED):**
```php
,array('acción','alemán','almacén','año','años', ... 'compañía','compañias', ... 'español'
```

### Byte-Level Proof

**Wrong encoding (double-encoded):**
```
Hex: C3 83 C2 B1
Displays as: Ã±
```

**Correct encoding:**
```
Hex: C3 B1
Displays as: ñ
```

The file was already saved as UTF-8 (good!), but someone in the past had copy-pasted text from a Latin-1 source or a double-encoded source into this UTF-8 file, creating this mojibake (character corruption).

## What Was Changed

Modified file: `/lamp/www/quantix/inc/ia_utilerias.php`

### Line 934 (comment)
```diff
- * ajusta imagen al tamaÃ±o
+ * ajusta imagen al tamaño
```

### Line 1245 (replacement array)
```diff
-            ,array('acción','alemán','almacén','aÃ±o','aÃ±os','área','áreas','artículo','artículos'
+            ,array('acción','alemán','almacén','año','años','área','áreas','artículo','artículos'
```

### Line 1247 (replacement array continued)
```diff
-                ,'cálculo','caractér','catálogo','catálogos','código','comentario','compaÃ±ía','compaÃ±ias','C.P.'
+                ,'cálculo','caractér','catálogo','catálogos','código','comentario','compañía','compañias','C.P.'
```

### Line 1249 (replacement array continued)
```diff
-                ,'economía','electrónica','electrónico','electrónicos','espaÃ±ol'
+                ,'economía','electrónica','electrónico','electrónicos','español'
```

## Total Changes

- **6 replacements** of double-encoded "Ã±" with correctly-encoded "ñ"
- **1 file** modified: `ia_utilerias.php`
- **0 database changes** required
- **0 configuration changes** required

## Impact

This fix affects ANY field name in the entire Quantix system that contains the Spanish letter "ñ" when converted to a label by the iaCase framework.

### Examples of Fields Now Fixed:
- `ano_de_ingresos` → "Año de Ingresos" ✓
- `compania_nombre` → "Compañía Nombre" ✓
- `idioma_espanol` → "Español" ✓
- Any field with "año", "años", "compañía", "compañias", or "español" in the name ✓

## How To Verify The Fix

1. Open any page that uses the field `ano_de_ingresos` (like Airbnb Transactions)
2. Look at the form label or column header
3. It should now say "Año de Ingresos" with a proper ñ character
4. If you see "AÃ±o de Ingresos" then something went wrong

### PHP Test Command
```bash
php -r "
\$field = 'ano_de_ingresos';
\$field = str_replace('_', ' ', \$field);
\$field = preg_replace('/\bano\b/i', 'año', \$field);
\$field = ucwords(\$field);
echo \$field . PHP_EOL;
"
```

Expected output: `Año De Ingresos`

## Why This Matters

### User Experience
Users see proper Spanish text instead of garbled mojibake. Professional appearance.

### Data Integrity
The actual DATA in the database was never affected. This was purely a DISPLAY issue in the user interface labels.

### Framework-Wide Fix
Since `to_label()` is used throughout the entire iaCase framework for ALL automatic label generation, this one fix improves labels across the entire application wherever these words appear.

## Prevention For Future

### When Editing ia_utilerias.php:
1. Make sure your text editor is set to UTF-8 encoding (not Latin-1, not Windows-1252)
2. When copying Spanish text, verify the ñ looks correct in your editor
3. If you see "Ã±" in your editor, you're looking at double-encoded text - FIX IT
4. Save the file as UTF-8 WITHOUT BOM (byte order mark)

### Text Editor Settings:
- **Encoding:** UTF-8
- **Line Endings:** Unix (LF) - since this is running on Linux
- **BOM:** None (no byte order mark needed for UTF-8)

## Additional Notes

The file `ia_utilerias.php` contains 6,000+ lines of utility functions. The `to_label()` function is just one small part, but it's used EVERYWHERE in the system for automatic UI label generation.

This is a core framework function that:
- Converts snake_case to Title Case
- Adds Spanish accents automatically
- Handles special abbreviations (RFC, CURP, IMSS, CP)
- Fixes Spanish articles (De→de, Del→del, etc.)

By fixing the source data in this function, we fixed labels across potentially hundreds of forms and grids throughout Quantix.

---

**Fixed by:** Claude Code
**Date:** December 29, 2025
**Commit message suggestion:** "Fix UTF-8 double-encoding in to_label() function - corrects ñ character display in Spanish labels"

---

# Dashboard Enhancements - December 2025

## 1. New Reservation Modules Added

### Added Hostify and Cloudbeds PMS Integration

Two new Property Management System (PMS) modules were added to the MrW - Reporte Propietarios system to track reservations from Hostify and Cloudbeds platforms.

#### Files Created/Modified

**Backend Models:**
- `/app/app_reservas_hostify.php` - Hostify reservation model
- `/app/app_reservas_cloudbeds.php` - Cloudbeds reservation model

**Frontend Pages:**
- `/backoffice/reservas_hostify.php` - Hostify reservations UI
- `/backoffice/reservas_cloudbeds.php` - Cloudbeds reservations UI

**Dashboard:**
- `/backoffice/index.php` - Added links to new modules in MrW card

**Documentation:**
- `/readme_mrw_reporte_propietarios.md` - Comprehensive documentation update

#### Database Tables

**`reservas_hostify`** - Lightweight reservation tracking
- Primary key: `reservas_hostify_id` (UUID)
- Core fields: confirmation_code, guest_name, check_in, check_out, channel, anuncio, total_price, nights, currency
- Purpose: Track basic reservation info from Hostify PMS

**`reservas_cloudbeds`** - Comprehensive guest management
- Primary key: `reservas_cloudbed_id` (UUID)
- 30+ fields including:
  - Guest info: name, email, phone_number, gender, country, city
  - Reservation: reservation_number, third_party_confirmation_number, source, status
  - Room: room_number, room_type
  - Financial: accommodation_total, amount_paid, grand_total, deposit, balance_due
  - Documents: type_of_document, document_issuing_country
  - Dates: check_in_date, check_out_date, reservation_date
  - Other: adults, children, guest_status, estimated_arrival_time
- Purpose: Full-featured guest and booking management from Cloudbeds PMS

#### Dashboard Changes (index.php)

**Location:** MrW · Reporte Propietarios card

**Added links:**
```html
<a href="reservas_hostify.php">Hostify Reservations</a>
<a href="reservas_cloudbeds.php">Cloudbeds Reservations</a>
```

**Position:** After the existing OTA transaction links (Airbnb, Agoda, Booking, CasitaMX)

#### Integration Notes

- Both modules follow standard iaCase framework pattern
- Auto-CRUD functionality through `iacase_base` parent class
- Full permissions enabled (insert, update, delete, export, list, read)
- Standard audit trail (alta_db, alta_por, ultimo_cambio, ultimo_cambio_por)
- jqGrid integration for listing and filtering
- UUID primary keys for data integrity

---

## 2. Dashboard Card State Persistence

### Problem Solved
Users had to re-expand the same dashboard card every time they returned from a module, breaking workflow continuity.

### Solution Implemented
Added localStorage-based card state persistence to remember which card was last opened.

#### File Modified
`/backoffice/index.php` - Dashboard JavaScript

#### Changes Made

**Before:**
```javascript
// Open default card only
document.querySelector('#quantix-2025 .qx-card[data-open]')?.classList.add('open');
```

**After:**
```javascript
cards.forEach((card, index) => {
    const head = card.querySelector('.qx-card-head');
    head.addEventListener('click', () => {
        const open = card.classList.contains('open');
        cards.forEach(c => c.classList.remove('open'));
        if (!open) {
            card.classList.add('open');
            // Remember which card is open
            localStorage.setItem('quantix-open-card', index);
        } else {
            // Clear if closing the card
            localStorage.removeItem('quantix-open-card');
        }
    });
});

// Restore previously open card, or open default
const savedCardIndex = localStorage.getItem('quantix-open-card');
if (savedCardIndex !== null && cards[savedCardIndex]) {
    cards[savedCardIndex].classList.add('open');
} else {
    document.querySelector('#quantix-2025 .qx-card[data-open]')?.classList.add('open');
}
```

#### How It Works

1. **On Card Click:**
   - Opens clicked card
   - Saves card index to `localStorage` key: `'quantix-open-card'`
   - If closing a card, removes the saved state

2. **On Page Load:**
   - Checks for saved card index in localStorage
   - If found: Opens that card automatically
   - If not found: Falls back to default card (Bintech with `data-open` attribute)

#### User Experience Improvement

**Before:**
1. User opens MrW card
2. Clicks "Hostify Reservations"
3. Works in module
4. Returns to index
5. **Card is closed** - must click MrW again ❌

**After:**
1. User opens MrW card
2. Clicks "Hostify Reservations"
3. Works in module
4. Returns to index
5. **MrW card still open** - ready to click next module ✅

#### Technical Details

- **Storage:** Browser localStorage (persists across sessions)
- **Key:** `'quantix-open-card'`
- **Value:** Card index (0-based: 0=Bintech, 1=MrW, 2=Ari Ben)
- **Scope:** Per browser, per domain
- **Fallback:** Default card if no saved state or invalid index

#### Browser Compatibility
- localStorage is supported in all modern browsers (IE8+)
- Graceful degradation: Falls back to default card if localStorage unavailable

---

## 3. Documentation Updates

### Files Updated

**`/readme_mrw_reporte_propietarios.md`** - Complete system documentation

Added comprehensive sections for:

#### System Overview
- Updated to mention PMS integration alongside OTA platforms
- Clarified system now manages both transactions AND reservations

#### Business Domain
- Added "Property Management System reservations (Hostify, Cloudbeds)" to managed items

#### App Files (Model Layer)
New section: **Reservation Models (Property Management Systems)**
- Documented both Hostify and Cloudbeds models
- Listed all fields and data tracked by each system
- Explained primary keys and table structure

#### Backoffice Pages (UI Layer)
New section: **Reservation Management (Property Management Systems)**
- Documented UI functionality for each PMS module
- Listed features available in each interface

#### Database Tables
New section: **Reservation Tables (Property Management Systems)**
- Complete field breakdown for both tables
- Organized by category (guest info, financial, dates, etc.)
- Included audit trail fields

#### Key Features
New section: **3. Property Management System (PMS) Integration**
- Detailed feature list for Hostify
- Detailed feature list for Cloudbeds
- Distinction between lightweight vs comprehensive tracking

#### File Organization
- Updated directory tree to include new model and UI files
- Added database tables to structure diagram

#### Summary
- Added "Property Management Systems (PMS) Integrated" section
- Listed both Hostify and Cloudbeds as supported systems

### Documentation Quality
- Maintained consistent formatting with existing docs
- Used same structure and style as OTA transaction documentation
- Added emojis where appropriate for visual scanning
- Included technical details for developers
- Provided user-facing explanations for business users

---

## Summary of Changes (December 30, 2025)

### What Was Added
1. **2 new PMS modules** (Hostify, Cloudbeds)
2. **2 new database tables** (reservas_hostify, reservas_cloudbeds)
3. **4 new files** (2 models + 2 UI pages)
4. **Dashboard card state persistence** (localStorage-based)
5. **Comprehensive documentation** (readme_mrw_reporte_propietarios.md)

### What Was Improved
1. **User experience** - Card state remembered across navigation
2. **System integration** - Unified PMS data alongside OTA transactions
3. **Documentation** - Complete coverage of all new features

### Impact
- **MrW system** now manages 4 OTA platforms + 2 PMS systems = **6 data sources**
- **User workflow** significantly improved with persistent card state
- **Documentation** remains current and comprehensive

### Files Modified
- `/backoffice/index.php` - Dashboard links + card persistence
- `/readme_mrw_reporte_propietarios.md` - Complete documentation update
- `/readme_updates.md` - This file (changelog)

### Files Created
- `/app/app_reservas_hostify.php`
- `/app/app_reservas_cloudbeds.php`
- `/backoffice/reservas_hostify.php`
- `/backoffice/reservas_cloudbeds.php`

### Database Changes
- New table: `reservas_hostify`
- New table: `reservas_cloudbeds`

### Zero Breaking Changes
- All changes are additive
- Existing functionality unchanged
- Backward compatible

---

**Updated by:** Claude Code
**Date:** December 30, 2025
**Commit message suggestion:** "Add Hostify and Cloudbeds PMS modules, implement dashboard card state persistence, update documentation"
