<?php
/**
 * Html DETAILS element report: header totals, problems DETAILS and each test file a DETAILS.
 *
 * @usage en phpunit.xml
 * <extensions>
 *      <extension class="DetailReporter" file="/wamp/www/vitex/tests/DetailReporter.php">
 *          <arguments>
*               <string>tests/reports/testDetails.html</string>
 *              <string>Pruebas realizadas</string>
 *          </arguments>
 *      </extension>
 *  </extensions>
 *
 *
 * Con los hooks de phpUnit registra en un array:
 *   cada que inicia el proceso de un archivoTest.php, registra con key archivoTest.php un details-div-ol
 *   a cada prueba le escribe su <li> , llevando la cuenta de cuantas pruebas tuvieron que resultado (mark), y un registro de las pruebas no Ok
 *   al cambiar de archivoTest.php a otro archivo o terminar la prueba acomplete del details y agrega </ol></div></details>
 *   al terminar las pruebas, genera un archivo .html con
 *    htmlStart
 *    muestra los totales (header)
 *    de existir pruebas no Ok las reporta en un details por cuasa (mark)
 *    las pruebas realizadas
 *    htmlEnd
 *
 * @require phpUnit test hooks: phpUnit > 7, probado con phpUnit 9.5
 *
 * @version 2021-11-29
 */

declare(strict_types=1);

use JetBrains\PhpStorm\ArrayShape;
use JetBrains\PhpStorm\ExpectedValues;
use JetBrains\PhpStorm\Pure;

final class DetailReporter implements
    PHPUnit\Runner\Hook, PHPUnit\Runner\TestHook, PHPUnit\Runner\BeforeFirstTestHook,
    PHPUnit\Runner\BeforeTestHook,
    PHPUnit\Runner\AfterIncompleteTestHook,
    PHPUnit\Runner\AfterRiskyTestHook,
    PHPUnit\Runner\AfterTestWarningHook,
    PHPUnit\Runner\AfterTestErrorHook,
    PHPUnit\Runner\AfterTestFailureHook,
    PHPUnit\Runner\AfterSkippedTestHook,
    PHPUnit\Runner\AfterSuccessfulTestHook,
    PHPUnit\Runner\AfterTestHook,
    PHPUnit\Runner\AfterLastTestHook
{
    protected const MARK_NUM = 'Num';
    protected const MARK_OK = 'Ok';
    protected const MARK_FAIL = 'Fail';
    protected const MARK_RISKY = 'Risky';
    protected const MARK_INCOMPLETE = 'Incomplete';
    protected const MARK_SKIPPED = 'Skipped';
    protected const MARK_ERROR = 'Error';
    protected const MARK_WARNING = 'Warning';

    protected string $outputFile = 'detailTestsReport.html';
    protected string $corriendo = '';
    protected string $title = 'Pruebas realizadas';

    /** @var array  $markHtml */
    protected array $markHtml = [
        self::MARK_FAIL => '<span title="Fail, un bug">&cross;</span>',
        self::MARK_RISKY => '<span title="Risky: Ejecuto código que no debía, no esta en su @covers/@uses">&#10082;</span>',
        self::MARK_INCOMPLETE => '<span title="Incomplete, incompleta">&#8265;</span>',
        self::MARK_ERROR => '<span title="Error de Php">&#10811;</span>',
        self::MARK_WARNING => '<span title="Warning de php">&#9888;</span>',
        self::MARK_SKIPPED => '<span title="Skipped, no se ejecuto una/la prueba">!</span>',
        self::MARK_OK => '<span  class="okColor" title="Ok">✔</span>',
        self::MARK_NUM => '',
    ];

    /** @var array $markErrorHtml */
    protected array $markErrorHtml = [
        self::MARK_FAIL => '<span title="Fail, un bug">&cross;</span>',
        self::MARK_RISKY => '<span title="Risky: Ejecuto código que no debía, no esta en su @covers/@uses">&#10082;</span>',
        self::MARK_INCOMPLETE => '<span title="Incomplete, incompleta">&#8265;</span>',
        self::MARK_ERROR => '<span title="Error de Php">&#10811;</span>',
        self::MARK_WARNING => '<span title="Warning de php">&#9888;</span>',
        self::MARK_SKIPPED => '<span title="Skipped, no se ejecuto una/la prueba">!</span>',
    ];

    /** @var array|string[] $report */
    protected array $report = [];
    /** @var array|string[] $report */
    protected array $reportProblems = [];

    protected const SHAPE_COUNT = [self::MARK_NUM => "int", self::MARK_OK => "int", self::MARK_FAIL => "int",
        self::MARK_RISKY => "int", self::MARK_INCOMPLETE => "int", self::MARK_SKIPPED => "int",
        self::MARK_ERROR => "int", self::MARK_WARNING => "int"];

    protected const EXPECT_MARK = [self::MARK_OK, self::MARK_FAIL,
        self::MARK_RISKY, self::MARK_INCOMPLETE,self::MARK_SKIPPED,
        self::MARK_ERROR, self::MARK_WARNING];

    /** @var array $count = #[ArrayShape(DetailReporter::SHAPE_COUNT)] */
    protected array $count;

    /** @var array $count = #[ArrayShape(DetailReporter::SHAPE_COUNT)] */
    protected array $countTotals;

    protected string $previousTestClass = '';

    public function __construct($outputFilePath, $title = '') {
        if(!empty($outputFilePath))
            $this->outputFile = $outputFilePath;
        if(!empty($title))
            $this->title = $title;
        $this->countTotals = $this->count = $this->initCounts();

        global $argv, $argc;
        $lastArg = $argc === 1 ? 'Todas las pruebas' :  $argv[$argc - 1];
        $this->corriendo = $lastArg[0] === '-' ? 'Todas las pruebas' :
            htmlentities(    str_replace('tests/', '', $lastArg ?? '' )  );
        for($i=0; $i < $argc; ++$i) {
            if($argv[$i] === '--testsuite') {
                $this->corriendo = htmlentities( $argv[$i + 1] ?? '¿?' );
                break;
            }
        }
    }

    protected function htmlStart():string {
        $mark = $this->countTotals[self::MARK_NUM] === 0 ||
            $this->countTotals[self::MARK_NUM] !== $this->countTotals[self::MARK_OK] ?
                '&#10060;' : '✔';
        $corriendo = str_replace('.php', '', basename($this->corriendo) );
        return <<< HTML
<!DOCTYPE html>
<html lang="es-MX es">
<head>
    <meta charset="UTF-8">
    <title>&#9879; $mark $corriendo</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <meta http-equiv="X-UA-Compatible" content="IE=edge"/>
    <style>
        * {box-sizing: border-box}
        BODY {margin:0;padding:0;line-height: 3em;}
        .problemColor {color:red;}
        .okColor {color:green;}
        .dimmed {color:darkgray;font-size: 0.8em}
        .letra_chica {font-size: 0.7em}
        
        .totals {margin: 0.1em; padding:1em; width:99%;font-size:1.5em;border:16px inset silver;line-height: 1.2em;
            text-align: center; display:flex; justify-content: space-around}
        .flex_item {}    

        .content {margin:1em 2em}
        .resultadosContainer {margin: 1em 2em; border:12px inset silver; color:green;padding: 0.2em 1em;width: fit-content}
        .resultadosContainer LEGEND {color:black}
        .problemsContainer {margin: 1em 2em; border:12px inset red; color:red;padding: 0.2em 1em;width: fit-content}
        .problemDiv {border:red solid 1px;width: fit-content;margin:0.2em 1em}
        
        SUMMARY {color:green;cursor:pointer;text-decoration: underline;font-size: 1.5em}
        DETAILS > DIV {margin:0.5em 1em;padding: 0.5em 1em; border:green solid 1px;width: fit-content;line-height: 1.5em; }
        P.message {margin:0 1em; padding: 0; font-size: 0.8em; color:darkred;font-weight:bold}
    </style>
</head>
<body>
HTML;
    }

    protected function reportTotals(array $count, bool $all = false):string {

        $mark = $count[self::MARK_NUM] === 0 || $count[self::MARK_NUM] !== $count[self::MARK_OK] ?
            '<span title="Errores" class="problemColor">&#10060;</span> ' :
            '<span  class="okColor" title="Ok">✔</span> ';

        $this->title = "$mark $this->corriendo";
        $class = $count[self::MARK_NUM] !== $count[self::MARK_OK] ? ' class="problemColor" ' : '';
        $reporte = [
            "<div class='flex_item letra_chica'><span $class>$mark $this->corriendo" . '</span><br/>' .
            Date('Y-m-d H:i') . '</div>',
        ];

        $totals = '<div class="flex_item" title="Pruebas Ok">' . $this->markHtml[self::MARK_OK]  . '<br>' .  $this->intFormat($count[self::MARK_OK]);
        $totals .= ( ($count[self::MARK_NUM] !== $count[self::MARK_OK] || $all) ?  '<span title="Total de pruebas">/' . $this->intFormat($count[self::MARK_NUM]) . '</span>' : '');
        $reporte[] = $totals . "</div>";

        foreach($this->markErrorHtml as $mark => $html)
            if($all || $count[$mark] > 0)
                $reporte[] =
                    '<div class="flex_item  ' . ($count[$mark] ? 'problemColor' : 'dimmed') . '" title="Pruebas fallaron">' . $html .
                    $this->intFormat($count[$mark]) . "<br>$mark</div>";
        return "<fieldset class='totals'><legend>&nbsp;$this->title</legend>" . implode('     ', $reporte) . "</fieldset>";
    }

    #[Pure] protected function reportProblems():string {
        if(empty($this->reportProblems))
            return '';
        $previousMark = false;
        $lines = [];
        foreach($this->reportProblems as $mark => $arr) {
            if(empty($arr))
                continue;
            if($previousMark)
                $lines[] = "</ol></div></details>";
            else
                $previousMark = true;
            $summary = $this->intFormat(count($arr)) . ' ' . $this->markErrorHtml[$mark] . ' ' . $mark;
            $lines[] = "<details class='problemColor'><summary class='problemColor'>$summary</summary><div class='problemDiv'><ol>";
            foreach($arr as $l)
                $lines[] = "<li>$l";
        }
        if($previousMark)
            $lines[] = "</ol></div></details>";
        return "\r\n\t<fieldset class='problemsContainer'><legend>&nbsp;¡Problemas!&nbsp;</legend>\r\n\t\t" . implode("\r\n\t\t", $lines) . "\r\n\t</fieldset>";
    }

    protected function outputTest(
        #[ExpectedValues(DetailReporter::EXPECT_MARK)]
        string $markName,
        string $test,
        string $message = '') {
        $this->count[$markName]++;
        $this->countTotals[$markName]++;
        $this->count[self::MARK_NUM]++;
        $this->countTotals[self::MARK_NUM]++;
        $class = empty($message) ? '' : "class='problemColor'";

        $this->report[] = "<li $class><span $class>" . $this->markHtml[$markName] . "</span> " .
            $this->getTestName($test, $class) .
            (empty($message) ? '' : "<p class='message'> ➣ ". htmlentities($message ?? '') . "</p>");

        if($markName !== self::MARK_OK)
            $this->reportProblems[$markName][] = "<span $class>" . $this->markHtml[$markName] . "</span> " .
            htmlentities($this->previousTestClass) . '::' . $this->getTestName($test, $class) .
            (empty($message) ? '' : "<p class='message'> ➣ ". htmlentities($message ?? '') . "</p>");
    }

    protected function htmlEnd() : string {
        return <<< HTML
</div>
<hr>
</body>
</html>
HTML;
    }

//// helper functions
    /**
     * @return int[]
     */
    #[ArrayShape(DetailReporter::SHAPE_COUNT)]
    protected function initCounts():array {
        return [
            self::MARK_NUM => 0,
            self::MARK_OK => 0,
            self::MARK_FAIL => 0,
            self::MARK_RISKY => 0,
            self::MARK_INCOMPLETE => 0,
            self::MARK_SKIPPED => 0,
            self::MARK_ERROR => 0,
            self::MARK_WARNING => 0,
        ];
    }

    #[Pure] protected function markAndCount(array $count, $all = false):string {
        $markCount = [];
        if($all || $count[self::MARK_OK])
            $markCount[] = "<span class='okColor'>" . self::MARK_OK . ': ' . $this->intFormat($count[self::MARK_OK]) . '</span>';
        foreach($this->markErrorHtml as $mark => $html)
            if($all || $count[$mark] > 0)
                $markCount[] = $html . ': ' . $this->intFormat($count[$mark]);
        return implode(', ', $markCount);
    }

    protected function markOnly(array $count):string {
        foreach($this->markErrorHtml as $mark => $html)
            if($count[$mark] > 0)
                return $html;
        return $this->markHtml[self::MARK_OK];
    }

    #[Pure] protected function intFormat(int $num):string {
        /** @noinspection PhpRedundantOptionalArgumentInspection */
        return number_format($num, 0, '', ',');
    }

    protected function getTestName(string $test, string $colorClass):string {
        if(empty($test))
            return '¿?';

        $part =  explode('::', trim(str_replace('_', ' ', $test)));
        $testName = count($part) > 1 ? $part[1] : $part[0];
        if(empty($testName))
            return '¿?';
        if(str_starts_with($testName, 'test'))
            $testName = substr($testName, 4);

        if(str_contains($testName, ' with data set "')) {
            $iPos = strpos($testName, '(');
            if($iPos !== false)
                $testName = substr($testName, 0, $iPos-1) .
                    "<span class='dimmed'> " . substr($testName, $iPos) . "</span>";

            return "<span $colorClass>" . str_replace(
                    [
                        ' with data set "',],
                    [", test: \""],
                    $testName
                ) . "</span>";
        }

        return "<span $colorClass>" . $testName . "</span>";
    }

//////////
    public function executeBeforeFirstTest(): void{}

    public function executeAfterLastTest(): void
    {
        $this->endTestClass();
        file_put_contents(
            $this->outputFile,
            $this->htmlStart() . $this->reportTotals($this->countTotals, true) . '<div class="content">' .
            $this->reportProblems() .
            "<fieldset class='resultadosContainer'><legend>&nbsp;Resultados&nbsp;</legend>" .
            implode("\r\n", $this->report) . '</fieldset>' .  $this->htmlEnd()
        );
    }

/** each test */
    public function executeBeforeTest(string $test): void
    {
        $testClass = explode('::', $test)[0];

        if($this->previousTestClass === $testClass)
            return;
        if($this->previousTestClass !== '') {
            $this->endTestClass();
        }

        $this->count = $this->initCounts();
        $this->report[$testClass] = "<details data-tests=''><summary class=''>" . $testClass . "</summary><div><ol>";
        $this->previousTestClass = $testClass;
    }

    protected function endTestClass() {
        $mark = $this->markOnly($this->count);
        $class = $mark === $this->markHtml[self::MARK_OK] ? '' : " class='problemColor'";
        $testNums = json_encode($this->count);
        $this->report[$this->previousTestClass] =
            "<details data-tests='$testNums'>
            <summary $class>" . $mark . ' ' .
            htmlentities($this->previousTestClass ?? '') . ' ' . $this->markAndCount($this->count) . "</summary><div><ol>";
        $this->report[] = "</ol></div></details>";
    }

//////////
    public function executeAfterIncompleteTest(string $test, string $message, float $time): void
    {
        $this->outputTest(self::MARK_INCOMPLETE, $test, $message);
    }

    public function executeAfterRiskyTest(string $test, string $message, float $time): void {
        $this->outputTest(self::MARK_RISKY, $test, $message);
    }

    public function executeAfterTestWarning(string $test, string $message, float $time): void
    {
        $this->outputTest(self::MARK_WARNING, $test, $message);
    }

    public function executeAfterTestError(string $test, string $message, float $time): void
    {
        $this->outputTest(self::MARK_ERROR, $test, $message);
    }

    public function executeAfterTestFailure(string $test, string $message, float $time): void
    {
        $this->outputTest(self::MARK_FAIL, $test, $message);

    }

    public function executeAfterSkippedTest(string $test, string $message, float $time): void
    {
        $this->outputTest(self::MARK_SKIPPED, $test, $message);
    }

    public function executeAfterSuccessfulTest(string $test, float $time): void
    {
        $this->outputTest(self::MARK_OK, $test);
    }

    /**
     * This hook will fire after any test, regardless of the result.
     *
     * For more fine-grained control, have a look at the other hooks
     * that extend PHPUnit\Runner\Hook.
     */
    public function executeAfterTest(string $test, float $time): void{}

}
