<?php
declare(strict_types=1);

/**
 * Tests para ValidateHelper
 *
 * @covers ValidateHelperTest
 * @uses ::strit
 *
 * @version 18/11/2021
 */

use PHPUnit\Framework\TestCase;

// global $usuarios; $usuarios = getForTestsUserIdTypes([]);

class ValidateHelperTest extends TestCase
{

    /**
     * @covers ValidateHelper::validate
     * @covers ValidateHelper::deduceFieldName
     * @covers ValidateHelper::getMessage

     * @dataProvider providerRequiredMessageLabels
     */
    public function testRequiredMessageLabels(array $arrKeyValue, array|string $rule, array $labels, array $messages, array $expected) {
        $this->assertSame($expected,
            ValidateHelper::validate($arrKeyValue, $rule, $labels, $messages, true)
        );
        $this->assertSame($arrKeyValue, $arrKeyValue, "¡Cambio el valor a validar!");
    }

    public function providerRequiredMessageLabels():array {
        return [
            'required con valor' => [
                ['campo_requerido' => 'con valor', 'campo_sin_reglas' => 33],
                ['campo_requerido' => ['required', ]],
                [],
                [],
                []
            ],
            'regla no existe' => [
                ['campo_regla_no_existe' => 'con valor', 'campo_sin_reglas' => 33],
                ['campo_regla_no_existe' => ['reglaNoExiste']],
                [],
                [],
                ['campo_regla_no_existe' => "Para campo_regla_no_existe Regla desconocida: '0'"]
            ],
            'required empty string' => [
                ['campo_requerido' => ''],
                ['campo_requerido' => 'required'],
                [],
                [],
                ['campo_requerido' => 'El campo <b>Campo Requerido</b> es requerido']
            ],
            'required empty string con label' => [
                ['campo_requerido' => ''],
                ['campo_requerido' => ['required']],
                ['campo_requerido' => 'campo obligatorio'],
                [],
                ['campo_requerido' => 'El campo <b>campo obligatorio</b> es requerido']
            ],
            'required empty string con rule label' => [
                ['campo_requerido' => ''],
                ['campo_requerido' => ['required', 'label' => 'campo obligatorio']],
                [],
                [],
                ['campo_requerido' => 'El campo <b>campo obligatorio</b> es requerido']
            ],
            'required empty string con mensaje' => [
                ['campo_requerido' => ''],
                ['campo_requerido' => ['required']],
                [],
                ['campo_requerido' => ['required' => 'Campo Requerido debe ser llenado']],
                ['campo_requerido' => 'Campo Requerido debe ser llenado']
            ],
            'required no viene' => [
                ['otro_campo' => ''],
                ['campo_requerido' => ['required']],
                [],
                [],
                ['campo_requerido' => 'El campo <b>Campo Requerido</b> es requerido']
            ],

        ];
    }

    /**
     * @covers ValidateHelper::validate
     * @covers ValidateHelper::deduceFieldName
     * @covers ValidateHelper::getMessage
     * @covers ValidateHelper::validateYMD
     * @dataProvider provider_number
     */
    public function test_numeric(array $arrKeyValue, array $rule,array $expected) {
        $this->assertSame($expected,
            ValidateHelper::validate($arrKeyValue, $rule)
        );
        $this->assertSame($arrKeyValue, $arrKeyValue, "¡Cambio el valor a validar!");
    }
        public function provider_number() {
            return [
              'numero Ok' => [
                  ['numero' => '3.141592'],
                  ['numero' => 'numeric|required'],
                  []
              ],
                'numero Mal' => [
                    ['numero' => 'Tres'],
                    ['numero' => 'numeric|required'],
                    ['numero' => 'El campo <b>Numero</b> debe ser numérico']
                ],
            ];
        }

    /**
     * @covers ValidateHelper::validate
     * @covers ValidateHelper::deduceFieldName
     * @covers ValidateHelper::getMessage
     * @covers ValidateHelper::validateYMD
     * @dataProvider providerStrings
     */
    public function testStrings( array $arrKeyValue, array $rule,array $expected) {
        $this->assertSame($expected,
            ValidateHelper::validate($arrKeyValue, $rule)
        );
        $this->assertSame($arrKeyValue, $arrKeyValue, "¡Cambio el valor a validar!");
    }

    public function providerStrings():array {
        return [
            'string' => [
                ['string' => 'string'],
                ['string' => 'strings|required'],
                []
            ],
            'string pasa mínimo' => [
                ['string' => 'string'],
                ['string' => 'strings|required|min:2'],
                []
            ],
            'string NO pasa mínimo' => [
                ['string' => 'string'],
                ['string' => 'strings|required|min:22'],
                [ 'string' => 'El campo <b>String</b> debe contener al menos 22 caracteres']
            ],
            'string pasa máximo' => [
                ['string' => 'string'],
                ['string' => 'strings|required|max:22'],
                []
            ],
            'string NO pasa máximo' => [
                ['string' => 'string'],
                ['string' => 'strings|required|max:2'],
                [ 'string' => 'El campo <b>String</b> no debe ser mayor a 2 caracteres']
            ],
            'un float como string' => [
                ['string' => '3.1415'],
                ['string' => 'strings|required'],
                []
            ],
            'un float' => [
                ['string' => 2.7172] ,
                ['string' => 'strings|required'],
                ['string' => 'El campo <b>String</b> debe ser un una cadena de letras']
            ],
            'un array' => [
                ['string' => [1.44]] ,
                ['string' => ['strings', 'required']],
                ['string' => 'El campo <b>String</b> debe ser un una cadena de letras']
            ],
            'empty string required' => [
                ['string'=>''],
                ['string' => ['strings', 'required']],
                ['string' => 'El campo <b>String</b> es requerido']
            ]
        ];
    }

    /**
     * @covers ValidateHelper::validate
     * @covers ValidateHelper::deduceFieldName
     * @covers ValidateHelper::getMessage

     * @dataProvider providerIntegers
     */
    public function testIntegers(array $rule, array $arrKeyValue, array $expected) {
        $this->assertSame($expected,
            ValidateHelper::validate($arrKeyValue, $rule)
        );
        $this->assertSame($arrKeyValue, $arrKeyValue, "¡Cambio el valor a validar!");
    }

    public function providerIntegers():array {
        return [
            'int positivo' => [
                ['integer' => [ 'int', 'scale'=>2]],
                ['integer' => 3],
                []
            ],
            'int as string positivo' => [
                ['integer' => [ 'int']],
                ['integer' => '33456'],
                []
            ],
            'cero' => [
                ['integer' => [ 'int', 'min:0']],
                ['integer' => 0],
                []
            ],
            'cero as string' => [
                ['integer' => [ 'int', 'min:0']],
                ['integer' => '0'],
                []
            ],
            'int negativo' => [
                ['integer' => [ 'int']],
                ['integer' => -1],
                []
            ],
            'int as string negativo' => [
                ['integer' => [ 'int']],
                ['integer' => '-2'],
                []
            ],

            'empty string' => [
                ['integer' => [ 'int', 'min:0']],
                ['integer' => ''],
                []
            ],

            "'10.00'" => [
                ['integer' => [ 'int']],
                ['integer' => '10.00'],
                []
            ],

            "'-10.00'" => [
                ['integer' => [ 'int']],
                ['integer' => '-10.00'],
                []
            ],

            "no numerico" => [
                ['integer' => [ 'int']],
                ['integer' => 'asdf'],
                ['integer' => 'El campo <b>Integer</b> debe ser numérico']
            ],

            "con decimales" => [
                ['integer' => [ 'int']],
                ['integer' => 5.001],
                ['integer' => 'El campo <b>Integer</b> debe ser un número entero']
            ],

            "con decimales negativo" => [
                ['integer' => [ 'int']],
                ['integer' => -5.101],
                ['integer' => 'El campo <b>Integer</b> debe ser un número entero']
            ],
            "con decimales negativo 2" => [
                ['integer' => [ 'int']],
                ['integer' => -9.999],
                ['integer' => 'El campo <b>Integer</b> debe ser un número entero']
            ],

            'checa equal pero es desigual' => [
                ['integer' => [ 'int', 'equal' => '1']],
                ['integer' => 11],
                ['integer' => 'El valor para el campo <b>Integer</b> no coincide']
            ],

            'equal' => [
                ['integer' => [ 'int', 'equal' => 11]],
                ['integer' => '11'],
                []
            ],

            'checa is_equal pero es desigual' => [
                ['integer' => [ 'int', 'is_equal' => '1']],
                ['integer' => 11],
                ['integer' => 'El valor para el campo <b>Integer</b> no coincide con el esperado']
            ],

            'is_equal' => [
                ['integer' => [ 'int', 'equal' => 11]],
                ['integer' => '11'],
                []
            ],

            'bajo el mínimo' => [
                ['integer' => [ 'int', 'min' => 0]],
                ['integer' => -1],
                ['integer' => 'El campo <b>Integer</b> debe ser mayor o igual a 0']
            ],

            'mayor al máximo' => [
                ['integer' => [ 'int', 'max' => '111']],
                ['integer' => '999'],
                ['integer' => 'El campo <b>Integer</b> debe ser menor o igual a 111']
            ],

            'int en algunos valores' => [
                ['integer' => [ 'int', 'in' => ['2', '4', '6', '8']]],
                ['integer' => '8'],
                []
            ],
            'int no en algunos valores' => [
                ['integer' => [ 'int', 'in' => ['2', '4', '6', '8']]],
                ['integer' => '9'],
                ['integer' => 'El valor para el campo <b>Integer</b> no existe']
            ],
        ];
    }


    /**
     * @covers ValidateHelper::validate
     * @covers ValidateHelper::deduceFieldName
     * @covers ValidateHelper::getMessage
     * @covers ValidateHelper::validateYMD
     * @dataProvider providerDates
     */
    public function testDates( array $arrKeyValue, array $rule,array $expected) {
        $this->assertSame($expected,
            ValidateHelper::validate($arrKeyValue, $rule)
        );
        $this->assertSame($arrKeyValue, $arrKeyValue, "¡Cambio el valor a validar!");
    }

    public function providerDates():array {
        return [
            'recibe de mas' => [
                ['de_mas' => '2021-1-2-1' ],
                ['de_mas' => ['date'] ],
                ['de_mas' => 'El campo <b>De Mas</b> debe ser una fecha válida']
            ],

            'date MAL mes' => [
                ['mes_letras' => '03-fe-2021' ],
                ['mes_letras' => ['date'] ],
                ['mes_letras' => 'El campo <b>Mes Letras</b> debe ser una fecha válida']
            ],
            'date MAL letras' => [
                ['letras' => 'lu-feb-delPresente' ],
                ['letras' => ['date'] ],
                ['letras' => 'El campo <b>Letras</b> debe ser una fecha válida']
            ],
            'recibe mes-ano' => [
                ['mes_ano' => '03-21' ],
                ['mes_ano' => ['date'] ],
                ['mes_ano' => 'El campo <b>Mes Ano</b> debe ser una fecha válida']
            ],

            'date MAL d/m/a' => [
                ['fecha_alta' => '30-12-2021' ],
                ['fecha_alta' => ['date', 'min'=>Date('Y-m-d', strtotime('tomorrow'))] ],
                ['fecha_alta' => 'El campo <b>Fecha Alta</b> debe ser una fecha válida']
            ],
            'date MAL m/d/a' => [
                ['fecha_alta' => '12-30-2021' ],
                ['fecha_alta' => ['date'] ],
                ['fecha_alta' => 'El campo <b>Fecha Alta</b> debe ser una fecha válida']
            ],
            'date MAL 2' => [
                ['fecha_alta' => '2021-3-4' ],
                ['fecha_alta' => ['date'] ],
                ['fecha_alta' => 'El campo <b>Fecha Alta</b> debe ser una fecha válida']
            ],
            'date mes 13' => [
                ['mes_13' => '2021-13-14' ],
                ['mes_13' => ['date'] ],
                ['mes_13' => 'El campo <b>Mes 13</b> debe ser una fecha válida']
            ],
            'date Ok' => [
              ['fecha_alta' => '2021-02-03' ],
              ['fecha_alta' => ['date'] ],
              []
            ],
            'date MinMax Ok' => [
                ['fecha_minMax_ok' => '2021-02-03' ],
                ['fecha_minMax_ok' => ['date', 'min'=>'2021-01-01', 'max'=>'2021-12-31'] ],
                []
            ],
            'date Mínima Mal' => [
                ['fecha_min_mal' => '2021-02-03' ],
                ['fecha_min_mal' => ['date', 'min'=>'2022-01-01'] ],
                ['fecha_min_mal' => 'El campo <b>Fecha Min Mal</b> debe ser mayor o igual 2022-01-01']
            ],
            'date Máxima Mal' => [
                ['fecha_max_mal' => '2021-02-03' ],
                ['fecha_max_mal' => ['date', 'max'=>'2021-01-01'] ],
                ['fecha_max_mal' => 'El campo <b>Fecha Max Mal</b> debe ser menor o igual a 2021-01-01']
            ],
            'date hoy, máximo' => [
                ['hoy_maximo_hoy' => Date('Y-m-d') ],
                ['hoymaximo_hoy' => ['date', 'max'=>Date('Y-m-d')] ],
                []
            ],
            'date hoy, máximo es date pero llega dateTime' => [
                ['hoy_maximo_dateTime' => Date('Y-m-d') ],
                ['hoy_maximo_dateTime' => ['date', 'max'=>Date('Y-m-d H:i:s')] ],
                []
            ],
            'date hoy, mínimo es date pero llega dateTime' => [
                ['hoy_min_dateTime' => Date('Y-m-d') ],
                ['hoy_min_dateTime' => ['date', 'min'=>Date('Y-m-d H:i:s')] ],
                []
            ],
            'date Máximo rule today mando mañana' => [
                ['fecha_maximo_hoy_mal' => Date('Y-m-d', strtotime("tomorrow")) ],
                ['fecha_maximo_hoy_mal' => ['date', 'max'=>'today'] ],
                ['fecha_maximo_hoy_mal' => 'El campo <b>Fecha Maximo Hoy Mal</b> debe ser menor o igual a '.Date('Y-m-d')]
            ],
            'date Máximo rule today mando hoy' => [
                ['fecha_maximo_hoy_ok' => Date('Y-m-d' ) ],
                ['fecha_maximo_hoy_ok' => ['date', 'max'=>'today'] ],
                []
            ],
                /*
            'date Time Ok' => [
                ['fecha_hora_alta' => '2021-02-03 23:11:12' ],
                ['fecha_hora_alta' => ['date'] ],
                []
            ],
            */
            'valida' =>[
                ['fecha' => '2022-01-02'],
                ['fecha' => 'date|required'],
                []
            ],
            'invalida' =>[
                ['fecha' => '2022-02-30'],
                ['fecha' => 'date|required'],
                ['fecha' => 'El campo <b>Fecha</b> debe ser una fecha válida']
            ],
            'invalida es dateTime' =>[
                ['fecha' => '2022-01-02 10:11:12'],
                ['fecha' => 'date|required'],
                ['fecha' => 'El campo <b>Fecha</b> debe ser una fecha válida']
            ],
            'empty required' =>[
                ['fecha' => ''],
                ['fecha' => 'date|required'],
                ['fecha' => 'El campo <b>Fecha</b> debe ser una fecha válida']
            ],
            'empty optional' =>[
                ['fecha' => ''],
                ['fecha' => 'date'],
                []
            ],
        ];
    }


    /**
     * @covers ValidateHelper::fieldit
     * @uses ::strim
     * @throws ReflectionException
     */
    public function test_fieldit() {
        $v = new ValidateHelper();
        $method = getProtectedMethod('fieldit', $v);
        $in = [
            'campo' => 'campo_Ok',
            'campo Quoted' => '`campo_Ok`',
            'campo sin trim' => '  campo_Ok  ',
            'tabla.campo' => 'tabla.campo',
            'tabla.campo Quoted' => '`tabla`.`campo`',
            'espacios' => 'ta bla.campo',
            'con injection' => '`;`delete from tabla`;.campo',
        ];
        $expected = [
            'campo' => '`campo_Ok`',
            'campo Quoted' => '`campo_Ok`',
            'campo sin trim' => '`campo_Ok`',
            'tabla.campo' => '`tabla`.`campo`',
            'tabla.campo Quoted' => '`tabla`.`campo`',
            'espacios' => '`ta bla`.`campo`',
            'con injection' => '`;delete from tabla;`.`campo`',
        ];
        foreach($in as $k => $d)
            $this->assertEquals($expected[$k], $method->invoke(null,$d), $k);
    }
}
