<?php
declare(strict_types=1);

/**
 * Tests para CNumeroaLetra
 *
 * @covers CNumeroaLetraTest
 * @covers CNumeroaLetra
 *
 * @version 18/10/2021
 */

use PHPUnit\Framework\TestCase;



class CNumeroaLetraTest extends TestCase {

    /**
     * @covers CNumeroaLetra
     * @covers CNumeroaLetra::__toString
     * @uses ::strim
     */
    public function testDefaults() {
        $cn = new CNumeroaLetra();
        $cn->setNumero(1515);
        $this->assertEquals(strim("*** MIL QUINIENTOS QUINCE PESOS 00/100 M.N. ***"), strim($cn->letra())
        , "Default settings");
        $cn->setNumero(11012013014);
        $this->assertEquals(strim("*** ONCE MIL DOCE MILLONES TRECE MIL CATORCE PESOS 00/100 M.N. ***"),
            strim($cn->letra())
            , "11,012,013,014");
        $cn->setNumero(15016017018);
        $this->assertEquals(strim("*** QUINCE MIL DIECISEIS MILLONES DIECISIETE MIL DIECIOCHO PESOS 00/100 M.N. ***"), strim($cn->letra())
            , "15,016,017,018");

        $cn->setNumero(30);
        $this->assertEquals(strim("*** TREINTA PESOS 00/100 M.N. ***"), strim($cn->__toString()),
         "30 to string");
    }


    /**
     * @covers CNumeroaLetra
     * @uses ::strim
     * @dataProvider  providerCNumeroLetra
     *
     */
    public function testCNumeroLetra($numero, $prefijo, $sufijo, $moneda, $genero, $mayusculas, $expected) {
        $cn = new CNumeroaLetra();
        $cn->setNumero($numero);
        $cn->setPrefijo($prefijo);
        $cn->setSufijo($sufijo);
        $cn->setMoneda($moneda);
        $cn->setGenero($genero);
        $cn->setMayusculas($mayusculas);
        $this->assertEquals(strim($expected), strim($cn->letra()));
    }

    public function providerCNumeroLetra() {
        return [
            'cero' => [0, 'preFijo ', 'suFijio', 'moneda', 0, 1,
                'PREFIJO CERO  MONEDA 00/100 SUFIJIO'],
            '1.23 femenino' => [1.23, 'preFijo ', 'suFijio', 'moneda', 0, 1,
                'PREFIJO UNA  MONEDA 23/100 SUFIJIO'],
            '1.99 masculinio' => [1.99, 'preFijo ', 'suFijio', 'moneda', 1, 1,
                'PREFIJO UNO  MONEDA 99/100 SUFIJIO'],

            'once' => [11, 'preFijo ', 'suFijio', 'moneda', 1, 1,
                'PREFIJO ONCE MONEDA 00/100 SUFIJIO'],

            '12.38 masculinio' => [12.38, 'preFijo ', 'suFijio', 'moneda', 1, 1,
                'PREFIJO DOCE  MONEDA 38/100 SUFIJIO'],

            '34.56 femenino' => [34.56, 'preFijo ', 'suFijio', 'moneda', 0, 1,
                'PREFIJO TREINTA Y CUATRO MONEDA 56/100 SUFIJIO'],

            '454.56 femenino' => [454.01, 'preFijo ', 'suFijio', 'moneda', 0, 1,
                'PREFIJO CUATROCIENTAS CINCUENTA Y CUATRO MONEDA 01/100 SUFIJIO'],

            '1500.01 femenino' => [1500.01, 'preFijo ', 'suFijio', 'moneda', 0, 1,
                'PREFIJO MIL QUINIENTAS MONEDA 01/100 SUFIJIO'],

            '1500.01 masc' => [1500.01, 'preFijo ', 'suFijio', 'moneda', 1, 1,
                'PREFIJO MIL QUINIENTOS MONEDA 01/100 SUFIJIO'],

            '14518.01 femenino' => [14518.01, 'preFijo ', 'suFijio', 'moneda', 0, 1,
                'PREFIJO CATORCE MIL QUINIENTAS DIECIOCHO MONEDA 01/100 SUFIJIO'],

            '234,567,890.00 minusculas' => ['234567890.00', 'preFijo ', 'suFijio', 'moneda', 0, 0,
                'prefijo doscientas treinta y cuatro millones quinientas sesenta y siete mil ochocientas noventa moneda 00/100 sufijio'],

        ];
    }

}
