<?php /** @noinspection PhpIllegalPsrClassPathInspection */

declare(strict_types=1);

use PHPUnit\Framework\TestCase;
use Vitex\Tester;

/**
 *
 * @covers MovBodegaDo
 * @uses Iac\inc\sql\IacMysqli
 * @uses ::assoc_mysql2mysqli
 * @uses ::ia_singleton
 * @uses ::ia_singleread
 * @uses sqlMysqli::log_trace
 * @uses ::strit
 *
 */
class MovBodegaDoTest extends TestCase
{

    /**
     * @covers MovBodegaDo::producto_bodega_id_deduce
     * @uses ::ia_guid
     */
    public function test_producto_bodega_id_deduce() {
        $bodega_id = ia_guid();
        $producto_general_id = ia_guid();
        $color_id = ia_guid();
        $mov = new MovBodegaDo();
        $this->assertEquals(
            $bodega_id. "_" . $producto_general_id . "_" . $color_id,
            $mov->producto_bodega_id_deduce($bodega_id, $producto_general_id, $color_id)
        );
    }

    /**
     * @covers MovBodegaDo::arrayKeyToSqlIn
     *
     * @dataProvider provider_arrayKeyToSqlIn
     */
    public function test_arrayKeyToSqlIn(array $arr, string $expected) {
        $mov = new MovBodegaDo();
        $method = getProtectedMethod('arrayKeyToSqlIn', $mov);
        $this->assertEquals($expected, $method->invokeArgs($mov, [$arr]) );
    }
        public function provider_arrayKeyToSqlIn():array {
            return [
                'Sin elementos' => [[], "('  ')"],
                'Un elemento' => [[1], "('0')"],
                'Varios elementos' => [[1, 'dos'=>'2', "pac'man"=>3], "('0','dos','pac''man')"],
            ];
        }

    /**
     * @covers MovBodegaDo::origenForzaGrupo
     */
    public function test_origenForzaGrupo() {
        $mov = new MovBodegaDo();
        $method = getProtectedMethod('origenForzaGrupo', $mov);
        $origenNoEsBodega = ia_singleread("SELECT origen_bodega_id FROM origen_bodega WHERE es='TIENDA'");

        $origenBodegaIdGrupoEmpty = ia_singleread(
            "SELECT ob.origen_bodega_id 
                    FROM origen_bodega ob JOIN bodega b ON ob.bodega_id=b.bodega_id 
                    WHERE es='BODEGA' AND b.activo='Si' AND b.grupo='' ", 'DB ERROR O NO ENCONTRE');
        if(empty($origenBodegaIdGrupoEmpty))
            $origenBodegaIdGrupoEmpty = 'DB ERROR O NO ENCONTRE';

        $bodegaConGrupo = ia_singleton(
            "SELECT ob.origen_bodega_id, grp.origen_bodega_id as grupo_origen_bodega_id
                    FROM origen_bodega ob JOIN bodega b ON ob.bodega_id=b.bodega_id 
                     JOIN origen_bodega grp ON grp.clave = b.grupo AND grp.es = 'GRUPO BODEGA'
                    WHERE ob.es='BODEGA' AND b.activo='Si' AND b.grupo<>'' AND b.grupo<>'Sin Grupo' ");

        $tests = [
          'empty origen_id' => ['origen_id'=> '','expected' => ''],
          'origen_id NO es bodega' => ['origen_id'=> $origenNoEsBodega,'expected' => $origenNoEsBodega],
          'origen_id es bodega con grupo en blanco' => ['origen_id'=> $origenBodegaIdGrupoEmpty, 'expected' => $origenBodegaIdGrupoEmpty],
          'origen_id es bodega con grupo puesto' => ['origen_id'=> $bodegaConGrupo['origen_bodega_id'], 'expected' => $bodegaConGrupo['grupo_origen_bodega_id']],
        ];
        $bodegaConGrupoSinGrupo = ia_singleread(
            "SELECT ob.origen_bodega_id
                    FROM origen_bodega ob JOIN bodega b ON ob.bodega_id=b.bodega_id 
                    WHERE ob.es='BODEGA' AND b.activo='Si' AND b.grupo<>'' AND b.grupo='Sin Grupo' ");

        $tests['bodega SIN GRUPO en grupo dice SIN GRUPO'] = ['origen_id'=>$bodegaConGrupoSinGrupo, 'expected'=>$bodegaConGrupoSinGrupo];
        foreach($tests as $caseName => $data)
            $this->assertEquals($data['expected'], $method->invokeArgs($mov, [$data['origen_id']]),$caseName );
    }

    /**
     * @covers MovBodegaDo::validaOrigenId
     */
    public function test_validaOrigenId() {
        $mov = new MovBodegaDo();
        $method = getProtectedMethod('validaOrigenId', $mov);
        $bodega_id = ia_singleread("SELECT bodega_id FROM bodega WHERE activo='Si' AND grupo NOT IN ('china', 'pedido')");
        $origen_misma_bodega_id = ia_singleread("SELECT origen_bodega_id FROM origen_bodega WHERE bodega_id=" . strit($bodega_id));
        $origen_otra_bodega_id = ia_singleread("SELECT origen_bodega_id FROM origen_bodega WHERE es='bodega' AND bodega_id<>" . strit($bodega_id));
        $origen_tienda_id = ia_singleread("SELECT origen_bodega_id FROM origen_bodega WHERE es='tienda'");

        //@TODO? $this->assertFalse($method->invokeArgs($mov, [$bodega_id, 'no existo']), 'origen_id invalido');
        $this->assertFalse($method->invokeArgs($mov, [$bodega_id, $origen_misma_bodega_id]), 'origen_id es la misma bodega');
        $this->assertTrue($method->invokeArgs($mov, [$bodega_id, $origen_otra_bodega_id]), 'origen_id es ok es otra bodega');
        $this->assertTrue($method->invokeArgs($mov, [$bodega_id, $origen_tienda_id]), 'origen_id es ok es tienda');
        //@TODO falta tienda/bodega inactiva?
    }

    /**
     * @covers MovBodegaDo::origen2TipoMovimiento
     * @dataProvider provider_origen2TipoMovimiento
     */
    public function test_origen2TipoMovimiento($nota, $expected) {
        $this->assertEquals($expected, MovBodegaDo::origen2TipoMovimiento($nota));
    }

    public function provider_origen2TipoMovimiento():array {
        $es = ia_sqlKeyValue('SELECT es, origen_bodega_id FROM origen_bodega');
        return [
            'Bodega de Bodega, Entrada'=>[['entrada_salida'=>'Entrada','origen_id' => $es['BODEGA'] ], 'TRASLADO'],
            'Bodega a Bodega, Salida'=>[['entrada_salida'=>'salida','origen_id' => $es['BODEGA'] ], 'TRASLADO'],
            'Bodega de Grupo, Entrada'=>[['entrada_salida'=>'Entrada','origen_id' => $es['GRUPO BODEGA'] ], 'TRASLADO'],
            'Bodega a Grupo, Salida'=>[['entrada_salida'=>'salida','origen_id' => $es['GRUPO BODEGA'] ], 'TRASLADO'],
            'Tienda devolucion'=>[['entrada_salida'=>'Entrada','origen_id' => $es['TIENDA'] ], 'DEVOLUCION'],
            'Tienda venta'=>[['entrada_salida'=>'Salida','origen_id' => $es['TIENDA'] ], 'MOVIMIENTO'],
            'Cliente devolucion'=>[['entrada_salida'=>'Entrada','origen_id' => $es['CLIENTE'] ], 'DEVOLUCION'],
            'Cliente venta'=>[['entrada_salida'=>'Salida','origen_id' => $es['CLIENTE'] ], 'VENTA CLIENTE'],
            'Entrada de fabrica'=>[['entrada_salida'=>'Entrada','origen_id' => $es['IMPORTACION'] ], 'CONTAINER'],
            'Salida a fabrica'=>[['entrada_salida'=>'Salida','origen_id' => $es['IMPORTACION'] ], 'DEVOLUCION FABRICANTE'],
            'Correccion'=>[['entrada_salida'=>'Salida','origen_id' => $es['CORRECCION'] ], 'CORRECCION'],
            'Origen nota found'=>[['entrada_salida'=>'Salida','origen_id' => 'xx' ], 'CORRECCION'],
        ];
    }

    /**
     * @covers MovBodegaDo::limpiaMovimiento
     * @covers ::strim
     */
    public function test_limpiaMovimiento() {
        $mov = new MovBodegaDo();
        $method = getProtectedMethod('limpiaMovimiento', $mov);
        $movimiento =[
          'chofer_responsable' => '  chofer_responsable sin    uppercase    ',
          'cliente' => ' cliente  sin    uppercase ',
          'algo_id' => ' termina en _id  sin strim ni uppercase ',
          'rollos' => '1,234',
          'quantity' => '1,234,567.89',
            'un array' => [' solo le    hace strim  ', 'a todos    sus elementos'],
            'string' => '   otro string    strim y upper CASE'
        ];
        $expected =[
            'chofer_responsable' => 'chofer_responsable sin uppercase',
            'cliente' => 'cliente sin uppercase',
            'algo_id' => ' termina en _id  sin strim ni uppercase ',
            'rollos' => '1234',
            'quantity' => '1234567.89',
            'un array' => ['solo le hace strim', 'a todos sus elementos'],
            'string' => 'OTRO STRING STRIM Y UPPER CASE'
        ];
        $this->assertEquals($expected, $method->invokeArgs($mov, [$movimiento]) );

    }

    /**
     * @covers MovBodegaDo::bodegaIdNumeroToNotaBodegaId
     */
    public function test_bodegaIdNumeroToNotaBodegaId() {

        $mov = new MovBodegaDo();
        $nota_bodega = ia_singleton('SELECT nota_bodega_id, bodega_id, numero FROM nota_bodega LIMIT 1');
        if(empty($nota_bodega))
            $this->assertEquals('un id', 'Error en el query o not found');
        else {
            $nota_bodega_id = $mov->bodegaIdNumeroToNotaBodegaId($nota_bodega['bodega_id'], $nota_bodega['numero']);
            $this->assertEquals($nota_bodega['nota_bodega_id'], $nota_bodega_id, 'En nota_bodega');
            $no_existe = $mov->bodegaIdNumeroToNotaBodegaId($nota_bodega['bodega_id'], '-1');
            $this->assertEquals('', $no_existe, 'En nota_bodega no existe el numero');
            $no_existe = $mov->bodegaIdNumeroToNotaBodegaId('NO EXISTE', '1');
            $this->assertEquals('', $no_existe, 'En nota_bodega no existe la bodega');
        }
    }


    /**
     * @covers MovBodegaDo::bodegaIdUltimoNumeroToNotaBodegaId
     */
    public function test_bodegaIdUltimoNumeroToNotaBodegaId() {
        $maxNum = ia_singleton("SELECT bodega_id, nota_bodega_id FROM nota_bodega ORDER BY bodega_id, numero DESC LIMIT 1");
        if(empty($maxNum))
            $this->assertEquals("Necesito notas registradas", "Prueba no ejecutada, sql error or on data?");
        $mov = new MovBodegaDo();
        $this->assertEquals($maxNum['nota_bodega_id'], $mov->bodegaIdUltimoNumeroToNotaBodegaId($maxNum['bodega_id']));
    }


}
