<?php


/** @noinspection PhpArrayShapeAttributeCanBeAddedInspection */
/** @noinspection SqlResolve */

declare(strict_types=1);

use Iac\inc\sql\IacSqlException;
use PHPUnit\Framework\TestCase;

/**
 * @covers sqlMysqli
 * @covers sqlMysqli::__construct
 * @covers sqlMysqli::log_sql_error
 * @covers sqlMysqli::log_trace
 * @covers Iac\inc\sql\IacMysqli
 *
 * @uses ::assoc_mysql2mysqli
 */
class SqlConverterTest  extends TestCase {


    /**
     * Crea tabla testIaMysqli para estas pruebas.
     *
     * Corre antes de todas las pruebas de esta clase
     * Pone bodegas y usuarios de prueba
     */
    public static function setUpBeforeClass(): void {
        $method = __METHOD__;
        ia_query("DROP /*$method*/ TABLE IF EXISTS testIaMysqli");
        $testTable = "CREATE/*$method*/ TABLE IF NOT EXISTS testIaMysqli(
            id int unsigned not null primary key auto_increment,
            str varchar(32) NOT NULL DEFAULT '',
            num decimal(12,2) NOT NULL DEFAULT '0.00',
            entero int not null default '0',
            null_value int null default null 
        )";
        ia_query($testTable);

        ia_query("insert /*$method*/ into testIaMysqli(id,str,num,entero,null_value) VALUES
            (1, 'uno', 1.11, 11, 1),
            (2, 'dos con null', 2, 22, null),
            (3, 'para rollback', 33, 33, null)
        ");
    }

    /**
     * Corre al terminarse todas las pruebas de esta clase
     * Elimina bodegas y usuarios de prueba
     */
    public static function tearDownAfterClass(): void {
        ia_query("DROP TABLE IF EXISTS testIaMysqli");
    }

////////////////

    /**
     * @covers ::ia_begin()
     * @covers ::ia_query()
     * @covers ::ia_commit()
     * @uses ::ia_singleread()
     *
     * @dataProvider provider_doStatementOrSqlErrors
     */
    public function test_commit(bool $testSqlError) {
        global $gSqlClass;
        $this->assertFalse(ia_query("UPDATE testIaMysqli SET num=33 WHERE id=3"), 'Dio error el primer statement, no pude probar');
        $this->assertFalse(ia_begin(), "Error en el commit");

        $this->assertEquals('0', ia_singleread('select @@autocommit'), 'Autocommit paso a cero');

        $this->assertEquals(1, $gSqlClass->begins_get(), 'Cuenta de begins, al iniciar begin');
        $this->assertFalse(ia_query("UPDATE testIaMysqli SET num=44 WHERE id=3"), 'Dio error el primer statement, no pude probar');
        if($testSqlError) {
            $this->assertTrue(ia_query("UPDATE testIaMysqli SET numx=55 WHERE id=3"), 'SQL Error durante begin ');
            $this->assertFalse(ia_commit(), "Commit después de sql error");
            $this->assertEquals(
                '44.00',
                ia_singleread("SELECT num FROM testIaMysqli WHERE id=3"),
                'En error el usuario hace commit!'
            );
        } else {
            $this->assertFalse(ia_query("UPDATE testIaMysqli SET num=55 WHERE id=3"));
            $this->assertFalse(ia_commit(), "Commit sin error");
            $this->assertEquals(
                '55.00',
                ia_singleread("SELECT num FROM testIaMysqli WHERE id=3"),
                'En sql se realizo el rollback'
            );
        }

        $this->assertEquals('1', ia_singleread('select @@autocommit') , 'Autocommit regreso a uno');
        $this->assertEquals(0, $gSqlClass->begins_get(), 'Cuenta de begins, regreso a cero después del rollback');
    }

    /**
     * @covers ::ia_begin()
     * @covers ::ia_query()
     * @covers ::ia_rollback()
     * @uses ::ia_singleread()
     *
     * @dataProvider provider_doStatementOrSqlErrors
     */
    public function test_rollback(bool $testSqlError) {
        global $gSqlClass;
        $this->assertFalse(ia_query("UPDATE testIaMysqli SET num=33 WHERE id=3"), 'Dio error el primer statement, no pude probar');
        $this->assertFalse(ia_begin());

        $this->assertEquals('0', ia_singleread('select @@autocommit'), 'Autocommit paso a cero');

        $this->assertEquals(1, $gSqlClass->begins_get(), 'Cuenta de begins, al iniciar begin');
        $this->assertFalse(ia_query("UPDATE testIaMysqli SET num=44 WHERE id=3"), 'Dio error el primer statement, no pude probar');
        if($testSqlError) {
            $this->assertTrue(ia_query("UPDATE testIaMysqli SET numx=55 WHERE id=3"), 'SQL Error durante begin ');
            $this->assertFalse(ia_rollback(), "Rollback por el error");
        } else {
            $this->assertFalse(ia_query("UPDATE testIaMysqli SET num=55 WHERE id=3"));

            $this->assertFalse(ia_rollback(), "Rollback sin error");
        }
        $this->assertEquals(
            '33.00',
            ia_singleread("SELECT num FROM testIaMysqli WHERE id=3"),
            'En sql se realizo el rollback'
        );
        $this->assertEquals('1', ia_singleread('select @@autocommit') , 'Autocommit regreso a uno');
        $this->assertEquals(0, $gSqlClass->begins_get(), 'Cuenta de begins, regreso a cero después del rollback');
    }

        public function provider_doStatementOrSqlErrors():array {
            return [
                'Sql statements' => [false],
                'Sql error' => [true],
            ];
        }

    /**
     * @covers sqlMysqli
     */
    public function test_sqlMysqli() {
        global $gIAsql, $gSqlClass;
        $gSqlClass->close();
        $gSqlClass = new sqlMysqli($gIAsql['host'],$gIAsql['user'],$gIAsql['pwd'],$gIAsql['dbname'],$gIAsql['port']);
        try {
            $gSqlClass->connect();
            $this->assertTrue(true);
        } catch (IacSqlException) {
            $this->fail("Error al conectarse a la db");
        }

    }
    /**
     * @covers ::ia_insertIded
     * @covers ::ia_query
     * @covers ::ia_singleton()
     * @covers ::last_inserted_id()
     * @uses ::strit()
     * @dataProvider provider_ia_insertIded()
     */
    public function test_ia_insertIded(string $sql, string|false $expected, array $expectedRow) {
        $error = ia_query("ALTER TABLE testIaMysqli AUTO_INCREMENT = 15");
        $this->assertFalse($error, "ia_query error al preparar ia_insertIded test");
        if($error)
            return;
        $id = ia_insertIded($sql);
        $this->assertEquals($expected, $id);
        if($expected === false)
            return;
        $this->assertEquals($id, last_inserted_id());

        $this->assertEquals(
            $expectedRow,
            ia_singleton("SELECT id,str,num FROM testIaMysqli WHERE id=" . strit($expected) ),
            "inserted row read ok"
        );
        $this->assertFalse(ia_query("DELETE FROM testIaMysqli WHERE id=" . strit($expected) ),
            "ia_query delete from DELETE FROM testIaMysqli WHERE id= ".strit($expected)
        );
    }


        public function provider_ia_insertIded():array {
        return [
            'Error' => ['INSERT INTO testIaMysqli VALUES(1,2)',
                false, []
            ],
            'Insert and getId: ' => ["INSERT INTO testIaMysqli(str,num) VALUES('id=15, num=16',16)",
                '15', ['id'=>'15', 'str' => 'id=15, num=16', 'num' => '16.00']
            ],
        ];
    }

    /**
     * @covers ::ia_singleread
     *
     * @dataProvider provider_ia_singleread
     */
    public function test_ia_singleread(string $sql, string $dflt, string|null|false $expected) {
        $this->assertEquals($expected, ia_singleread($sql, $dflt));
    }

        public function provider_ia_singleread():array {
            return [
                'Not found' => [
                    "SELECT * FROM testIaMysqli WHERE id = 0", 'NOT FOUND',
                    'NOT FOUND',
                ],
                'Error' => [
                    "SELECT * FROM testIaMysqlix WHERE id = tranca", '',
                    false,
                ],
                'select 1 column' => [
                    "SELECT id FROM testIaMysqli WHERE id = 2", '',
                    '2',
                ],
                'select * ' => [
                    "SELECT * FROM testIaMysqli", '',
                    '1',
                ],
            ];
        }

    /**
     * @covers ::ia_singleton()
     *
     * @dataProvider provider_ia_singleton
     */
    public function test_ia_singleton(string $sqlQuery, int $assoc, array|false $expected) {
        $this->assertEquals(
            $expected,
            ia_singleton($sqlQuery, $assoc)
        );
    }

        public function provider_ia_singleton():array {
        return [
            'MYSQLI_NUM, con nulls' => [
                "SELECT * FROM testIaMysqli WHERE id = 2",
                MYSQLI_NUM, ['2', 'dos con null', '2.00', '22', null],
            ],
            'MYSQLI_ASSOC, con nulls' => [
                "SELECT id, str, null_value FROM testIaMysqli WHERE id = 2",
                MYSQLI_ASSOC, ['id' => '2', 'str' => 'dos con null', 'null_value' => null],
            ],
            'Not found' => [
                "SELECT * FROM testIaMysqli WHERE id = 0",
                MYSQLI_ASSOC, [],
            ],
            'Error' => [
                "SELECT * FROM testIaMysqlix WHERE id = tranca",
                MYSQLI_ASSOC, false,
            ],
            'MYSQLI_NUM, primer renglón encontrado' => [
                "SELECT * FROM testIaMysqli WHERE id <= 2 ORDER BY id desc",
                MYSQLI_NUM, ['2', 'dos con null', '2.00', '22', null],
            ],
            'MYSQLI_ASSOC, primer renglón encontrado' => [
                "SELECT id, str, null_value FROM testIaMysqli WHERE id <= 2 ORDER BY id DESC",
                MYSQLI_ASSOC, ['id' => '2', 'str' => 'dos con null', 'null_value' => null],
            ],
        ];
    }

    /**
     * @covers ::ia_singletonFull()
     *
     * @dataProvider provider_ia_singletonFull
     */
    public function test_ia_singleton_full(string $sqlQuery, mixed $enNull, array|false $expected) {
        $this->assertEquals(
            $expected,
            ia_singletonFull($sqlQuery, $enNull)

        );
    }

        public function provider_ia_singletonFull():array {
        return [
            'con nulls' => [
                "SELECT id, str, null_value FROM testIaMysqli WHERE id = 2",
                33, ['id' => '2', 'str' => 'dos con null', 'null_value' => null],
            ],
            'Not found' => [
                "SELECT id, str, null_value FROM testIaMysqli WHERE id = 0",
                null, ['id' => null, 'str' => null, 'null_value' => null],
            ],
            'Error' => [
                "SELECT * FROM testIaMysqlix WHERE id = tranca",
                33, false,
            ],
            'Primer renglón encontrado' => [
                "SELECT * FROM testIaMysqli WHERE id <= 2 ORDER BY id desc",
                33, ['id' => '2', 'str' => 'dos con null', 'num' => '2.00', 'entero' => '22', 'null_value' => null],
            ],

        ];
    }

    /**
     * @covers ::ia_sqlKeyValue
     */
    public function test_ia_sqlKeyValue() {
        $this->assertEquals(
            [
                '1' => 'uno',
                '2' => 'dos con null',
                ],
            ia_sqlKeyValue("SELECT id, str FROM testIaMysqli WHERE id <= 2 ORDER BY id"),
            "Obten key value"
        );
        $this->assertEquals(
            false,
            ia_sqlKeyValue("SELECT id, str, campoInvalido FROM testIaMysqlix WHERE id <= 2 ORDER BY id"),
            "Error"
        );

    }

    /**
     * @covers ::ia_sqlVector
     */
    public function test_ia_sqlVector() {
        $this->assertEquals(
            [
               'uno',
                'dos con null',
            ],
            ia_sqlVector("SELECT str, id, num FROM testIaMysqli WHERE id <= 2 ORDER BY id"),
            "Obten key value"
        );
        $this->assertEquals(
            false,
            ia_sqlVector("SELECT id, str, campoInvalido FROM testIaMysqlix WHERE id <= 2 ORDER BY id"),
            "Error"
        );

    }

    /**
     * @covers ::ia_sqlArrayIndx
     *
     * @dataProvider provider_ia_sqlArrayIndx
     */
    public function test_ia_sqlArrayIndx(string $sql, int $assoc, array|false $expected) {
        $this->assertEquals($expected, ia_sqlArrayIndx($sql, $assoc));
    }

        public function provider_ia_sqlArrayIndx():array {
            return [
              'Sql Error' => ["SELECT invalidaColumn FROM invalidaTable", MYSQLI_ASSOC, false],

                'MYSQLI_NUM' => ["SELECT id,str FROM testIaMysqli WHERE id <= 2 ORDER BY id", MYSQLI_NUM,
                    [['1', 'uno'], ['2', 'dos con null']]
                ],

                'MYSQLI_ASSOC' => ["SELECT id,str FROM testIaMysqli WHERE id <= 2 ORDER BY id", MYSQLI_ASSOC,
                    [ ['id' => '1', 'str' => 'uno'], ['id' => '2',  'str' => 'dos con null']]
                ],
                'MYSQLI_BOTH' => ["SELECT id,str FROM testIaMysqli WHERE id <= 2 ORDER BY id", MYSQLI_BOTH,
                    [ ['id' => '1', 'str' => 'uno','1', 'uno'], ['id' => '2',  'str' => 'dos con null','2', 'dos con null']]
                ],
            ];
        }

    /**
     * @covers ::ia_sqlArray
     *
     * @dataProvider provider_ia_sqlArray
     */
    public function test_ia_sqlArray(string $sql, int $assoc, array|false $expected) {
        $this->assertEquals($expected, ia_sqlArray($sql, 'id', $assoc));
    }

    public function provider_ia_sqlArray():array {

        return [
            'Sql Error' => ["SELECT invalidaColumn FROM invalidaTable", MYSQLI_ASSOC, false],
            'MYSQLI_ASSOC' => ["SELECT id, str, str FROM testIaMysqli WHERE id <= 2 ORDER BY id", MYSQLI_ASSOC,
                ['1' =>  ['id' => '1', 'str' => 'uno'], '2' => ['id' => '2',  'str' => 'dos con null']]
            ],
        ];
    }


    /**
     * @covers  ::assoc_mysql2mysqli()
     *
     * @dataProvider providerAssoc_mysql2mysqli
     */
    public function testAssoc_mysql2mysqli(int$assoc, int $expected) {
        $this->assertEquals($expected, assoc_mysql2mysqli($assoc));
    }

        public function providerAssoc_mysql2mysqli():array {
        return [
            "MYSQLI_ASSOC" => [MYSQLI_ASSOC, MYSQLI_ASSOC],
            "MYSQLI_NUM" => [MYSQLI_NUM,MYSQLI_NUM],
            "MYSQLI_BOTH" => [MYSQLI_BOTH, MYSQLI_BOTH],

            "MYSQL_ASSOC" => [MYSQL_ASSOC, MYSQLI_ASSOC],
            "MYSQL_NUM" => [MYSQL_NUM,MYSQLI_NUM],
            "MYSQI_BOTH" => [MYSQL_BOTH, MYSQLI_BOTH],

            "Otro número" => [MYSQLI_READ_DEFAULT_GROUP, MYSQLI_ASSOC],
        ];
    }


    /**
     * @covers ::trace_sql()
     * @covers ::sql_trace_get()
     */
    public function test_trace_sql() {
        global $gSqlClass;
        $original = $gSqlClass->traceOn;
        trace_sql(true);
        $this->assertTrue($gSqlClass->traceOn, "change trace sql to true");
        $gSqlClass->single_read("SELECT 'test_sql_trace_get trace on' FROM dual");

        $log = sql_trace_get();
        $logEntry = $log[count($log) - 1];
        $this->assertTrue(str_starts_with($logEntry, "SELECT 'test_sql_trace_get trace on' FROM dual"), "sql trace On, logs the query");

        trace_sql(false);
        $this->assertFalse($gSqlClass->traceOn, "change trace sql to false");
        $gSqlClass->single_read("SELECT 'test_sql_trace_get trace off' FROM dual");
        $log = sql_trace_get();
        $logEntry = $log[count($log) - 1];
        $this->assertTrue(!str_contains($logEntry, "SELECT 'test_sql_trace_get trace off' FROM dual"), "sql trace On, logs the query");



        $gSqlClass->traceOn = $original;
    }

    /**
     * @covers ::metaDataOnSet
     */
    public function test_metaDataOnSet() {
        $prev = metaDataOnSet(true);
        $this->assertTrue(metaDataOnSet(false));
        $this->assertFalse(metaDataOnSet(true));
        metaDataOnSet($prev);
    }

    /**
     * @covers ::affected_rows
     * @covers ::ia_query()
     * @uses ::ia_singleread()
     */
    public function test_affected_rows() {
        $prev = ia_singleread("SELECT num FROM testIaMysqli WHERE id=1");
        $change = $prev + 1;
        ia_query("UPDATE testIaMysqli set num = $change WHERE id = 1");
        $this->assertEquals(1, affected_rows(), 'Rows changed');
        ia_query("UPDATE testIaMysqli set num = $change WHERE id = 1");
        $this->assertEquals(0, affected_rows(), 'No rows changed');
        ia_query("UPDATE testIaMysqli set num = $change + 1 WHERE id = 1");
        $this->assertEquals(1, affected_rows(), 'Rows changed');

    }

    /**
     *
     * @covers ::ia_query()
     * @uses ::ia_singleread()
     */
    public function test_ia_query() {
        $prev = ia_singleread("SELECT num FROM testIaMysqli WHERE id=1");
        $change = $prev + 1;
        ia_query("UPDATE testIaMysqli set num = $change WHERE id = 1", true, false);
        $this->assertEquals("$change", ia_singleread("SELECT num FROM testIaMysqli WHERE id=1"));
        ia_query("UPDATE testIaMysqli set num = $prev WHERE id = 1");
    }

    /**
     *
     * @covers ::ia_query()
     * @covers ::ia_insertIded()
     * @covers ::ia_singleread()
     * @covers ::ia_singleton()
     * @covers ::ia_singletonFull()
     * @covers ::ia_sqlArray()
     * @covers ::ia_sqlArrayIndx()
     * @covers ::ia_sqlSelectMultiKey()
     * @covers ::ia_sqlKeyValue()
     * @covers ::ia_sqlVector()
     *
     * @covers Iac\inc\sql\IacSqlException
     *
     * @dataProvider provider_sql_error
     */
    public function test_sql_error(string $func, string $sql, bool|array $expected, string|int $param2 = null) {
        global $gSqlClass;
            $gSqlClass->throwSqlException_set(true);
            $this->assertEquals($expected,$func($sql, $param2), "On Error $func returns false");
            $gSqlClass->throwSqlException_set(false);
    }

    public function provider_sql_error():array {
        return [
            'ia_query' => ['ia_query', 'UPDATE unkownTable SET asdf=3 WHERE qwerty=9', true],
            'ia_insertIded' => ['ia_insertIded', 'INSERT INTO asdf(qwerty) VALUES(1,2)', false],
            'ia_singleread' => ['ia_singleread', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', false],

            'ia_singleton' => ['ia_singleton', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', false],
            'ia_singletonFull' => ['ia_singletonFull', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', false],
            'ia_sqlArray' => ['ia_sqlArray', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', false],
            'ia_sqlArrayIndx' => ['ia_sqlArrayIndx', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', false],
            'ia_sqlSelectMultiKey' => ['ia_sqlSelectMultiKey', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', [], 1],
            'ia_sqlKeyValue' => ['ia_sqlKeyValue', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', false],
            'ia_sqlVector' => ['ia_sqlVector', 'SELECT asdf FROM unkownTable  WHERE qwerty=9', false],
        ];
    }
}
