<?php
/** @noinspection SqlNoDataSourceInspection */
/** @noinspection SqlResolve */
declare(strict_types=1);

use Iac\inc\sql\IacSqlBuilder;
use PHPUnit\Framework\TestCase;

/**
 * @uses ::strim
 */
class IacSqlBuilderTest extends TestCase {

    /**
     * @covers \Iac\inc\sql\IacSqlBuilder::where
     *
     * @covers \Iac\inc\sql\IacSqlBuilder::arrayToSqlIn
     * @covers \Iac\inc\sql\IacSqlBuilder::strit
     * @covers Iac\inc\sql\IacStringer::strim
     * @uses Iac\inc\sql\IacSqlBuilder::fieldit
     * @uses ::strit
     *
     * @dataProvider providerWhere
     */
    public function testWhere(string|array $where, string $extraString, string $booleanOperator, array $fieldNameDontQuote, $expected) {
        $sqlBuilder = new IacSqlBuilder();
        $this->assertEquals($expected,
          $sqlBuilder->where($where, $extraString, $booleanOperator, $fieldNameDontQuote)
        );
    }


    public function providerWhere() {
        return [
            'where string' => ["string=2", '', 'AND', [], "string=2"],
            'fieldName=>value joined with and' => [['field1'=>1,'table.field2'=>3,], '',
                'AND', [],
                "(`field1`='1' AND `table`.`field2`='3')"],
            'fieldName=>value joined with or' => [['`field1`'=>null,'table.field2'=>'NOW()'], ' otro = 3', 'OR', [],
                "(`field1` IS NULL OR `table`.`field2`=NOW()) OR ( otro = 3)"],
            'fieldName=>value dont quote' => [['field1'=>1,'table.field2'=>3, 'dontQuoteMe'=>'func()'], '',
                'AND', ['dontQuoteMe'],
                "(`field1`='1' AND `table`.`field2`='3' AND `dontQuoteMe`=func())"],
            'fieldName multiple values ie: in clause' => [['fieldIn'=> ['cat', 'dog', 'CURRENT_DATE'] ], '' , 'AND', [],
                "(`fieldIn` IN ('cat','dog',CURRENT_DATE))"],
            'fieldName value is empty array' => [['fieldIn'=> [] ], '' , 'AND', [],
                "(`fieldIn` IN ('\t'))"],
        ];
    }

    /**
     * @covers \Iac\inc\sql\IacSqlBuilder::insertInto
     *
     * @covers Iac\inc\sql\IacStringer::strim
     * @uses Iac\inc\sql\IacSqlBuilder::fieldit
     *
     */
    public function testInsertInto() {
        $sqlBuilder = new IacSqlBuilder();
        $this->assertEquals("INSERT INTO `db`.`table`(`fieldOne`,`field2`) ",
            $sqlBuilder->insertInto('db.`table`', ['fieldOne'=>1, '`field2`'=>2])
        );
    }

    /**
     * @covers Iac\inc\sql\IacSqlBuilder::fieldit
     *
     * @covers Iac\inc\sql\IacStringer::strim
     *
     * @dataProvider providerFieldIt
     */
    public function testFieldIt($fieldName, $expected) {
        $sqlBuilder = new IacSqlBuilder();
        $this->assertEquals($expected, $sqlBuilder->fieldit($fieldName));
    }

    public function providerFieldIt():array {
        return [
          'fieldName not quoted' => ['fieldName', "`fieldName`"],
          'fieldName quoted' => ['`fieldName`', "`fieldName`"],
          'fieldName badly quoted' => ['`fieldName`;DELETE * FROM A', "`fieldName;DELETE * FROM A`"],

            'db.fieldName not quoted' => ['db.fieldName', "`db`.`fieldName`"],
            'db.fieldName quoted' => ['`db`.`fieldName`', "`db`.`fieldName`"],
            'db.fieldName db quoted' => ['`db`.fieldName', "`db`.`fieldName`"],
            'db.fieldName fieldName quoted' => ['db.`fieldName`', "`db`.`fieldName`"],
            'schema db.fieldName ' => ['schema.db.fieldName', "`schema`.`db`.`fieldName`"],
        ];
    }

    /**
     * @covers Iac\inc\sql\IacSqlBuilder::insertOnDuplicateKey
     *
     * @uses Iac\inc\sql\IacSqlBuilder::fieldit
     * @uses Iac\inc\sql\IacStringer::strim
     * @dataProvider providerInsertOnDuplicateKey
     */
    public function testInsertOnDuplicateKey($values, $onUpdate, $fieldNameDontUpdate, $expected ) {
        $sqlBuilder = new IacSqlBuilder();
        $this->assertEquals($expected,
            $sqlBuilder->insertOnDuplicateKey($values, $onUpdate, $fieldNameDontUpdate)
        );
    }

    public function providerInsertOnDuplicateKey():array {
        return [
            'on duplicate update automatic' => [
                ['col1'=>1, '`col2`'=>2, 'col3'=>3], '', ['col3', 'colNotInArray'],
                " ON DUPLICATE KEY UPDATE `col1`=VALUES(`col1`),`col2`=VALUES(`col2`)"
            ],
            'on duplicate add manual' => [
                ['col1'=>1, '`col2`'=>2], 'fecha=NOW()', [],
                " ON DUPLICATE KEY UPDATE fecha=NOW(),`col1`=VALUES(`col1`),`col2`=VALUES(`col2`)"
            ],
            'on duplicate add manual with commas' => [
                ['col1'=>1, '`col2`'=>2], ',fecha=NOW(),', [],
                " ON DUPLICATE KEY UPDATE fecha=NOW(),`col1`=VALUES(`col1`),`col2`=VALUES(`col2`)"
            ],
            'only manual' => [
                ['col1'=>1, '`col2`'=>2], ',fecha=NOW(),', ['col1','`col2`'],
                " ON DUPLICATE KEY UPDATE fecha=NOW()"
            ],
            'only manual empty values array' => [
                [], 'fecha=NOW()', ['col1','`col2`'],
                " ON DUPLICATE KEY UPDATE fecha=NOW()"
            ],
            'empty' => [
                ['col1'=>1, '`col2`'=>2], '', ['col1','`col2`'],
                ""
            ],
        ];
    }

    /**
     * @covers \Iac\inc\sql\IacSqlBuilder::insertValues
     *
     * @covers \Iac\inc\sql\IacSqlBuilder::strit
     *
     * @dataProvider providerInsertValues
     */
    public function testInsertValues($values, $dontQuote, $expected) {
        $sqlBuilder = new IacSqlBuilder();
        $this->assertEquals(strim($expected),
            strim($sqlBuilder->insertValues($values, $dontQuote))
        );
    }

    public function providerInsertValues():array {
        return [
            'all values quoted by default' => [
                ['col1'=>1, 'col2'=>"Do's", 'hoy'=>'NOW()'],
                [],
                "('1','Do''s', NOW())"
            ],
            "'' changes to null" => [
                ['col1'=>'', 'col2'=>NULL, 'hoy'=>'NOW()'],
                [],
                "( NULL, NULL, NOW())"
            ],
            'dont quote a field by default' => [
                ['col1'=>1, 'col2'=>"Do's", 'hoy'=>'NOW()'],
                ['col1','colNotInValues'],
                "( 1,'Do''s', NOW())"
            ],
            'empty values' => [[], [], ''],
        ];
    }

    /**
     * @covers \Iac\inc\sql\IacSqlBuilder::insert
     *
     * @covers Iac\inc\sql\IacSqlBuilder::fieldit
     * @covers Iac\inc\sql\IacStringer::strim
     * @covers Iac\inc\sql\IacStringer::strit
     * @dataProvider providerInsert
     */
    public function testInsert(string $expected, string $table, array $values,
                               bool $autoOnUpdate = false, string $onUpdate = '', array $fieldNameDontOnUpdate = [], array $fieldNameDontQuote = []) {
        $sqlBuilder = new IacSqlBuilder();
        $this->assertEquals(strim($expected),
            strim($sqlBuilder->insert($table, $values,
                $autoOnUpdate, $onUpdate, $fieldNameDontOnUpdate,
            $fieldNameDontQuote))
        );
    }

    public function providerInsert():array {
        return [
          "no auto update, default" => [
              "INSERT  /* INSERTA tabla **/ INTO tabla(`col1`,`col2`,`col3`) VALUES('1','2',NOW())",
              'tabla', ['col1'=>1, '`col2`'=>2, 'col3'=>'NOW()']
          ],
            "no auto update, dont quote" => [
                "INSERT  /* INSERTA tabla **/ INTO tabla(`col1`,`col2`,`col3`) VALUES(1,'2',NOW())",
                'tabla', ['col1'=>1, '`col2`'=>2, 'col3'=>'NOW()'], false, '', [], ['col1', 'colNotInValues']
            ],
            "manual on update" => [
                "INSERT  /* INSERTA tabla **/ INTO tabla(`col1`,`col2`,`col3`) VALUES('1','2',NOW()) ON DUPLICATE KEY UPDATE col2=23",
                'tabla', ['col1'=>1, '`col2`'=>2, 'col3'=>'NOW()'], false, "col2=23"
            ],
            "manual on update commas" => [
                "INSERT  /* INSERTA tabla **/ INTO tabla(`col1`,`col2`,`col3`) VALUES('1','2',NOW()) ON DUPLICATE KEY UPDATE col2=23",
                'tabla', ['col1'=>1, '`col2`'=>2, 'col3'=>'NOW()'], false, ",col2=23,"
            ],
            "automatic on update" => [
                "INSERT  /* INSERTA tabla **/ INTO tabla(`col1`,`col2`,`col3`) VALUES(NULL,'2',NOW()) ON DUPLICATE KEY UPDATE `col1`=VALUES(`col1`),`col2`=VALUES(`col2`),`col3`=VALUES(`col3`)",
                'tabla', ['col1'=>NULL, '`col2`'=>2, 'col3'=>'NOW()'], true
            ],
            "exclude on update" => [
                "INSERT  /* INSERTA tabla **/ INTO tabla(`col1`,`col2`,`col3`) VALUES(NULL,'2',NOW()) ON DUPLICATE KEY UPDATE `col2`=VALUES(`col2`),`col3`=VALUES(`col3`)",
                'tabla', ['col1'=>NULL, '`col2`'=>2, 'col3'=>'NOW()'], true, '', ['col1']
            ],
        ];
    }

    /**
     *  @covers \Iac\inc\sql\IacSqlBuilder::update
     *  @uses \Iac\inc\sql\IacSqlBuilder::where
     *  @uses \Iac\inc\sql\IacSqlBuilder::fieldit()
     *  @uses Iac\inc\sql\IacStringer::strit
     *  @uses Iac\inc\sql\IacStringer::strim
     *
     * @dataProvider providerUpdate
     */
    public function testUpdate($expected, $table, $values, $where=[], $fieldNameDontQuote=[], $whereDontQuoteFieldName=[], $comment='') {
        $sqlBuilder = new IacSqlBuilder();
        $this->assertEquals(strim($expected),
            strim($sqlBuilder->update($table, $values, $where, $fieldNameDontQuote, $whereDontQuoteFieldName, $comment))
        );
    }

    public function providerUpdate():array {
        return [
          "Values and where" => [
            "UPDATE /* ACTUALIZA tabla */ `tabla` SET `c1`='1',`c2`=NOW(),`c3`='Tre''s',`c4`=NULL WHERE (`id`='Tre''s')"  ,
              'tabla', ['c1'=>1, 'c2'=>"NOW()", 'c3'=>"Tre's", '`c4`'=>NULL], ["id" => "Tre's"]
          ],
            "Dont quote" => [
                "UPDATE /* comentario */ `tabla` SET `c1`=1,`c2`=NOW(),`c3`='Tre''s',`c4`=NULL WHERE (`id`=2)"  ,
                'tabla', ['c1'=>1, 'c2'=>"NOW()", 'c3'=>"Tre's", '`c4`'=>NULL], ["id" => 2], ['c1'], ['id'], 'comentario'
            ],
        ];
    }
}